// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "ash/quick_pair/fast_pair_handshake/fast_pair_encryption.h"

#include <algorithm>
#include <array>
#include <iterator>

#include "base/base64.h"
#include "testing/gtest/include/gtest/gtest.h"

namespace ash {
namespace quick_pair {
namespace fast_pair_encryption {

constexpr std::array<uint8_t, kBlockSizeBytes> aes_key_bytes = {
    0xA0, 0xBA, 0xF0, 0xBB, 0x95, 0x1F, 0xF7, 0xB6,
    0xCF, 0x5E, 0x3F, 0x45, 0x61, 0xC3, 0x32, 0x1D};

std::string DecodeKey(const std::string& encoded_key) {
  std::string key;
  base::Base64Decode(encoded_key, &key);
  return key;
}

class FastPairEncryptionTest : public testing::Test {};

TEST_F(FastPairEncryptionTest, EncryptBytes_Success) {
  std::array<uint8_t, kBlockSizeBytes> input = {
      0xF3, 0x0F, 0x4E, 0x78, 0x6C, 0x59, 0xA7, 0xBB,
      0xF3, 0x87, 0x3B, 0x5A, 0x49, 0xBA, 0x97, 0xEA};

  std::array<uint8_t, kBlockSizeBytes> expected = {
      0xAC, 0x9A, 0x16, 0xF0, 0x95, 0x3A, 0x3F, 0x22,
      0x3D, 0xD1, 0x0C, 0xF5, 0x36, 0xE0, 0x9E, 0x9C};

  EXPECT_EQ(EncryptBytes(aes_key_bytes, input), expected);
}

TEST_F(FastPairEncryptionTest, EncryptBytes_Failure) {
  std::array<uint8_t, kBlockSizeBytes> input = {
      0xF3, 0x0F, 0x4E, 0x78, 0x6C, 0x59, 0xA7, 0xBA,
      0xF3, 0x87, 0x3B, 0x5A, 0x49, 0xBA, 0x97, 0xEA};

  std::array<uint8_t, kBlockSizeBytes> expected = {
      0xAC, 0x9A, 0x16, 0xF0, 0x95, 0x3A, 0x3F, 0x22,
      0x3D, 0xD1, 0x0C, 0xF5, 0x36, 0xE0, 0x9E, 0x9C};

  EXPECT_NE(EncryptBytes(aes_key_bytes, input), expected);
}

TEST_F(FastPairEncryptionTest, GenerateKeysWithEcdhKeyAgreement_EmptyKey) {
  EXPECT_FALSE(GenerateKeysWithEcdhKeyAgreement("").has_value());
}

TEST_F(FastPairEncryptionTest, GenerateKeysWithEcdhKeyAgreement_ShortKey) {
  EXPECT_FALSE(GenerateKeysWithEcdhKeyAgreement("too_short").has_value());
}

TEST_F(FastPairEncryptionTest, GenerateKeysWithEcdhKeyAgreement_InvalidKey) {
  EXPECT_FALSE(
      GenerateKeysWithEcdhKeyAgreement(
          DecodeKey("U2PWc3FHTxah/o0YT9n1VRvtm57SNIRSXOEBXm4fdtMo+06tNoFlt8D0/"
                    "2BsN8auolz5ikwLRvQh+MiQ6oYveg=="))
          .has_value());
}

TEST_F(FastPairEncryptionTest, GenerateKeysWithEcdhKeyAgreement_ValidKey) {
  EXPECT_TRUE(
      GenerateKeysWithEcdhKeyAgreement(
          DecodeKey("U2PWc3FHTxah/o0YU9n1VRvtm57SNIRSXOEBXm4fdtMo+06tNoFlt8D0/"
                    "2BsN8auolz5ikwLRvQh+MiQ6oYveg=="))
          .has_value());
}

// `input`, `secret_key`, and `expected` values taken from the Fast Pair spec:
// https://developers.google.com/nearby/fast-pair/specifications/appendix/testcases#hmac-sha256
// The spec concatenates `nonce` || `input` into a single input vector which are
// distinct here.
TEST_F(FastPairEncryptionTest, GenerateHmacSha256_Success) {
  const std::vector<uint8_t> input = {0xEE, 0x4A, 0x24, 0x83, 0x73, 0x80, 0x52,
                                      0xE4, 0x4E, 0x9B, 0x2A, 0x14, 0x5E, 0x5D,
                                      0xDF, 0xAA, 0x44, 0xB9, 0xE5, 0x53, 0x6A,
                                      0xF4, 0x38, 0xE1, 0xE5, 0xC6};

  const std::array<uint8_t, kNonceSizeBytes> nonce = {0x00, 0x01, 0x02, 0x03,
                                                      0x04, 0x05, 0x06, 0x07};

  const std::array<uint8_t, kSecretKeySizeBytes> secret_key = {
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};

  const std::array<uint8_t, kHmacSizeBytes> expected = {
      0x55, 0xEC, 0x5E, 0x60, 0x55, 0xAF, 0x6E, 0x92, 0x61, 0x8B, 0x7D,
      0x87, 0x10, 0xD4, 0x41, 0x37, 0x09, 0xAB, 0x5D, 0xA2, 0x7C, 0xA2,
      0x6A, 0x66, 0xF5, 0x2E, 0x5A, 0xD4, 0xE8, 0x20, 0x90, 0x52};

  EXPECT_EQ(GenerateHmacSha256(secret_key, nonce, input), expected);
}

TEST_F(FastPairEncryptionTest, GenerateHmacSha256_Failure) {
  // The first byte is 0xFF in place of 0xEE.
  const std::vector<uint8_t> input = {0xFF, 0x4A, 0x24, 0x83, 0x73, 0x80, 0x52,
                                      0xE4, 0x4E, 0x9B, 0x2A, 0x14, 0x5E, 0x5D,
                                      0xDF, 0xAA, 0x44, 0xB9, 0xE5, 0x53, 0x6A,
                                      0xF4, 0x38, 0xE1, 0xE5, 0xC6};

  const std::array<uint8_t, kNonceSizeBytes> nonce = {0x00, 0x01, 0x02, 0x03,
                                                      0x04, 0x05, 0x06, 0x07};

  const std::array<uint8_t, kSecretKeySizeBytes> secret_key = {
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};

  const std::array<uint8_t, kHmacSizeBytes> expected = {
      0x55, 0xEC, 0x5E, 0x60, 0x55, 0xAF, 0x6E, 0x92, 0x61, 0x8B, 0x7D,
      0x87, 0x10, 0xD4, 0x41, 0x37, 0x09, 0xAB, 0x5D, 0xA2, 0x7C, 0xA2,
      0x6A, 0x66, 0xF5, 0x2E, 0x5A, 0xD4, 0xE8, 0x20, 0x90, 0x52};

  EXPECT_NE(GenerateHmacSha256(secret_key, nonce, input), expected);
}

TEST_F(FastPairEncryptionTest, GenerateHmacSha256_EmptyParamCombos_NoCrash) {
  const std::vector<uint8_t> input = {0xEE, 0x4A, 0x24, 0x83, 0x73, 0x80, 0x52,
                                      0xE4, 0x4E, 0x9B, 0x2A, 0x14, 0x5E, 0x5D,
                                      0xDF, 0xAA, 0x44, 0xB9, 0xE5, 0x53, 0x6A,
                                      0xF4, 0x38, 0xE1, 0xE5, 0xC6};

  const std::array<uint8_t, kNonceSizeBytes> nonce = {0x00, 0x01, 0x02, 0x03,
                                                      0x04, 0x05, 0x06, 0x07};

  const std::array<uint8_t, kSecretKeySizeBytes> secret_key = {
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};
  std::vector<uint8_t> empty_input;
  std::array<uint8_t, kNonceSizeBytes> empty_nonce;
  std::array<uint8_t, kSecretKeySizeBytes> empty_secret_key;
  for (size_t i = 0; i < 2; i++) {
    for (size_t j = 0; j < 2; j++) {
      for (size_t k = 0; k < 2; k++) {
        GenerateHmacSha256(i ? secret_key : empty_secret_key,
                           j ? nonce : empty_nonce, k ? input : empty_input);
      }
    }
  }
}

TEST_F(FastPairEncryptionTest, GenerateHmacSha256_OneByteData) {
  const std::vector<uint8_t> input = {};

  const std::array<uint8_t, kNonceSizeBytes> nonce = {0x00};

  const std::array<uint8_t, kSecretKeySizeBytes> secret_key = {
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};

  const std::array<uint8_t, kHmacSizeBytes> expected = {
      0xBB, 0x07, 0xA3, 0xDE, 0x98, 0xC4, 0x97, 0x9C, 0x5F, 0x52, 0x19,
      0x39, 0x9F, 0xCB, 0xDA, 0xEB, 0xA3, 0x7E, 0xD7, 0xFA, 0x84, 0xAA,
      0x28, 0x2A, 0x76, 0xCF, 0xF3, 0xB6, 0x30, 0x36, 0x7E, 0x10};

  EXPECT_EQ(GenerateHmacSha256(secret_key, nonce, input), expected);
}

// `input`, `secret_key`, `nonce`, and `expected` values taken from the Fast
// Pair spec:
// https://developers.google.com/nearby/fast-pair/specifications/appendix/testcases#aes_encryption
TEST_F(FastPairEncryptionTest, EncryptAdditionalData_Success) {
  const std::vector<uint8_t> input = {0x53, 0x6F, 0x6D, 0x65, 0x6F, 0x6E, 0x65,
                                      0x27, 0x73, 0x20, 0x47, 0x6F, 0x6F, 0x67,
                                      0x6C, 0x65, 0x20, 0x48, 0x65, 0x61, 0x64,
                                      0x70, 0x68, 0x6F, 0x6E, 0x65};

  const std::array<uint8_t, kSecretKeySizeBytes> secret_key = {
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};

  std::array<uint8_t, kNonceSizeBytes> nonce = {0x00, 0x01, 0x02, 0x03,
                                                0x04, 0x05, 0x06, 0x07};

  const std::vector<uint8_t> expected = {
      0xEE, 0x4A, 0x24, 0x83, 0x73, 0x80, 0x52, 0xE4, 0x4E,
      0x9B, 0x2A, 0x14, 0x5E, 0x5D, 0xDF, 0xAA, 0x44, 0xB9,
      0xE5, 0x53, 0x6A, 0xF4, 0x38, 0xE1, 0xE5, 0xC6};

  EXPECT_EQ(EncryptAdditionalData(secret_key, nonce, input), expected);
}

TEST_F(FastPairEncryptionTest, EncryptAdditionalData_Failure) {
  // The first byte is 0x54 instead of 0x53
  const std::vector<uint8_t> input = {0x54, 0x6F, 0x6D, 0x65, 0x6F, 0x6E, 0x65,
                                      0x27, 0x73, 0x20, 0x47, 0x6F, 0x6F, 0x67,
                                      0x6C, 0x65, 0x20, 0x48, 0x65, 0x61, 0x64,
                                      0x70, 0x68, 0x6F, 0x6E, 0x65};

  const std::array<uint8_t, kSecretKeySizeBytes> secret_key = {
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};

  std::array<uint8_t, kNonceSizeBytes> nonce = {0x00, 0x01, 0x02, 0x03,
                                                0x04, 0x05, 0x06, 0x07};

  const std::vector<uint8_t> expected = {
      0xEE, 0x4A, 0x24, 0x83, 0x73, 0x80, 0x52, 0xE4, 0x4E,
      0x9B, 0x2A, 0x14, 0x5E, 0x5D, 0xDF, 0xAA, 0x44, 0xB9,
      0xE5, 0x53, 0x6A, 0xF4, 0x38, 0xE1, 0xE5, 0xC6};

  EXPECT_NE(EncryptAdditionalData(secret_key, nonce, input), expected);
}

TEST_F(FastPairEncryptionTest, EncryptAdditionalData_EmptyData) {
  const std::vector<uint8_t> input = {};

  const std::array<uint8_t, kSecretKeySizeBytes> secret_key = {
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};

  std::array<uint8_t, kNonceSizeBytes> nonce = {0x00, 0x01, 0x02, 0x03,
                                                0x04, 0x05, 0x06, 0x07};

  const std::vector<uint8_t> expected = {};

  EXPECT_EQ(EncryptAdditionalData(secret_key, nonce, input), expected);
}

TEST_F(FastPairEncryptionTest, EncryptAdditionalData_EmptyParamCombos_NoCrash) {
  std::vector<uint8_t> input = {0x53, 0x6F, 0x6D, 0x65, 0x6F, 0x6E, 0x65,
                                0x27, 0x73, 0x20, 0x47, 0x6F, 0x6F, 0x67,
                                0x6C, 0x65, 0x20, 0x48, 0x65, 0x61, 0x64,
                                0x70, 0x68, 0x6F, 0x6E, 0x65};
  std::array<uint8_t, kSecretKeySizeBytes> secret_key = {
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};
  std::array<uint8_t, kNonceSizeBytes> nonce = {0x00, 0x01, 0x02, 0x03,
                                                0x04, 0x05, 0x06, 0x07};
  std::vector<uint8_t> empty_input;
  std::array<uint8_t, kNonceSizeBytes> empty_nonce;
  std::array<uint8_t, kSecretKeySizeBytes> empty_secret_key;
  for (size_t i = 0; i < 2; i++) {
    for (size_t j = 0; j < 2; j++) {
      for (size_t k = 0; k < 2; k++) {
        EncryptAdditionalData(i ? secret_key : empty_secret_key,
                              j ? nonce : empty_nonce, k ? input : empty_input);
      }
    }
  }
}

TEST_F(FastPairEncryptionTest, EncryptAdditionalData_OneByteData) {
  const std::vector<uint8_t> input = {0x00};

  const std::array<uint8_t, kSecretKeySizeBytes> secret_key = {
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
      0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF};

  std::array<uint8_t, kNonceSizeBytes> nonce = {0x00, 0x01, 0x02, 0x03,
                                                0x04, 0x05, 0x06, 0x07};

  const std::vector<uint8_t> expected = {0xBD};

  EXPECT_EQ(EncryptAdditionalData(secret_key, nonce, input), expected);
}

}  // namespace fast_pair_encryption
}  // namespace quick_pair
}  // namespace ash
