/* SPDX-License-Identifier: BSD-3-Clause-Clear */
/*
 * Copyright (c) 2018-2019 The Linux Foundation. All rights reserved.
 * Copyright (c) 2021-2025 Qualcomm Innovation Center, Inc. All rights reserved.
 */

#ifndef ATH11K_WMI_H
#define ATH11K_WMI_H

#include <net/mac80211.h>
#include "htc.h"

struct ath11k_base;
struct ath11k;
struct ath11k_fw_stats;
struct ath11k_fw_dbglog;
struct ath11k_vif;
struct ath11k_reg_tpc_power_info;

#define PSOC_HOST_MAX_NUM_SS (8)

/* defines to set Packet extension values which can be 0 us, 8 usec or 16 usec */
#define MAX_HE_NSS               8
#define MAX_HE_MODULATION        8
#define MAX_HE_RU                4
#define HE_MODULATION_NONE       7
#define HE_PET_0_USEC            0
#define HE_PET_8_USEC            1
#define HE_PET_16_USEC           2

#define WMI_MAX_CHAINS		 8

#define WMI_MAX_NUM_SS                    MAX_HE_NSS
#define WMI_MAX_NUM_RU                    MAX_HE_RU

#define WMI_TLV_CMD(grp_id) (((grp_id) << 12) | 0x1)
#define WMI_TLV_EV(grp_id) (((grp_id) << 12) | 0x1)
#define WMI_TLV_CMD_UNSUPPORTED 0
#define WMI_TLV_PDEV_PARAM_UNSUPPORTED 0
#define WMI_TLV_VDEV_PARAM_UNSUPPORTED 0

struct wmi_cmd_hdr {
	u32 cmd_id;
} __packed;

struct wmi_tlv {
	u32 header;
	u8 value[];
} __packed;

#define WMI_TLV_LEN	GENMASK(15, 0)
#define WMI_TLV_TAG	GENMASK(31, 16)
#define TLV_HDR_SIZE	sizeof_field(struct wmi_tlv, header)

#define WMI_CMD_HDR_CMD_ID      GENMASK(23, 0)
#define WMI_MAX_MEM_REQS        32
#define ATH11K_MAX_HW_LISTEN_INTERVAL 5

#define WLAN_SCAN_MAX_HINT_S_SSID        10
#define WLAN_SCAN_MAX_HINT_BSSID         10
#define MAX_RNR_BSS                    5

#define WLAN_SCAN_PARAMS_MAX_SSID    16
#define WLAN_SCAN_PARAMS_MAX_BSSID   4
#define WLAN_SCAN_PARAMS_MAX_IE_LEN  512

#define WMI_APPEND_TO_EXISTING_CHAN_LIST_FLAG 1

#define MAX_WMI_UTF_LEN 252
#define WMI_BA_MODE_BUFFER_SIZE_256  3
/*
 * HW mode config type replicated from FW header
 * @WMI_HOST_HW_MODE_SINGLE: Only one PHY is active.
 * @WMI_HOST_HW_MODE_DBS: Both PHYs are active in different bands,
 *                        one in 2G and another in 5G.
 * @WMI_HOST_HW_MODE_SBS_PASSIVE: Both PHYs are in passive mode (only rx) in
 *                        same band; no tx allowed.
 * @WMI_HOST_HW_MODE_SBS: Both PHYs are active in the same band.
 *                        Support for both PHYs within one band is planned
 *                        for 5G only(as indicated in WMI_MAC_PHY_CAPABILITIES),
 *                        but could be extended to other bands in the future.
 *                        The separation of the band between the two PHYs needs
 *                        to be communicated separately.
 * @WMI_HOST_HW_MODE_DBS_SBS: 3 PHYs, with 2 on the same band doing SBS
 *                           as in WMI_HW_MODE_SBS, and 3rd on the other band
 * @WMI_HOST_HW_MODE_DBS_OR_SBS: Two PHY with one PHY capabale of both 2G and
 *                        5G. It can support SBS (5G + 5G) OR DBS (5G + 2G).
 * @WMI_HOST_HW_MODE_MAX: Max hw_mode_id. Used to indicate invalid mode.
 */
enum wmi_host_hw_mode_config_type {
	WMI_HOST_HW_MODE_SINGLE       = 0,
	WMI_HOST_HW_MODE_DBS          = 1,
	WMI_HOST_HW_MODE_SBS_PASSIVE  = 2,
	WMI_HOST_HW_MODE_SBS          = 3,
	WMI_HOST_HW_MODE_DBS_SBS      = 4,
	WMI_HOST_HW_MODE_DBS_OR_SBS   = 5,

	/* keep last */
	WMI_HOST_HW_MODE_MAX
};

/* HW mode priority values used to detect the preferred HW mode
 * on the available modes.
 */
enum wmi_host_hw_mode_priority {
	WMI_HOST_HW_MODE_DBS_SBS_PRI,
	WMI_HOST_HW_MODE_DBS_PRI,
	WMI_HOST_HW_MODE_DBS_OR_SBS_PRI,
	WMI_HOST_HW_MODE_SBS_PRI,
	WMI_HOST_HW_MODE_SBS_PASSIVE_PRI,
	WMI_HOST_HW_MODE_SINGLE_PRI,

	/* keep last the lowest priority */
	WMI_HOST_HW_MODE_MAX_PRI
};

enum WMI_HOST_WLAN_BAND {
	WMI_HOST_WLAN_2G_CAP	= 0x1,
	WMI_HOST_WLAN_5G_CAP	= 0x2,
	WMI_HOST_WLAN_2G_5G_CAP	= WMI_HOST_WLAN_2G_CAP | WMI_HOST_WLAN_5G_CAP,
};

/* Parameters used for WMI_VDEV_PARAM_AUTORATE_MISC_CFG command.
 * Used only for HE auto rate mode.
 */
enum {
	/* HE LTF related configuration */
	WMI_HE_AUTORATE_LTF_1X = BIT(0),
	WMI_HE_AUTORATE_LTF_2X = BIT(1),
	WMI_HE_AUTORATE_LTF_4X = BIT(2),

	/* HE GI related configuration */
	WMI_AUTORATE_400NS_GI = BIT(8),
	WMI_AUTORATE_800NS_GI = BIT(9),
	WMI_AUTORATE_1600NS_GI = BIT(10),
	WMI_AUTORATE_3200NS_GI = BIT(11),
};

enum {
	WMI_HOST_VDEV_FLAGS_NON_MBSSID_AP       = 0x00000001,
	WMI_HOST_VDEV_FLAGS_TRANSMIT_AP         = 0x00000002,
	WMI_HOST_VDEV_FLAGS_NON_TRANSMIT_AP     = 0x00000004,
	WMI_HOST_VDEV_FLAGS_EMA_MODE            = 0x00000008,
	WMI_HOST_VDEV_FLAGS_SCAN_MODE_VAP       = 0x00000010,
};

/*
 * wmi command groups.
 */
enum wmi_cmd_group {
	/* 0 to 2 are reserved */
	WMI_GRP_START = 0x3,
	WMI_GRP_SCAN = WMI_GRP_START,
	WMI_GRP_PDEV		= 0x4,
	WMI_GRP_VDEV           = 0x5,
	WMI_GRP_PEER           = 0x6,
	WMI_GRP_MGMT           = 0x7,
	WMI_GRP_BA_NEG         = 0x8,
	WMI_GRP_STA_PS         = 0x9,
	WMI_GRP_DFS            = 0xa,
	WMI_GRP_ROAM           = 0xb,
	WMI_GRP_OFL_SCAN       = 0xc,
	WMI_GRP_P2P            = 0xd,
	WMI_GRP_AP_PS          = 0xe,
	WMI_GRP_RATE_CTRL      = 0xf,
	WMI_GRP_PROFILE        = 0x10,
	WMI_GRP_SUSPEND        = 0x11,
	WMI_GRP_BCN_FILTER     = 0x12,
	WMI_GRP_WOW            = 0x13,
	WMI_GRP_RTT            = 0x14,
	WMI_GRP_SPECTRAL       = 0x15,
	WMI_GRP_STATS          = 0x16,
	WMI_GRP_ARP_NS_OFL     = 0x17,
	WMI_GRP_NLO_OFL        = 0x18,
	WMI_GRP_GTK_OFL        = 0x19,
	WMI_GRP_CSA_OFL        = 0x1a,
	WMI_GRP_CHATTER        = 0x1b,
	WMI_GRP_TID_ADDBA      = 0x1c,
	WMI_GRP_MISC           = 0x1d,
	WMI_GRP_GPIO           = 0x1e,
	WMI_GRP_FWTEST         = 0x1f,
	WMI_GRP_TDLS           = 0x20,
	WMI_GRP_RESMGR         = 0x21,
	WMI_GRP_STA_SMPS       = 0x22,
	WMI_GRP_WLAN_HB        = 0x23,
	WMI_GRP_RMC            = 0x24,
	WMI_GRP_MHF_OFL        = 0x25,
	WMI_GRP_LOCATION_SCAN  = 0x26,
	WMI_GRP_OEM            = 0x27,
	WMI_GRP_NAN            = 0x28,
	WMI_GRP_COEX           = 0x29,
	WMI_GRP_OBSS_OFL       = 0x2a,
	WMI_GRP_LPI            = 0x2b,
	WMI_GRP_EXTSCAN        = 0x2c,
	WMI_GRP_DHCP_OFL       = 0x2d,
	WMI_GRP_IPA            = 0x2e,
	WMI_GRP_MDNS_OFL       = 0x2f,
	WMI_GRP_SAP_OFL        = 0x30,
	WMI_GRP_OCB            = 0x31,
	WMI_GRP_SOC            = 0x32,
	WMI_GRP_PKT_FILTER     = 0x33,
	WMI_GRP_MAWC           = 0x34,
	WMI_GRP_PMF_OFFLOAD    = 0x35,
	WMI_GRP_BPF_OFFLOAD    = 0x36,
	WMI_GRP_NAN_DATA       = 0x37,
	WMI_GRP_PROTOTYPE      = 0x38,
	WMI_GRP_MONITOR        = 0x39,
	WMI_GRP_REGULATORY     = 0x3a,
	WMI_GRP_HW_DATA_FILTER = 0x3b,
	WMI_GRP_WLM            = 0x3c,
	WMI_GRP_11K_OFFLOAD    = 0x3d,
	WMI_GRP_TWT            = 0x3e,
	WMI_GRP_MOTION_DET     = 0x3f,
	WMI_GRP_SPATIAL_REUSE  = 0x40,
};

#define WMI_CMD_GRP(grp_id) (((grp_id) << 12) | 0x1)
#define WMI_EVT_GRP_START_ID(grp_id) (((grp_id) << 12) | 0x1)

#define WMI_CMD_UNSUPPORTED 0

enum wmi_tlv_cmd_id {
	WMI_INIT_CMDID = 0x1,
	WMI_START_SCAN_CMDID = WMI_TLV_CMD(WMI_GRP_SCAN),
	WMI_STOP_SCAN_CMDID,
	WMI_SCAN_CHAN_LIST_CMDID,
	WMI_SCAN_SCH_PRIO_TBL_CMDID,
	WMI_SCAN_UPDATE_REQUEST_CMDID,
	WMI_SCAN_PROB_REQ_OUI_CMDID,
	WMI_SCAN_ADAPTIVE_DWELL_CONFIG_CMDID,
	WMI_PDEV_SET_REGDOMAIN_CMDID = WMI_TLV_CMD(WMI_GRP_PDEV),
	WMI_PDEV_SET_CHANNEL_CMDID,
	WMI_PDEV_SET_PARAM_CMDID,
	WMI_PDEV_PKTLOG_ENABLE_CMDID,
	WMI_PDEV_PKTLOG_DISABLE_CMDID,
	WMI_PDEV_SET_WMM_PARAMS_CMDID,
	WMI_PDEV_SET_HT_CAP_IE_CMDID,
	WMI_PDEV_SET_VHT_CAP_IE_CMDID,
	WMI_PDEV_SET_DSCP_TID_MAP_CMDID,
	WMI_PDEV_SET_QUIET_MODE_CMDID,
	WMI_PDEV_GREEN_AP_PS_ENABLE_CMDID,
	WMI_PDEV_GET_TPC_CONFIG_CMDID,
	WMI_PDEV_SET_BASE_MACADDR_CMDID,
	WMI_PDEV_DUMP_CMDID,
	WMI_PDEV_SET_LED_CONFIG_CMDID,
	WMI_PDEV_GET_TEMPERATURE_CMDID,
	WMI_PDEV_SET_LED_FLASHING_CMDID,
	WMI_PDEV_SMART_ANT_ENABLE_CMDID,
	WMI_PDEV_SMART_ANT_SET_RX_ANTENNA_CMDID,
	WMI_PDEV_SET_ANTENNA_SWITCH_TABLE_CMDID,
	WMI_PDEV_SET_CTL_TABLE_CMDID,
	WMI_PDEV_SET_MIMOGAIN_TABLE_CMDID,
	WMI_PDEV_FIPS_CMDID,
	WMI_PDEV_GET_ANI_CCK_CONFIG_CMDID,
	WMI_PDEV_GET_ANI_OFDM_CONFIG_CMDID,
	WMI_PDEV_GET_NFCAL_POWER_CMDID,
	WMI_PDEV_GET_TPC_CMDID,
	WMI_MIB_STATS_ENABLE_CMDID,
	WMI_PDEV_SET_PCL_CMDID,
	WMI_PDEV_SET_HW_MODE_CMDID,
	WMI_PDEV_SET_MAC_CONFIG_CMDID,
	WMI_PDEV_SET_ANTENNA_MODE_CMDID,
	WMI_SET_PERIODIC_CHANNEL_STATS_CONFIG_CMDID,
	WMI_PDEV_WAL_POWER_DEBUG_CMDID,
	WMI_PDEV_SET_REORDER_TIMEOUT_VAL_CMDID,
	WMI_PDEV_SET_WAKEUP_CONFIG_CMDID,
	WMI_PDEV_GET_ANTDIV_STATUS_CMDID,
	WMI_PDEV_GET_CHIP_POWER_STATS_CMDID,
	WMI_PDEV_SET_STATS_THRESHOLD_CMDID,
	WMI_PDEV_MULTIPLE_VDEV_RESTART_REQUEST_CMDID,
	WMI_PDEV_UPDATE_PKT_ROUTING_CMDID,
	WMI_PDEV_CHECK_CAL_VERSION_CMDID,
	WMI_PDEV_SET_DIVERSITY_GAIN_CMDID,
	WMI_PDEV_DIV_GET_RSSI_ANTID_CMDID,
	WMI_PDEV_BSS_CHAN_INFO_REQUEST_CMDID,
	WMI_PDEV_UPDATE_PMK_CACHE_CMDID,
	WMI_PDEV_UPDATE_FILS_HLP_PKT_CMDID,
	WMI_PDEV_UPDATE_CTLTABLE_REQUEST_CMDID,
	WMI_PDEV_CONFIG_VENDOR_OUI_ACTION_CMDID,
	WMI_PDEV_SET_AC_TX_QUEUE_OPTIMIZED_CMDID,
	WMI_PDEV_SET_RX_FILTER_PROMISCUOUS_CMDID,
	WMI_PDEV_DMA_RING_CFG_REQ_CMDID,
	WMI_PDEV_HE_TB_ACTION_FRM_CMDID,
	WMI_PDEV_PKTLOG_FILTER_CMDID,
	WMI_PDEV_SET_RAP_CONFIG_CMDID,
	WMI_PDEV_DSM_FILTER_CMDID,
	WMI_PDEV_FRAME_INJECT_CMDID,
	WMI_PDEV_TBTT_OFFSET_SYNC_CMDID,
	WMI_PDEV_SET_SRG_BSS_COLOR_BITMAP_CMDID,
	WMI_PDEV_SET_SRG_PARTIAL_BSSID_BITMAP_CMDID,
	WMI_PDEV_SET_SRG_OBSS_COLOR_ENABLE_BITMAP_CMDID,
	WMI_PDEV_SET_SRG_OBSS_BSSID_ENABLE_BITMAP_CMDID,
	WMI_PDEV_SET_NON_SRG_OBSS_COLOR_ENABLE_BITMAP_CMDID,
	WMI_PDEV_SET_NON_SRG_OBSS_BSSID_ENABLE_BITMAP_CMDID,
	WMI_PDEV_GET_TPC_STATS_CMDID,
	WMI_PDEV_ENABLE_DURATION_BASED_TX_MODE_SELECTION_CMDID,
	WMI_PDEV_GET_DPD_STATUS_CMDID,
	WMI_PDEV_SET_BIOS_SAR_TABLE_CMDID,
	WMI_PDEV_SET_BIOS_GEO_TABLE_CMDID,
	WMI_VDEV_CREATE_CMDID = WMI_TLV_CMD(WMI_GRP_VDEV),
	WMI_VDEV_DELETE_CMDID,
	WMI_VDEV_START_REQUEST_CMDID,
	WMI_VDEV_RESTART_REQUEST_CMDID,
	WMI_VDEV_UP_CMDID,
	WMI_VDEV_STOP_CMDID,
	WMI_VDEV_DOWN_CMDID,
	WMI_VDEV_SET_PARAM_CMDID,
	WMI_VDEV_INSTALL_KEY_CMDID,
	WMI_VDEV_WNM_SLEEPMODE_CMDID,
	WMI_VDEV_WMM_ADDTS_CMDID,
	WMI_VDEV_WMM_DELTS_CMDID,
	WMI_VDEV_SET_WMM_PARAMS_CMDID,
	WMI_VDEV_SET_GTX_PARAMS_CMDID,
	WMI_VDEV_IPSEC_NATKEEPALIVE_FILTER_CMDID,
	WMI_VDEV_PLMREQ_START_CMDID,
	WMI_VDEV_PLMREQ_STOP_CMDID,
	WMI_VDEV_TSF_TSTAMP_ACTION_CMDID,
	WMI_VDEV_SET_IE_CMDID,
	WMI_VDEV_RATEMASK_CMDID,
	WMI_VDEV_ATF_REQUEST_CMDID,
	WMI_VDEV_SET_DSCP_TID_MAP_CMDID,
	WMI_VDEV_FILTER_NEIGHBOR_RX_PACKETS_CMDID,
	WMI_VDEV_SET_QUIET_MODE_CMDID,
	WMI_VDEV_SET_CUSTOM_AGGR_SIZE_CMDID,
	WMI_VDEV_ENCRYPT_DECRYPT_DATA_REQ_CMDID,
	WMI_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_CMDID,
	WMI_VDEV_SET_ARP_STAT_CMDID,
	WMI_VDEV_GET_ARP_STAT_CMDID,
	WMI_VDEV_GET_TX_POWER_CMDID,
	WMI_VDEV_LIMIT_OFFCHAN_CMDID,
	WMI_VDEV_SET_CUSTOM_SW_RETRY_TH_CMDID,
	WMI_VDEV_CHAINMASK_CONFIG_CMDID,
	WMI_VDEV_GET_BCN_RECEPTION_STATS_CMDID,
	WMI_VDEV_GET_MWS_COEX_INFO_CMDID,
	WMI_VDEV_DELETE_ALL_PEER_CMDID,
	WMI_VDEV_BSS_MAX_IDLE_TIME_CMDID,
	WMI_VDEV_AUDIO_SYNC_TRIGGER_CMDID,
	WMI_VDEV_AUDIO_SYNC_QTIMER_CMDID,
	WMI_VDEV_SET_PCL_CMDID,
	WMI_VDEV_GET_BIG_DATA_CMDID,
	WMI_VDEV_GET_BIG_DATA_P2_CMDID,
	WMI_VDEV_SET_TPC_POWER_CMDID,
	WMI_PEER_CREATE_CMDID = WMI_TLV_CMD(WMI_GRP_PEER),
	WMI_PEER_DELETE_CMDID,
	WMI_PEER_FLUSH_TIDS_CMDID,
	WMI_PEER_SET_PARAM_CMDID,
	WMI_PEER_ASSOC_CMDID,
	WMI_PEER_ADD_WDS_ENTRY_CMDID,
	WMI_PEER_REMOVE_WDS_ENTRY_CMDID,
	WMI_PEER_MCAST_GROUP_CMDID,
	WMI_PEER_INFO_REQ_CMDID,
	WMI_PEER_GET_ESTIMATED_LINKSPEED_CMDID,
	WMI_PEER_SET_RATE_REPORT_CONDITION_CMDID,
	WMI_PEER_UPDATE_WDS_ENTRY_CMDID,
	WMI_PEER_ADD_PROXY_STA_ENTRY_CMDID,
	WMI_PEER_SMART_ANT_SET_TX_ANTENNA_CMDID,
	WMI_PEER_SMART_ANT_SET_TRAIN_INFO_CMDID,
	WMI_PEER_SMART_ANT_SET_NODE_CONFIG_OPS_CMDID,
	WMI_PEER_ATF_REQUEST_CMDID,
	WMI_PEER_BWF_REQUEST_CMDID,
	WMI_PEER_REORDER_QUEUE_SETUP_CMDID,
	WMI_PEER_REORDER_QUEUE_REMOVE_CMDID,
	WMI_PEER_SET_RX_BLOCKSIZE_CMDID,
	WMI_PEER_ANTDIV_INFO_REQ_CMDID,
	WMI_BCN_TX_CMDID = WMI_TLV_CMD(WMI_GRP_MGMT),
	WMI_PDEV_SEND_BCN_CMDID,
	WMI_BCN_TMPL_CMDID,
	WMI_BCN_FILTER_RX_CMDID,
	WMI_PRB_REQ_FILTER_RX_CMDID,
	WMI_MGMT_TX_CMDID,
	WMI_PRB_TMPL_CMDID,
	WMI_MGMT_TX_SEND_CMDID,
	WMI_OFFCHAN_DATA_TX_SEND_CMDID,
	WMI_PDEV_SEND_FD_CMDID,
	WMI_BCN_OFFLOAD_CTRL_CMDID,
	WMI_BSS_COLOR_CHANGE_ENABLE_CMDID,
	WMI_VDEV_BCN_OFFLOAD_QUIET_CONFIG_CMDID,
	WMI_FILS_DISCOVERY_TMPL_CMDID,
	WMI_ADDBA_CLEAR_RESP_CMDID = WMI_TLV_CMD(WMI_GRP_BA_NEG),
	WMI_ADDBA_SEND_CMDID,
	WMI_ADDBA_STATUS_CMDID,
	WMI_DELBA_SEND_CMDID,
	WMI_ADDBA_SET_RESP_CMDID,
	WMI_SEND_SINGLEAMSDU_CMDID,
	WMI_STA_POWERSAVE_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_STA_PS),
	WMI_STA_POWERSAVE_PARAM_CMDID,
	WMI_STA_MIMO_PS_MODE_CMDID,
	WMI_PDEV_DFS_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_DFS),
	WMI_PDEV_DFS_DISABLE_CMDID,
	WMI_DFS_PHYERR_FILTER_ENA_CMDID,
	WMI_DFS_PHYERR_FILTER_DIS_CMDID,
	WMI_PDEV_DFS_PHYERR_OFFLOAD_ENABLE_CMDID,
	WMI_PDEV_DFS_PHYERR_OFFLOAD_DISABLE_CMDID,
	WMI_VDEV_ADFS_CH_CFG_CMDID,
	WMI_VDEV_ADFS_OCAC_ABORT_CMDID,
	WMI_ROAM_SCAN_MODE = WMI_TLV_CMD(WMI_GRP_ROAM),
	WMI_ROAM_SCAN_RSSI_THRESHOLD,
	WMI_ROAM_SCAN_PERIOD,
	WMI_ROAM_SCAN_RSSI_CHANGE_THRESHOLD,
	WMI_ROAM_AP_PROFILE,
	WMI_ROAM_CHAN_LIST,
	WMI_ROAM_SCAN_CMD,
	WMI_ROAM_SYNCH_COMPLETE,
	WMI_ROAM_SET_RIC_REQUEST_CMDID,
	WMI_ROAM_INVOKE_CMDID,
	WMI_ROAM_FILTER_CMDID,
	WMI_ROAM_SUBNET_CHANGE_CONFIG_CMDID,
	WMI_ROAM_CONFIGURE_MAWC_CMDID,
	WMI_ROAM_SET_MBO_PARAM_CMDID,
	WMI_ROAM_PER_CONFIG_CMDID,
	WMI_ROAM_BTM_CONFIG_CMDID,
	WMI_ENABLE_FILS_CMDID,
	WMI_OFL_SCAN_ADD_AP_PROFILE = WMI_TLV_CMD(WMI_GRP_OFL_SCAN),
	WMI_OFL_SCAN_REMOVE_AP_PROFILE,
	WMI_OFL_SCAN_PERIOD,
	WMI_P2P_DEV_SET_DEVICE_INFO = WMI_TLV_CMD(WMI_GRP_P2P),
	WMI_P2P_DEV_SET_DISCOVERABILITY,
	WMI_P2P_GO_SET_BEACON_IE,
	WMI_P2P_GO_SET_PROBE_RESP_IE,
	WMI_P2P_SET_VENDOR_IE_DATA_CMDID,
	WMI_P2P_DISC_OFFLOAD_CONFIG_CMDID,
	WMI_P2P_DISC_OFFLOAD_APPIE_CMDID,
	WMI_P2P_DISC_OFFLOAD_PATTERN_CMDID,
	WMI_P2P_SET_OPPPS_PARAM_CMDID,
	WMI_P2P_LISTEN_OFFLOAD_START_CMDID,
	WMI_P2P_LISTEN_OFFLOAD_STOP_CMDID,
	WMI_AP_PS_PEER_PARAM_CMDID = WMI_TLV_CMD(WMI_GRP_AP_PS),
	WMI_AP_PS_PEER_UAPSD_COEX_CMDID,
	WMI_AP_PS_EGAP_PARAM_CMDID,
	WMI_PEER_RATE_RETRY_SCHED_CMDID = WMI_TLV_CMD(WMI_GRP_RATE_CTRL),
	WMI_WLAN_PROFILE_TRIGGER_CMDID = WMI_TLV_CMD(WMI_GRP_PROFILE),
	WMI_WLAN_PROFILE_SET_HIST_INTVL_CMDID,
	WMI_WLAN_PROFILE_GET_PROFILE_DATA_CMDID,
	WMI_WLAN_PROFILE_ENABLE_PROFILE_ID_CMDID,
	WMI_WLAN_PROFILE_LIST_PROFILE_ID_CMDID,
	WMI_PDEV_SUSPEND_CMDID = WMI_TLV_CMD(WMI_GRP_SUSPEND),
	WMI_PDEV_RESUME_CMDID,
	WMI_ADD_BCN_FILTER_CMDID = WMI_TLV_CMD(WMI_GRP_BCN_FILTER),
	WMI_RMV_BCN_FILTER_CMDID,
	WMI_WOW_ADD_WAKE_PATTERN_CMDID = WMI_TLV_CMD(WMI_GRP_WOW),
	WMI_WOW_DEL_WAKE_PATTERN_CMDID,
	WMI_WOW_ENABLE_DISABLE_WAKE_EVENT_CMDID,
	WMI_WOW_ENABLE_CMDID,
	WMI_WOW_HOSTWAKEUP_FROM_SLEEP_CMDID,
	WMI_WOW_IOAC_ADD_KEEPALIVE_CMDID,
	WMI_WOW_IOAC_DEL_KEEPALIVE_CMDID,
	WMI_WOW_IOAC_ADD_WAKE_PATTERN_CMDID,
	WMI_WOW_IOAC_DEL_WAKE_PATTERN_CMDID,
	WMI_D0_WOW_ENABLE_DISABLE_CMDID,
	WMI_EXTWOW_ENABLE_CMDID,
	WMI_EXTWOW_SET_APP_TYPE1_PARAMS_CMDID,
	WMI_EXTWOW_SET_APP_TYPE2_PARAMS_CMDID,
	WMI_WOW_ENABLE_ICMPV6_NA_FLT_CMDID,
	WMI_WOW_UDP_SVC_OFLD_CMDID,
	WMI_WOW_HOSTWAKEUP_GPIO_PIN_PATTERN_CONFIG_CMDID,
	WMI_WOW_SET_ACTION_WAKE_UP_CMDID,
	WMI_RTT_MEASREQ_CMDID = WMI_TLV_CMD(WMI_GRP_RTT),
	WMI_RTT_TSF_CMDID,
	WMI_VDEV_SPECTRAL_SCAN_CONFIGURE_CMDID = WMI_TLV_CMD(WMI_GRP_SPECTRAL),
	WMI_VDEV_SPECTRAL_SCAN_ENABLE_CMDID,
	WMI_REQUEST_STATS_CMDID = WMI_TLV_CMD(WMI_GRP_STATS),
	WMI_MCC_SCHED_TRAFFIC_STATS_CMDID,
	WMI_REQUEST_STATS_EXT_CMDID,
	WMI_REQUEST_LINK_STATS_CMDID,
	WMI_START_LINK_STATS_CMDID,
	WMI_CLEAR_LINK_STATS_CMDID,
	WMI_GET_FW_MEM_DUMP_CMDID,
	WMI_DEBUG_MESG_FLUSH_CMDID,
	WMI_DIAG_EVENT_LOG_CONFIG_CMDID,
	WMI_REQUEST_WLAN_STATS_CMDID,
	WMI_REQUEST_RCPI_CMDID,
	WMI_REQUEST_PEER_STATS_INFO_CMDID,
	WMI_REQUEST_RADIO_CHAN_STATS_CMDID,
	WMI_SET_ARP_NS_OFFLOAD_CMDID = WMI_TLV_CMD(WMI_GRP_ARP_NS_OFL),
	WMI_ADD_PROACTIVE_ARP_RSP_PATTERN_CMDID,
	WMI_DEL_PROACTIVE_ARP_RSP_PATTERN_CMDID,
	WMI_NETWORK_LIST_OFFLOAD_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_NLO_OFL),
	WMI_APFIND_CMDID,
	WMI_PASSPOINT_LIST_CONFIG_CMDID,
	WMI_NLO_CONFIGURE_MAWC_CMDID,
	WMI_GTK_OFFLOAD_CMDID = WMI_TLV_CMD(WMI_GRP_GTK_OFL),
	WMI_CSA_OFFLOAD_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_CSA_OFL),
	WMI_CSA_OFFLOAD_CHANSWITCH_CMDID,
	WMI_CHATTER_SET_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_CHATTER),
	WMI_CHATTER_ADD_COALESCING_FILTER_CMDID,
	WMI_CHATTER_DELETE_COALESCING_FILTER_CMDID,
	WMI_CHATTER_COALESCING_QUERY_CMDID,
	WMI_PEER_TID_ADDBA_CMDID = WMI_TLV_CMD(WMI_GRP_TID_ADDBA),
	WMI_PEER_TID_DELBA_CMDID,
	WMI_STA_DTIM_PS_METHOD_CMDID,
	WMI_STA_UAPSD_AUTO_TRIG_CMDID,
	WMI_STA_KEEPALIVE_CMDID,
	WMI_BA_REQ_SSN_CMDID,
	WMI_ECHO_CMDID = WMI_TLV_CMD(WMI_GRP_MISC),
	WMI_PDEV_UTF_CMDID,
	WMI_DBGLOG_CFG_CMDID,
	WMI_PDEV_QVIT_CMDID,
	WMI_PDEV_FTM_INTG_CMDID,
	WMI_VDEV_SET_KEEPALIVE_CMDID,
	WMI_VDEV_GET_KEEPALIVE_CMDID,
	WMI_FORCE_FW_HANG_CMDID,
	WMI_SET_MCASTBCAST_FILTER_CMDID,
	WMI_THERMAL_MGMT_CMDID,
	WMI_HOST_AUTO_SHUTDOWN_CFG_CMDID,
	WMI_TPC_CHAINMASK_CONFIG_CMDID,
	WMI_SET_ANTENNA_DIVERSITY_CMDID,
	WMI_OCB_SET_SCHED_CMDID,
	WMI_RSSI_BREACH_MONITOR_CONFIG_CMDID,
	WMI_LRO_CONFIG_CMDID,
	WMI_TRANSFER_DATA_TO_FLASH_CMDID,
	WMI_CONFIG_ENHANCED_MCAST_FILTER_CMDID,
	WMI_VDEV_WISA_CMDID,
	WMI_DBGLOG_TIME_STAMP_SYNC_CMDID,
	WMI_SET_MULTIPLE_MCAST_FILTER_CMDID,
	WMI_READ_DATA_FROM_FLASH_CMDID,
	WMI_THERM_THROT_SET_CONF_CMDID,
	WMI_RUNTIME_DPD_RECAL_CMDID,
	WMI_GET_TPC_POWER_CMDID,
	WMI_IDLE_TRIGGER_MONITOR_CMDID,
	WMI_GPIO_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_GPIO),
	WMI_GPIO_OUTPUT_CMDID,
	WMI_TXBF_CMDID,
	WMI_FWTEST_VDEV_MCC_SET_TBTT_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_FWTEST),
	WMI_FWTEST_P2P_SET_NOA_PARAM_CMDID,
	WMI_UNIT_TEST_CMDID,
	WMI_FWTEST_CMDID,
	WMI_QBOOST_CFG_CMDID,
	WMI_TDLS_SET_STATE_CMDID = WMI_TLV_CMD(WMI_GRP_TDLS),
	WMI_TDLS_PEER_UPDATE_CMDID,
	WMI_TDLS_SET_OFFCHAN_MODE_CMDID,
	WMI_RESMGR_ADAPTIVE_OCS_EN_DIS_CMDID = WMI_TLV_CMD(WMI_GRP_RESMGR),
	WMI_RESMGR_SET_CHAN_TIME_QUOTA_CMDID,
	WMI_RESMGR_SET_CHAN_LATENCY_CMDID,
	WMI_STA_SMPS_FORCE_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_STA_SMPS),
	WMI_STA_SMPS_PARAM_CMDID,
	WMI_HB_SET_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_WLAN_HB),
	WMI_HB_SET_TCP_PARAMS_CMDID,
	WMI_HB_SET_TCP_PKT_FILTER_CMDID,
	WMI_HB_SET_UDP_PARAMS_CMDID,
	WMI_HB_SET_UDP_PKT_FILTER_CMDID,
	WMI_RMC_SET_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_RMC),
	WMI_RMC_SET_ACTION_PERIOD_CMDID,
	WMI_RMC_CONFIG_CMDID,
	WMI_RMC_SET_MANUAL_LEADER_CMDID,
	WMI_MHF_OFFLOAD_SET_MODE_CMDID = WMI_TLV_CMD(WMI_GRP_MHF_OFL),
	WMI_MHF_OFFLOAD_PLUMB_ROUTING_TBL_CMDID,
	WMI_BATCH_SCAN_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_LOCATION_SCAN),
	WMI_BATCH_SCAN_DISABLE_CMDID,
	WMI_BATCH_SCAN_TRIGGER_RESULT_CMDID,
	WMI_OEM_REQ_CMDID = WMI_TLV_CMD(WMI_GRP_OEM),
	WMI_OEM_REQUEST_CMDID,
	WMI_LPI_OEM_REQ_CMDID,
	WMI_NAN_CMDID = WMI_TLV_CMD(WMI_GRP_NAN),
	WMI_MODEM_POWER_STATE_CMDID = WMI_TLV_CMD(WMI_GRP_COEX),
	WMI_CHAN_AVOID_UPDATE_CMDID,
	WMI_COEX_CONFIG_CMDID,
	WMI_CHAN_AVOID_RPT_ALLOW_CMDID,
	WMI_COEX_GET_ANTENNA_ISOLATION_CMDID,
	WMI_SAR_LIMITS_CMDID,
	WMI_OBSS_SCAN_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_OBSS_OFL),
	WMI_OBSS_SCAN_DISABLE_CMDID,
	WMI_OBSS_COLOR_COLLISION_DET_CONFIG_CMDID,
	WMI_LPI_MGMT_SNOOPING_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_LPI),
	WMI_LPI_START_SCAN_CMDID,
	WMI_LPI_STOP_SCAN_CMDID,
	WMI_EXTSCAN_START_CMDID = WMI_TLV_CMD(WMI_GRP_EXTSCAN),
	WMI_EXTSCAN_STOP_CMDID,
	WMI_EXTSCAN_CONFIGURE_WLAN_CHANGE_MONITOR_CMDID,
	WMI_EXTSCAN_CONFIGURE_HOTLIST_MONITOR_CMDID,
	WMI_EXTSCAN_GET_CACHED_RESULTS_CMDID,
	WMI_EXTSCAN_GET_WLAN_CHANGE_RESULTS_CMDID,
	WMI_EXTSCAN_SET_CAPABILITIES_CMDID,
	WMI_EXTSCAN_GET_CAPABILITIES_CMDID,
	WMI_EXTSCAN_CONFIGURE_HOTLIST_SSID_MONITOR_CMDID,
	WMI_EXTSCAN_CONFIGURE_MAWC_CMDID,
	WMI_SET_DHCP_SERVER_OFFLOAD_CMDID = WMI_TLV_CMD(WMI_GRP_DHCP_OFL),
	WMI_IPA_OFFLOAD_ENABLE_DISABLE_CMDID = WMI_TLV_CMD(WMI_GRP_IPA),
	WMI_MDNS_OFFLOAD_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_MDNS_OFL),
	WMI_MDNS_SET_FQDN_CMDID,
	WMI_MDNS_SET_RESPONSE_CMDID,
	WMI_MDNS_GET_STATS_CMDID,
	WMI_SAP_OFL_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_SAP_OFL),
	WMI_SAP_SET_BLACKLIST_PARAM_CMDID,
	WMI_OCB_SET_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_OCB),
	WMI_OCB_SET_UTC_TIME_CMDID,
	WMI_OCB_START_TIMING_ADVERT_CMDID,
	WMI_OCB_STOP_TIMING_ADVERT_CMDID,
	WMI_OCB_GET_TSF_TIMER_CMDID,
	WMI_DCC_GET_STATS_CMDID,
	WMI_DCC_CLEAR_STATS_CMDID,
	WMI_DCC_UPDATE_NDL_CMDID,
	WMI_SOC_SET_PCL_CMDID = WMI_TLV_CMD(WMI_GRP_SOC),
	WMI_SOC_SET_HW_MODE_CMDID,
	WMI_SOC_SET_DUAL_MAC_CONFIG_CMDID,
	WMI_SOC_SET_ANTENNA_MODE_CMDID,
	WMI_PACKET_FILTER_CONFIG_CMDID = WMI_TLV_CMD(WMI_GRP_PKT_FILTER),
	WMI_PACKET_FILTER_ENABLE_CMDID,
	WMI_MAWC_SENSOR_REPORT_IND_CMDID = WMI_TLV_CMD(WMI_GRP_MAWC),
	WMI_PMF_OFFLOAD_SET_SA_QUERY_CMDID = WMI_TLV_CMD(WMI_GRP_PMF_OFFLOAD),
	WMI_BPF_GET_CAPABILITY_CMDID = WMI_TLV_CMD(WMI_GRP_BPF_OFFLOAD),
	WMI_BPF_GET_VDEV_STATS_CMDID,
	WMI_BPF_SET_VDEV_INSTRUCTIONS_CMDID,
	WMI_BPF_DEL_VDEV_INSTRUCTIONS_CMDID,
	WMI_BPF_SET_VDEV_ACTIVE_MODE_CMDID,
	WMI_MNT_FILTER_CMDID = WMI_TLV_CMD(WMI_GRP_MONITOR),
	WMI_SET_CURRENT_COUNTRY_CMDID = WMI_TLV_CMD(WMI_GRP_REGULATORY),
	WMI_11D_SCAN_START_CMDID,
	WMI_11D_SCAN_STOP_CMDID,
	WMI_SET_INIT_COUNTRY_CMDID,
	WMI_NDI_GET_CAP_REQ_CMDID = WMI_TLV_CMD(WMI_GRP_PROTOTYPE),
	WMI_NDP_INITIATOR_REQ_CMDID,
	WMI_NDP_RESPONDER_REQ_CMDID,
	WMI_NDP_END_REQ_CMDID,
	WMI_HW_DATA_FILTER_CMDID = WMI_TLV_CMD(WMI_GRP_HW_DATA_FILTER),
	WMI_TWT_ENABLE_CMDID = WMI_TLV_CMD(WMI_GRP_TWT),
	WMI_TWT_DISABLE_CMDID,
	WMI_TWT_ADD_DIALOG_CMDID,
	WMI_TWT_DEL_DIALOG_CMDID,
	WMI_TWT_PAUSE_DIALOG_CMDID,
	WMI_TWT_RESUME_DIALOG_CMDID,
	WMI_PDEV_OBSS_PD_SPATIAL_REUSE_CMDID =
				WMI_TLV_CMD(WMI_GRP_SPATIAL_REUSE),
	WMI_PDEV_OBSS_PD_SPATIAL_REUSE_SET_DEF_OBSS_THRESH_CMDID,
};

enum wmi_tlv_event_id {
	WMI_SERVICE_READY_EVENTID = 0x1,
	WMI_READY_EVENTID,
	WMI_SERVICE_AVAILABLE_EVENTID,
	WMI_SCAN_EVENTID = WMI_EVT_GRP_START_ID(WMI_GRP_SCAN),
	WMI_PDEV_TPC_CONFIG_EVENTID = WMI_TLV_CMD(WMI_GRP_PDEV),
	WMI_CHAN_INFO_EVENTID,
	WMI_PHYERR_EVENTID,
	WMI_PDEV_DUMP_EVENTID,
	WMI_TX_PAUSE_EVENTID,
	WMI_DFS_RADAR_EVENTID,
	WMI_PDEV_L1SS_TRACK_EVENTID,
	WMI_PDEV_TEMPERATURE_EVENTID,
	WMI_SERVICE_READY_EXT_EVENTID,
	WMI_PDEV_FIPS_EVENTID,
	WMI_PDEV_CHANNEL_HOPPING_EVENTID,
	WMI_PDEV_ANI_CCK_LEVEL_EVENTID,
	WMI_PDEV_ANI_OFDM_LEVEL_EVENTID,
	WMI_PDEV_TPC_EVENTID,
	WMI_PDEV_NFCAL_POWER_ALL_CHANNELS_EVENTID,
	WMI_PDEV_SET_HW_MODE_RESP_EVENTID,
	WMI_PDEV_HW_MODE_TRANSITION_EVENTID,
	WMI_PDEV_SET_MAC_CONFIG_RESP_EVENTID,
	WMI_PDEV_ANTDIV_STATUS_EVENTID,
	WMI_PDEV_CHIP_POWER_STATS_EVENTID,
	WMI_PDEV_CHIP_POWER_SAVE_FAILURE_DETECTED_EVENTID,
	WMI_PDEV_CSA_SWITCH_COUNT_STATUS_EVENTID,
	WMI_PDEV_CHECK_CAL_VERSION_EVENTID,
	WMI_PDEV_DIV_RSSI_ANTID_EVENTID,
	WMI_PDEV_BSS_CHAN_INFO_EVENTID,
	WMI_PDEV_UPDATE_CTLTABLE_EVENTID,
	WMI_PDEV_DMA_RING_CFG_RSP_EVENTID,
	WMI_PDEV_DMA_RING_BUF_RELEASE_EVENTID,
	WMI_PDEV_CTL_FAILSAFE_CHECK_EVENTID,
	WMI_PDEV_CSC_SWITCH_COUNT_STATUS_EVENTID,
	WMI_PDEV_COLD_BOOT_CAL_DATA_EVENTID,
	WMI_PDEV_RAP_INFO_EVENTID,
	WMI_CHAN_RF_CHARACTERIZATION_INFO_EVENTID,
	WMI_SERVICE_READY_EXT2_EVENTID,
	WMI_VDEV_START_RESP_EVENTID = WMI_TLV_CMD(WMI_GRP_VDEV),
	WMI_VDEV_STOPPED_EVENTID,
	WMI_VDEV_INSTALL_KEY_COMPLETE_EVENTID,
	WMI_VDEV_MCC_BCN_INTERVAL_CHANGE_REQ_EVENTID,
	WMI_VDEV_TSF_REPORT_EVENTID,
	WMI_VDEV_DELETE_RESP_EVENTID,
	WMI_VDEV_ENCRYPT_DECRYPT_DATA_RESP_EVENTID,
	WMI_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_STATUS_EVENTID,
	WMI_PEER_STA_KICKOUT_EVENTID = WMI_TLV_CMD(WMI_GRP_PEER),
	WMI_PEER_INFO_EVENTID,
	WMI_PEER_TX_FAIL_CNT_THR_EVENTID,
	WMI_PEER_ESTIMATED_LINKSPEED_EVENTID,
	WMI_PEER_STATE_EVENTID,
	WMI_PEER_ASSOC_CONF_EVENTID,
	WMI_PEER_DELETE_RESP_EVENTID,
	WMI_PEER_RATECODE_LIST_EVENTID,
	WMI_WDS_PEER_EVENTID,
	WMI_PEER_STA_PS_STATECHG_EVENTID,
	WMI_PEER_ANTDIV_INFO_EVENTID,
	WMI_PEER_RESERVED0_EVENTID,
	WMI_PEER_RESERVED1_EVENTID,
	WMI_PEER_RESERVED2_EVENTID,
	WMI_PEER_RESERVED3_EVENTID,
	WMI_PEER_RESERVED4_EVENTID,
	WMI_PEER_RESERVED5_EVENTID,
	WMI_PEER_RESERVED6_EVENTID,
	WMI_PEER_RESERVED7_EVENTID,
	WMI_PEER_RESERVED8_EVENTID,
	WMI_PEER_RESERVED9_EVENTID,
	WMI_PEER_RESERVED10_EVENTID,
	WMI_PEER_OPER_MODE_CHANGE_EVENTID,
	WMI_PEER_TX_PN_RESPONSE_EVENTID,
	WMI_PEER_CFR_CAPTURE_EVENTID,
	WMI_PEER_CREATE_CONF_EVENTID,
	WMI_MGMT_RX_EVENTID = WMI_TLV_CMD(WMI_GRP_MGMT),
	WMI_HOST_SWBA_EVENTID,
	WMI_TBTTOFFSET_UPDATE_EVENTID,
	WMI_OFFLOAD_BCN_TX_STATUS_EVENTID,
	WMI_OFFLOAD_PROB_RESP_TX_STATUS_EVENTID,
	WMI_MGMT_TX_COMPLETION_EVENTID,
	WMI_MGMT_TX_BUNDLE_COMPLETION_EVENTID,
	WMI_TBTTOFFSET_EXT_UPDATE_EVENTID,
	WMI_OFFCHAN_DATA_TX_COMPLETION_EVENTID,
	WMI_HOST_FILS_DISCOVERY_EVENTID,
	WMI_TX_DELBA_COMPLETE_EVENTID = WMI_TLV_CMD(WMI_GRP_BA_NEG),
	WMI_TX_ADDBA_COMPLETE_EVENTID,
	WMI_BA_RSP_SSN_EVENTID,
	WMI_AGGR_STATE_TRIG_EVENTID,
	WMI_ROAM_EVENTID = WMI_TLV_CMD(WMI_GRP_ROAM),
	WMI_PROFILE_MATCH,
	WMI_ROAM_SYNCH_EVENTID,
	WMI_P2P_DISC_EVENTID = WMI_TLV_CMD(WMI_GRP_P2P),
	WMI_P2P_NOA_EVENTID,
	WMI_P2P_LISTEN_OFFLOAD_STOPPED_EVENTID,
	WMI_AP_PS_EGAP_INFO_EVENTID = WMI_TLV_CMD(WMI_GRP_AP_PS),
	WMI_PDEV_RESUME_EVENTID = WMI_TLV_CMD(WMI_GRP_SUSPEND),
	WMI_WOW_WAKEUP_HOST_EVENTID = WMI_TLV_CMD(WMI_GRP_WOW),
	WMI_D0_WOW_DISABLE_ACK_EVENTID,
	WMI_WOW_INITIAL_WAKEUP_EVENTID,
	WMI_RTT_MEASUREMENT_REPORT_EVENTID = WMI_TLV_CMD(WMI_GRP_RTT),
	WMI_TSF_MEASUREMENT_REPORT_EVENTID,
	WMI_RTT_ERROR_REPORT_EVENTID,
	WMI_STATS_EXT_EVENTID = WMI_TLV_CMD(WMI_GRP_STATS),
	WMI_IFACE_LINK_STATS_EVENTID,
	WMI_PEER_LINK_STATS_EVENTID,
	WMI_RADIO_LINK_STATS_EVENTID,
	WMI_UPDATE_FW_MEM_DUMP_EVENTID,
	WMI_DIAG_EVENT_LOG_SUPPORTED_EVENTID,
	WMI_INST_RSSI_STATS_EVENTID,
	WMI_RADIO_TX_POWER_LEVEL_STATS_EVENTID,
	WMI_REPORT_STATS_EVENTID,
	WMI_UPDATE_RCPI_EVENTID,
	WMI_PEER_STATS_INFO_EVENTID,
	WMI_RADIO_CHAN_STATS_EVENTID,
	WMI_NLO_MATCH_EVENTID = WMI_TLV_CMD(WMI_GRP_NLO_OFL),
	WMI_NLO_SCAN_COMPLETE_EVENTID,
	WMI_APFIND_EVENTID,
	WMI_PASSPOINT_MATCH_EVENTID,
	WMI_GTK_OFFLOAD_STATUS_EVENTID = WMI_TLV_CMD(WMI_GRP_GTK_OFL),
	WMI_GTK_REKEY_FAIL_EVENTID,
	WMI_CSA_HANDLING_EVENTID = WMI_TLV_CMD(WMI_GRP_CSA_OFL),
	WMI_CHATTER_PC_QUERY_EVENTID = WMI_TLV_CMD(WMI_GRP_CHATTER),
	WMI_PDEV_DFS_RADAR_DETECTION_EVENTID = WMI_TLV_CMD(WMI_GRP_DFS),
	WMI_VDEV_DFS_CAC_COMPLETE_EVENTID,
	WMI_VDEV_ADFS_OCAC_COMPLETE_EVENTID,
	WMI_ECHO_EVENTID = WMI_TLV_CMD(WMI_GRP_MISC),
	WMI_PDEV_UTF_EVENTID,
	WMI_DEBUG_MESG_EVENTID,
	WMI_UPDATE_STATS_EVENTID,
	WMI_DEBUG_PRINT_EVENTID,
	WMI_DCS_INTERFERENCE_EVENTID,
	WMI_PDEV_QVIT_EVENTID,
	WMI_WLAN_PROFILE_DATA_EVENTID,
	WMI_PDEV_FTM_INTG_EVENTID,
	WMI_WLAN_FREQ_AVOID_EVENTID,
	WMI_VDEV_GET_KEEPALIVE_EVENTID,
	WMI_THERMAL_MGMT_EVENTID,
	WMI_DIAG_DATA_CONTAINER_EVENTID,
	WMI_HOST_AUTO_SHUTDOWN_EVENTID,
	WMI_UPDATE_WHAL_MIB_STATS_EVENTID,
	WMI_UPDATE_VDEV_RATE_STATS_EVENTID,
	WMI_DIAG_EVENTID,
	WMI_OCB_SET_SCHED_EVENTID,
	WMI_DEBUG_MESG_FLUSH_COMPLETE_EVENTID,
	WMI_RSSI_BREACH_EVENTID,
	WMI_TRANSFER_DATA_TO_FLASH_COMPLETE_EVENTID,
	WMI_PDEV_UTF_SCPC_EVENTID,
	WMI_READ_DATA_FROM_FLASH_EVENTID,
	WMI_REPORT_RX_AGGR_FAILURE_EVENTID,
	WMI_PKGID_EVENTID,
	WMI_GPIO_INPUT_EVENTID = WMI_TLV_CMD(WMI_GRP_GPIO),
	WMI_UPLOADH_EVENTID,
	WMI_CAPTUREH_EVENTID,
	WMI_RFKILL_STATE_CHANGE_EVENTID,
	WMI_TDLS_PEER_EVENTID = WMI_TLV_CMD(WMI_GRP_TDLS),
	WMI_STA_SMPS_FORCE_MODE_COMPL_EVENTID = WMI_TLV_CMD(WMI_GRP_STA_SMPS),
	WMI_BATCH_SCAN_ENABLED_EVENTID = WMI_TLV_CMD(WMI_GRP_LOCATION_SCAN),
	WMI_BATCH_SCAN_RESULT_EVENTID,
	WMI_OEM_CAPABILITY_EVENTID = WMI_TLV_CMD(WMI_GRP_OEM),
	WMI_OEM_MEASUREMENT_REPORT_EVENTID,
	WMI_OEM_ERROR_REPORT_EVENTID,
	WMI_OEM_RESPONSE_EVENTID,
	WMI_NAN_EVENTID = WMI_TLV_CMD(WMI_GRP_NAN),
	WMI_NAN_DISC_IFACE_CREATED_EVENTID,
	WMI_NAN_DISC_IFACE_DELETED_EVENTID,
	WMI_NAN_STARTED_CLUSTER_EVENTID,
	WMI_NAN_JOINED_CLUSTER_EVENTID,
	WMI_COEX_REPORT_ANTENNA_ISOLATION_EVENTID = WMI_TLV_CMD(WMI_GRP_COEX),
	WMI_LPI_RESULT_EVENTID = WMI_TLV_CMD(WMI_GRP_LPI),
	WMI_LPI_STATUS_EVENTID,
	WMI_LPI_HANDOFF_EVENTID,
	WMI_EXTSCAN_START_STOP_EVENTID = WMI_TLV_CMD(WMI_GRP_EXTSCAN),
	WMI_EXTSCAN_OPERATION_EVENTID,
	WMI_EXTSCAN_TABLE_USAGE_EVENTID,
	WMI_EXTSCAN_CACHED_RESULTS_EVENTID,
	WMI_EXTSCAN_WLAN_CHANGE_RESULTS_EVENTID,
	WMI_EXTSCAN_HOTLIST_MATCH_EVENTID,
	WMI_EXTSCAN_CAPABILITIES_EVENTID,
	WMI_EXTSCAN_HOTLIST_SSID_MATCH_EVENTID,
	WMI_MDNS_STATS_EVENTID = WMI_TLV_CMD(WMI_GRP_MDNS_OFL),
	WMI_SAP_OFL_ADD_STA_EVENTID = WMI_TLV_CMD(WMI_GRP_SAP_OFL),
	WMI_SAP_OFL_DEL_STA_EVENTID,
	WMI_OBSS_COLOR_COLLISION_DETECTION_EVENTID =
		WMI_EVT_GRP_START_ID(WMI_GRP_OBSS_OFL),
	WMI_OCB_SET_CONFIG_RESP_EVENTID = WMI_TLV_CMD(WMI_GRP_OCB),
	WMI_OCB_GET_TSF_TIMER_RESP_EVENTID,
	WMI_DCC_GET_STATS_RESP_EVENTID,
	WMI_DCC_UPDATE_NDL_RESP_EVENTID,
	WMI_DCC_STATS_EVENTID,
	WMI_SOC_SET_HW_MODE_RESP_EVENTID = WMI_TLV_CMD(WMI_GRP_SOC),
	WMI_SOC_HW_MODE_TRANSITION_EVENTID,
	WMI_SOC_SET_DUAL_MAC_CONFIG_RESP_EVENTID,
	WMI_MAWC_ENABLE_SENSOR_EVENTID = WMI_TLV_CMD(WMI_GRP_MAWC),
	WMI_BPF_CAPABILIY_INFO_EVENTID = WMI_TLV_CMD(WMI_GRP_BPF_OFFLOAD),
	WMI_BPF_VDEV_STATS_INFO_EVENTID,
	WMI_RMC_NEW_LEADER_EVENTID = WMI_TLV_CMD(WMI_GRP_RMC),
	WMI_REG_CHAN_LIST_CC_EVENTID = WMI_TLV_CMD(WMI_GRP_REGULATORY),
	WMI_11D_NEW_COUNTRY_EVENTID,
	WMI_REG_CHAN_LIST_CC_EXT_EVENTID,
	WMI_NDI_CAP_RSP_EVENTID = WMI_TLV_CMD(WMI_GRP_PROTOTYPE),
	WMI_NDP_INITIATOR_RSP_EVENTID,
	WMI_NDP_RESPONDER_RSP_EVENTID,
	WMI_NDP_END_RSP_EVENTID,
	WMI_NDP_INDICATION_EVENTID,
	WMI_NDP_CONFIRM_EVENTID,
	WMI_NDP_END_INDICATION_EVENTID,

	WMI_TWT_ENABLE_EVENTID = WMI_TLV_CMD(WMI_GRP_TWT),
	WMI_TWT_DISABLE_EVENTID,
	WMI_TWT_ADD_DIALOG_EVENTID,
	WMI_TWT_DEL_DIALOG_EVENTID,
	WMI_TWT_PAUSE_DIALOG_EVENTID,
	WMI_TWT_RESUME_DIALOG_EVENTID,
};

enum wmi_tlv_pdev_param {
	WMI_PDEV_PARAM_TX_CHAIN_MASK = 0x1,
	WMI_PDEV_PARAM_RX_CHAIN_MASK,
	WMI_PDEV_PARAM_TXPOWER_LIMIT2G,
	WMI_PDEV_PARAM_TXPOWER_LIMIT5G,
	WMI_PDEV_PARAM_TXPOWER_SCALE,
	WMI_PDEV_PARAM_BEACON_GEN_MODE,
	WMI_PDEV_PARAM_BEACON_TX_MODE,
	WMI_PDEV_PARAM_RESMGR_OFFCHAN_MODE,
	WMI_PDEV_PARAM_PROTECTION_MODE,
	WMI_PDEV_PARAM_DYNAMIC_BW,
	WMI_PDEV_PARAM_NON_AGG_SW_RETRY_TH,
	WMI_PDEV_PARAM_AGG_SW_RETRY_TH,
	WMI_PDEV_PARAM_STA_KICKOUT_TH,
	WMI_PDEV_PARAM_AC_AGGRSIZE_SCALING,
	WMI_PDEV_PARAM_LTR_ENABLE,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_BE,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_BK,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_VI,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_VO,
	WMI_PDEV_PARAM_LTR_AC_LATENCY_TIMEOUT,
	WMI_PDEV_PARAM_LTR_SLEEP_OVERRIDE,
	WMI_PDEV_PARAM_LTR_RX_OVERRIDE,
	WMI_PDEV_PARAM_LTR_TX_ACTIVITY_TIMEOUT,
	WMI_PDEV_PARAM_L1SS_ENABLE,
	WMI_PDEV_PARAM_DSLEEP_ENABLE,
	WMI_PDEV_PARAM_PCIELP_TXBUF_FLUSH,
	WMI_PDEV_PARAM_PCIELP_TXBUF_WATERMARK,
	WMI_PDEV_PARAM_PCIELP_TXBUF_TMO_EN,
	WMI_PDEV_PARAM_PCIELP_TXBUF_TMO_VALUE,
	WMI_PDEV_PARAM_PDEV_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_VDEV_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_PEER_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_BCNFLT_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_PMF_QOS,
	WMI_PDEV_PARAM_ARP_AC_OVERRIDE,
	WMI_PDEV_PARAM_DCS,
	WMI_PDEV_PARAM_ANI_ENABLE,
	WMI_PDEV_PARAM_ANI_POLL_PERIOD,
	WMI_PDEV_PARAM_ANI_LISTEN_PERIOD,
	WMI_PDEV_PARAM_ANI_OFDM_LEVEL,
	WMI_PDEV_PARAM_ANI_CCK_LEVEL,
	WMI_PDEV_PARAM_DYNTXCHAIN,
	WMI_PDEV_PARAM_PROXY_STA,
	WMI_PDEV_PARAM_IDLE_PS_CONFIG,
	WMI_PDEV_PARAM_POWER_GATING_SLEEP,
	WMI_PDEV_PARAM_RFKILL_ENABLE,
	WMI_PDEV_PARAM_BURST_DUR,
	WMI_PDEV_PARAM_BURST_ENABLE,
	WMI_PDEV_PARAM_HW_RFKILL_CONFIG,
	WMI_PDEV_PARAM_LOW_POWER_RF_ENABLE,
	WMI_PDEV_PARAM_L1SS_TRACK,
	WMI_PDEV_PARAM_HYST_EN,
	WMI_PDEV_PARAM_POWER_COLLAPSE_ENABLE,
	WMI_PDEV_PARAM_LED_SYS_STATE,
	WMI_PDEV_PARAM_LED_ENABLE,
	WMI_PDEV_PARAM_AUDIO_OVER_WLAN_LATENCY,
	WMI_PDEV_PARAM_AUDIO_OVER_WLAN_ENABLE,
	WMI_PDEV_PARAM_WHAL_MIB_STATS_UPDATE_ENABLE,
	WMI_PDEV_PARAM_VDEV_RATE_STATS_UPDATE_PERIOD,
	WMI_PDEV_PARAM_CTS_CBW,
	WMI_PDEV_PARAM_WNTS_CONFIG,
	WMI_PDEV_PARAM_ADAPTIVE_EARLY_RX_ENABLE,
	WMI_PDEV_PARAM_ADAPTIVE_EARLY_RX_MIN_SLEEP_SLOP,
	WMI_PDEV_PARAM_ADAPTIVE_EARLY_RX_INC_DEC_STEP,
	WMI_PDEV_PARAM_EARLY_RX_FIX_SLEEP_SLOP,
	WMI_PDEV_PARAM_BMISS_BASED_ADAPTIVE_BTO_ENABLE,
	WMI_PDEV_PARAM_BMISS_BTO_MIN_BCN_TIMEOUT,
	WMI_PDEV_PARAM_BMISS_BTO_INC_DEC_STEP,
	WMI_PDEV_PARAM_BTO_FIX_BCN_TIMEOUT,
	WMI_PDEV_PARAM_CE_BASED_ADAPTIVE_BTO_ENABLE,
	WMI_PDEV_PARAM_CE_BTO_COMBO_CE_VALUE,
	WMI_PDEV_PARAM_TX_CHAIN_MASK_2G,
	WMI_PDEV_PARAM_RX_CHAIN_MASK_2G,
	WMI_PDEV_PARAM_TX_CHAIN_MASK_5G,
	WMI_PDEV_PARAM_RX_CHAIN_MASK_5G,
	WMI_PDEV_PARAM_TX_CHAIN_MASK_CCK,
	WMI_PDEV_PARAM_TX_CHAIN_MASK_1SS,
	WMI_PDEV_PARAM_CTS2SELF_FOR_P2P_GO_CONFIG,
	WMI_PDEV_PARAM_TXPOWER_DECR_DB,
	WMI_PDEV_PARAM_AGGR_BURST,
	WMI_PDEV_PARAM_RX_DECAP_MODE,
	WMI_PDEV_PARAM_FAST_CHANNEL_RESET,
	WMI_PDEV_PARAM_SMART_ANTENNA_DEFAULT_ANTENNA,
	WMI_PDEV_PARAM_ANTENNA_GAIN,
	WMI_PDEV_PARAM_RX_FILTER,
	WMI_PDEV_SET_MCAST_TO_UCAST_TID,
	WMI_PDEV_PARAM_PROXY_STA_MODE,
	WMI_PDEV_PARAM_SET_MCAST2UCAST_MODE,
	WMI_PDEV_PARAM_SET_MCAST2UCAST_BUFFER,
	WMI_PDEV_PARAM_REMOVE_MCAST2UCAST_BUFFER,
	WMI_PDEV_PEER_STA_PS_STATECHG_ENABLE,
	WMI_PDEV_PARAM_IGMPMLD_AC_OVERRIDE,
	WMI_PDEV_PARAM_BLOCK_INTERBSS,
	WMI_PDEV_PARAM_SET_DISABLE_RESET_CMDID,
	WMI_PDEV_PARAM_SET_MSDU_TTL_CMDID,
	WMI_PDEV_PARAM_SET_PPDU_DURATION_CMDID,
	WMI_PDEV_PARAM_TXBF_SOUND_PERIOD_CMDID,
	WMI_PDEV_PARAM_SET_PROMISC_MODE_CMDID,
	WMI_PDEV_PARAM_SET_BURST_MODE_CMDID,
	WMI_PDEV_PARAM_EN_STATS,
	WMI_PDEV_PARAM_MU_GROUP_POLICY,
	WMI_PDEV_PARAM_NOISE_DETECTION,
	WMI_PDEV_PARAM_NOISE_THRESHOLD,
	WMI_PDEV_PARAM_DPD_ENABLE,
	WMI_PDEV_PARAM_SET_MCAST_BCAST_ECHO,
	WMI_PDEV_PARAM_ATF_STRICT_SCH,
	WMI_PDEV_PARAM_ATF_SCHED_DURATION,
	WMI_PDEV_PARAM_ANT_PLZN,
	WMI_PDEV_PARAM_MGMT_RETRY_LIMIT,
	WMI_PDEV_PARAM_SENSITIVITY_LEVEL,
	WMI_PDEV_PARAM_SIGNED_TXPOWER_2G,
	WMI_PDEV_PARAM_SIGNED_TXPOWER_5G,
	WMI_PDEV_PARAM_ENABLE_PER_TID_AMSDU,
	WMI_PDEV_PARAM_ENABLE_PER_TID_AMPDU,
	WMI_PDEV_PARAM_CCA_THRESHOLD,
	WMI_PDEV_PARAM_RTS_FIXED_RATE,
	WMI_PDEV_PARAM_PDEV_RESET,
	WMI_PDEV_PARAM_WAPI_MBSSID_OFFSET,
	WMI_PDEV_PARAM_ARP_DBG_SRCADDR,
	WMI_PDEV_PARAM_ARP_DBG_DSTADDR,
	WMI_PDEV_PARAM_ATF_OBSS_NOISE_SCH,
	WMI_PDEV_PARAM_ATF_OBSS_NOISE_SCALING_FACTOR,
	WMI_PDEV_PARAM_CUST_TXPOWER_SCALE,
	WMI_PDEV_PARAM_ATF_DYNAMIC_ENABLE,
	WMI_PDEV_PARAM_CTRL_RETRY_LIMIT,
	WMI_PDEV_PARAM_PROPAGATION_DELAY,
	WMI_PDEV_PARAM_ENA_ANT_DIV,
	WMI_PDEV_PARAM_FORCE_CHAIN_ANT,
	WMI_PDEV_PARAM_ANT_DIV_SELFTEST,
	WMI_PDEV_PARAM_ANT_DIV_SELFTEST_INTVL,
	WMI_PDEV_PARAM_STATS_OBSERVATION_PERIOD,
	WMI_PDEV_PARAM_TX_PPDU_DELAY_BIN_SIZE_MS,
	WMI_PDEV_PARAM_TX_PPDU_DELAY_ARRAY_LEN,
	WMI_PDEV_PARAM_TX_MPDU_AGGR_ARRAY_LEN,
	WMI_PDEV_PARAM_RX_MPDU_AGGR_ARRAY_LEN,
	WMI_PDEV_PARAM_TX_SCH_DELAY,
	WMI_PDEV_PARAM_ENABLE_RTS_SIFS_BURSTING,
	WMI_PDEV_PARAM_MAX_MPDUS_IN_AMPDU,
	WMI_PDEV_PARAM_PEER_STATS_INFO_ENABLE,
	WMI_PDEV_PARAM_FAST_PWR_TRANSITION,
	WMI_PDEV_PARAM_RADIO_CHAN_STATS_ENABLE,
	WMI_PDEV_PARAM_RADIO_DIAGNOSIS_ENABLE,
	WMI_PDEV_PARAM_MESH_MCAST_ENABLE,
	WMI_PDEV_PARAM_SET_CMD_OBSS_PD_THRESHOLD = 0xbc,
	WMI_PDEV_PARAM_SET_CMD_OBSS_PD_PER_AC = 0xbe,
	WMI_PDEV_PARAM_ENABLE_SR_PROHIBIT = 0xc6,
};

enum wmi_tlv_vdev_param {
	WMI_VDEV_PARAM_RTS_THRESHOLD = 0x1,
	WMI_VDEV_PARAM_FRAGMENTATION_THRESHOLD,
	WMI_VDEV_PARAM_BEACON_INTERVAL,
	WMI_VDEV_PARAM_LISTEN_INTERVAL,
	WMI_VDEV_PARAM_MULTICAST_RATE,
	WMI_VDEV_PARAM_MGMT_TX_RATE,
	WMI_VDEV_PARAM_SLOT_TIME,
	WMI_VDEV_PARAM_PREAMBLE,
	WMI_VDEV_PARAM_SWBA_TIME,
	WMI_VDEV_STATS_UPDATE_PERIOD,
	WMI_VDEV_PWRSAVE_AGEOUT_TIME,
	WMI_VDEV_HOST_SWBA_INTERVAL,
	WMI_VDEV_PARAM_DTIM_PERIOD,
	WMI_VDEV_OC_SCHEDULER_AIR_TIME_LIMIT,
	WMI_VDEV_PARAM_WDS,
	WMI_VDEV_PARAM_ATIM_WINDOW,
	WMI_VDEV_PARAM_BMISS_COUNT_MAX,
	WMI_VDEV_PARAM_BMISS_FIRST_BCNT,
	WMI_VDEV_PARAM_BMISS_FINAL_BCNT,
	WMI_VDEV_PARAM_FEATURE_WMM,
	WMI_VDEV_PARAM_CHWIDTH,
	WMI_VDEV_PARAM_CHEXTOFFSET,
	WMI_VDEV_PARAM_DISABLE_HTPROTECTION,
	WMI_VDEV_PARAM_STA_QUICKKICKOUT,
	WMI_VDEV_PARAM_MGMT_RATE,
	WMI_VDEV_PARAM_PROTECTION_MODE,
	WMI_VDEV_PARAM_FIXED_RATE,
	WMI_VDEV_PARAM_SGI,
	WMI_VDEV_PARAM_LDPC,
	WMI_VDEV_PARAM_TX_STBC,
	WMI_VDEV_PARAM_RX_STBC,
	WMI_VDEV_PARAM_INTRA_BSS_FWD,
	WMI_VDEV_PARAM_DEF_KEYID,
	WMI_VDEV_PARAM_NSS,
	WMI_VDEV_PARAM_BCAST_DATA_RATE,
	WMI_VDEV_PARAM_MCAST_DATA_RATE,
	WMI_VDEV_PARAM_MCAST_INDICATE,
	WMI_VDEV_PARAM_DHCP_INDICATE,
	WMI_VDEV_PARAM_UNKNOWN_DEST_INDICATE,
	WMI_VDEV_PARAM_AP_KEEPALIVE_MIN_IDLE_INACTIVE_TIME_SECS,
	WMI_VDEV_PARAM_AP_KEEPALIVE_MAX_IDLE_INACTIVE_TIME_SECS,
	WMI_VDEV_PARAM_AP_KEEPALIVE_MAX_UNRESPONSIVE_TIME_SECS,
	WMI_VDEV_PARAM_AP_ENABLE_NAWDS,
	WMI_VDEV_PARAM_ENABLE_RTSCTS,
	WMI_VDEV_PARAM_TXBF,
	WMI_VDEV_PARAM_PACKET_POWERSAVE,
	WMI_VDEV_PARAM_DROP_UNENCRY,
	WMI_VDEV_PARAM_TX_ENCAP_TYPE,
	WMI_VDEV_PARAM_AP_DETECT_OUT_OF_SYNC_SLEEPING_STA_TIME_SECS,
	WMI_VDEV_PARAM_EARLY_RX_ADJUST_ENABLE,
	WMI_VDEV_PARAM_EARLY_RX_TGT_BMISS_NUM,
	WMI_VDEV_PARAM_EARLY_RX_BMISS_SAMPLE_CYCLE,
	WMI_VDEV_PARAM_EARLY_RX_SLOP_STEP,
	WMI_VDEV_PARAM_EARLY_RX_INIT_SLOP,
	WMI_VDEV_PARAM_EARLY_RX_ADJUST_PAUSE,
	WMI_VDEV_PARAM_TX_PWRLIMIT,
	WMI_VDEV_PARAM_SNR_NUM_FOR_CAL,
	WMI_VDEV_PARAM_ROAM_FW_OFFLOAD,
	WMI_VDEV_PARAM_ENABLE_RMC,
	WMI_VDEV_PARAM_IBSS_MAX_BCN_LOST_MS,
	WMI_VDEV_PARAM_MAX_RATE,
	WMI_VDEV_PARAM_EARLY_RX_DRIFT_SAMPLE,
	WMI_VDEV_PARAM_SET_IBSS_TX_FAIL_CNT_THR,
	WMI_VDEV_PARAM_EBT_RESYNC_TIMEOUT,
	WMI_VDEV_PARAM_AGGR_TRIG_EVENT_ENABLE,
	WMI_VDEV_PARAM_IS_IBSS_POWER_SAVE_ALLOWED,
	WMI_VDEV_PARAM_IS_POWER_COLLAPSE_ALLOWED,
	WMI_VDEV_PARAM_IS_AWAKE_ON_TXRX_ENABLED,
	WMI_VDEV_PARAM_INACTIVITY_CNT,
	WMI_VDEV_PARAM_TXSP_END_INACTIVITY_TIME_MS,
	WMI_VDEV_PARAM_DTIM_POLICY,
	WMI_VDEV_PARAM_IBSS_PS_WARMUP_TIME_SECS,
	WMI_VDEV_PARAM_IBSS_PS_1RX_CHAIN_IN_ATIM_WINDOW_ENABLE,
	WMI_VDEV_PARAM_RX_LEAK_WINDOW,
	WMI_VDEV_PARAM_STATS_AVG_FACTOR,
	WMI_VDEV_PARAM_DISCONNECT_TH,
	WMI_VDEV_PARAM_RTSCTS_RATE,
	WMI_VDEV_PARAM_MCC_RTSCTS_PROTECTION_ENABLE,
	WMI_VDEV_PARAM_MCC_BROADCAST_PROBE_ENABLE,
	WMI_VDEV_PARAM_TXPOWER_SCALE,
	WMI_VDEV_PARAM_TXPOWER_SCALE_DECR_DB,
	WMI_VDEV_PARAM_MCAST2UCAST_SET,
	WMI_VDEV_PARAM_RC_NUM_RETRIES,
	WMI_VDEV_PARAM_CABQ_MAXDUR,
	WMI_VDEV_PARAM_MFPTEST_SET,
	WMI_VDEV_PARAM_RTS_FIXED_RATE,
	WMI_VDEV_PARAM_VHT_SGIMASK,
	WMI_VDEV_PARAM_VHT80_RATEMASK,
	WMI_VDEV_PARAM_PROXY_STA,
	WMI_VDEV_PARAM_VIRTUAL_CELL_MODE,
	WMI_VDEV_PARAM_RX_DECAP_TYPE,
	WMI_VDEV_PARAM_BW_NSS_RATEMASK,
	WMI_VDEV_PARAM_SENSOR_AP,
	WMI_VDEV_PARAM_BEACON_RATE,
	WMI_VDEV_PARAM_DTIM_ENABLE_CTS,
	WMI_VDEV_PARAM_STA_KICKOUT,
	WMI_VDEV_PARAM_CAPABILITIES,
	WMI_VDEV_PARAM_TSF_INCREMENT,
	WMI_VDEV_PARAM_AMPDU_PER_AC,
	WMI_VDEV_PARAM_RX_FILTER,
	WMI_VDEV_PARAM_MGMT_TX_POWER,
	WMI_VDEV_PARAM_NON_AGG_SW_RETRY_TH,
	WMI_VDEV_PARAM_AGG_SW_RETRY_TH,
	WMI_VDEV_PARAM_DISABLE_DYN_BW_RTS,
	WMI_VDEV_PARAM_ATF_SSID_SCHED_POLICY,
	WMI_VDEV_PARAM_HE_DCM,
	WMI_VDEV_PARAM_HE_RANGE_EXT,
	WMI_VDEV_PARAM_ENABLE_BCAST_PROBE_RESPONSE,
	WMI_VDEV_PARAM_FILS_MAX_CHANNEL_GUARD_TIME,
	WMI_VDEV_PARAM_HE_LTF = 0x74,
	WMI_VDEV_PARAM_ENABLE_DISABLE_RTT_RESPONDER_ROLE = 0x7d,
	WMI_VDEV_PARAM_BA_MODE = 0x7e,
	WMI_VDEV_PARAM_AUTORATE_MISC_CFG = 0x80,
	WMI_VDEV_PARAM_SET_HE_SOUNDING_MODE = 0x87,
	WMI_VDEV_PARAM_6GHZ_PARAMS = 0x99,
	WMI_VDEV_PARAM_PROTOTYPE = 0x8000,
	WMI_VDEV_PARAM_BSS_COLOR,
	WMI_VDEV_PARAM_SET_HEMU_MODE,
	WMI_VDEV_PARAM_HEOPS_0_31 = 0x8003,
};

enum wmi_tlv_peer_flags {
	WMI_PEER_AUTH		= 0x00000001,
	WMI_PEER_QOS		= 0x00000002,
	WMI_PEER_NEED_PTK_4_WAY	= 0x00000004,
	WMI_PEER_NEED_GTK_2_WAY	= 0x00000010,
	WMI_PEER_HE		= 0x00000400,
	WMI_PEER_APSD		= 0x00000800,
	WMI_PEER_HT		= 0x00001000,
	WMI_PEER_40MHZ		= 0x00002000,
	WMI_PEER_STBC		= 0x00008000,
	WMI_PEER_LDPC		= 0x00010000,
	WMI_PEER_DYN_MIMOPS	= 0x00020000,
	WMI_PEER_STATIC_MIMOPS	= 0x00040000,
	WMI_PEER_SPATIAL_MUX	= 0x00200000,
	WMI_PEER_TWT_REQ	= 0x00400000,
	WMI_PEER_TWT_RESP	= 0x00800000,
	WMI_PEER_VHT		= 0x02000000,
	WMI_PEER_80MHZ		= 0x04000000,
	WMI_PEER_PMF		= 0x08000000,
	WMI_PEER_IS_P2P_CAPABLE = 0x20000000,
	WMI_PEER_160MHZ         = 0x40000000,
	WMI_PEER_SAFEMODE_EN    = 0x80000000,
};

/** Enum list of TLV Tags for each parameter structure type. */
enum wmi_tlv_tag {
	WMI_TAG_LAST_RESERVED = 15,
	WMI_TAG_FIRST_ARRAY_ENUM,
	WMI_TAG_ARRAY_UINT32 = WMI_TAG_FIRST_ARRAY_ENUM,
	WMI_TAG_ARRAY_BYTE,
	WMI_TAG_ARRAY_STRUCT,
	WMI_TAG_ARRAY_FIXED_STRUCT,
	WMI_TAG_LAST_ARRAY_ENUM = 31,
	WMI_TAG_SERVICE_READY_EVENT,
	WMI_TAG_HAL_REG_CAPABILITIES,
	WMI_TAG_WLAN_HOST_MEM_REQ,
	WMI_TAG_READY_EVENT,
	WMI_TAG_SCAN_EVENT,
	WMI_TAG_PDEV_TPC_CONFIG_EVENT,
	WMI_TAG_CHAN_INFO_EVENT,
	WMI_TAG_COMB_PHYERR_RX_HDR,
	WMI_TAG_VDEV_START_RESPONSE_EVENT,
	WMI_TAG_VDEV_STOPPED_EVENT,
	WMI_TAG_VDEV_INSTALL_KEY_COMPLETE_EVENT,
	WMI_TAG_PEER_STA_KICKOUT_EVENT,
	WMI_TAG_MGMT_RX_HDR,
	WMI_TAG_TBTT_OFFSET_EVENT,
	WMI_TAG_TX_DELBA_COMPLETE_EVENT,
	WMI_TAG_TX_ADDBA_COMPLETE_EVENT,
	WMI_TAG_ROAM_EVENT,
	WMI_TAG_WOW_EVENT_INFO,
	WMI_TAG_WOW_EVENT_INFO_SECTION_BITMAP,
	WMI_TAG_RTT_EVENT_HEADER,
	WMI_TAG_RTT_ERROR_REPORT_EVENT,
	WMI_TAG_RTT_MEAS_EVENT,
	WMI_TAG_ECHO_EVENT,
	WMI_TAG_FTM_INTG_EVENT,
	WMI_TAG_VDEV_GET_KEEPALIVE_EVENT,
	WMI_TAG_GPIO_INPUT_EVENT,
	WMI_TAG_CSA_EVENT,
	WMI_TAG_GTK_OFFLOAD_STATUS_EVENT,
	WMI_TAG_IGTK_INFO,
	WMI_TAG_DCS_INTERFERENCE_EVENT,
	WMI_TAG_ATH_DCS_CW_INT,
	WMI_TAG_WLAN_DCS_CW_INT = /* ALIAS */
		WMI_TAG_ATH_DCS_CW_INT,
	WMI_TAG_ATH_DCS_WLAN_INT_STAT,
	WMI_TAG_WLAN_DCS_IM_TGT_STATS_T = /* ALIAS */
		WMI_TAG_ATH_DCS_WLAN_INT_STAT,
	WMI_TAG_WLAN_PROFILE_CTX_T,
	WMI_TAG_WLAN_PROFILE_T,
	WMI_TAG_PDEV_QVIT_EVENT,
	WMI_TAG_HOST_SWBA_EVENT,
	WMI_TAG_TIM_INFO,
	WMI_TAG_P2P_NOA_INFO,
	WMI_TAG_STATS_EVENT,
	WMI_TAG_AVOID_FREQ_RANGES_EVENT,
	WMI_TAG_AVOID_FREQ_RANGE_DESC,
	WMI_TAG_GTK_REKEY_FAIL_EVENT,
	WMI_TAG_INIT_CMD,
	WMI_TAG_RESOURCE_CONFIG,
	WMI_TAG_WLAN_HOST_MEMORY_CHUNK,
	WMI_TAG_START_SCAN_CMD,
	WMI_TAG_STOP_SCAN_CMD,
	WMI_TAG_SCAN_CHAN_LIST_CMD,
	WMI_TAG_CHANNEL,
	WMI_TAG_PDEV_SET_REGDOMAIN_CMD,
	WMI_TAG_PDEV_SET_PARAM_CMD,
	WMI_TAG_PDEV_SET_WMM_PARAMS_CMD,
	WMI_TAG_WMM_PARAMS,
	WMI_TAG_PDEV_SET_QUIET_CMD,
	WMI_TAG_VDEV_CREATE_CMD,
	WMI_TAG_VDEV_DELETE_CMD,
	WMI_TAG_VDEV_START_REQUEST_CMD,
	WMI_TAG_P2P_NOA_DESCRIPTOR,
	WMI_TAG_P2P_GO_SET_BEACON_IE,
	WMI_TAG_GTK_OFFLOAD_CMD,
	WMI_TAG_VDEV_UP_CMD,
	WMI_TAG_VDEV_STOP_CMD,
	WMI_TAG_VDEV_DOWN_CMD,
	WMI_TAG_VDEV_SET_PARAM_CMD,
	WMI_TAG_VDEV_INSTALL_KEY_CMD,
	WMI_TAG_PEER_CREATE_CMD,
	WMI_TAG_PEER_DELETE_CMD,
	WMI_TAG_PEER_FLUSH_TIDS_CMD,
	WMI_TAG_PEER_SET_PARAM_CMD,
	WMI_TAG_PEER_ASSOC_COMPLETE_CMD,
	WMI_TAG_VHT_RATE_SET,
	WMI_TAG_BCN_TMPL_CMD,
	WMI_TAG_PRB_TMPL_CMD,
	WMI_TAG_BCN_PRB_INFO,
	WMI_TAG_PEER_TID_ADDBA_CMD,
	WMI_TAG_PEER_TID_DELBA_CMD,
	WMI_TAG_STA_POWERSAVE_MODE_CMD,
	WMI_TAG_STA_POWERSAVE_PARAM_CMD,
	WMI_TAG_STA_DTIM_PS_METHOD_CMD,
	WMI_TAG_ROAM_SCAN_MODE,
	WMI_TAG_ROAM_SCAN_RSSI_THRESHOLD,
	WMI_TAG_ROAM_SCAN_PERIOD,
	WMI_TAG_ROAM_SCAN_RSSI_CHANGE_THRESHOLD,
	WMI_TAG_PDEV_SUSPEND_CMD,
	WMI_TAG_PDEV_RESUME_CMD,
	WMI_TAG_ADD_BCN_FILTER_CMD,
	WMI_TAG_RMV_BCN_FILTER_CMD,
	WMI_TAG_WOW_ENABLE_CMD,
	WMI_TAG_WOW_HOSTWAKEUP_FROM_SLEEP_CMD,
	WMI_TAG_STA_UAPSD_AUTO_TRIG_CMD,
	WMI_TAG_STA_UAPSD_AUTO_TRIG_PARAM,
	WMI_TAG_SET_ARP_NS_OFFLOAD_CMD,
	WMI_TAG_ARP_OFFLOAD_TUPLE,
	WMI_TAG_NS_OFFLOAD_TUPLE,
	WMI_TAG_FTM_INTG_CMD,
	WMI_TAG_STA_KEEPALIVE_CMD,
	WMI_TAG_STA_KEEPALIVE_ARP_RESPONSE,
	WMI_TAG_P2P_SET_VENDOR_IE_DATA_CMD,
	WMI_TAG_AP_PS_PEER_CMD,
	WMI_TAG_PEER_RATE_RETRY_SCHED_CMD,
	WMI_TAG_WLAN_PROFILE_TRIGGER_CMD,
	WMI_TAG_WLAN_PROFILE_SET_HIST_INTVL_CMD,
	WMI_TAG_WLAN_PROFILE_GET_PROF_DATA_CMD,
	WMI_TAG_WLAN_PROFILE_ENABLE_PROFILE_ID_CMD,
	WMI_TAG_WOW_DEL_PATTERN_CMD,
	WMI_TAG_WOW_ADD_DEL_EVT_CMD,
	WMI_TAG_RTT_MEASREQ_HEAD,
	WMI_TAG_RTT_MEASREQ_BODY,
	WMI_TAG_RTT_TSF_CMD,
	WMI_TAG_VDEV_SPECTRAL_CONFIGURE_CMD,
	WMI_TAG_VDEV_SPECTRAL_ENABLE_CMD,
	WMI_TAG_REQUEST_STATS_CMD,
	WMI_TAG_NLO_CONFIG_CMD,
	WMI_TAG_NLO_CONFIGURED_PARAMETERS,
	WMI_TAG_CSA_OFFLOAD_ENABLE_CMD,
	WMI_TAG_CSA_OFFLOAD_CHANSWITCH_CMD,
	WMI_TAG_CHATTER_SET_MODE_CMD,
	WMI_TAG_ECHO_CMD,
	WMI_TAG_VDEV_SET_KEEPALIVE_CMD,
	WMI_TAG_VDEV_GET_KEEPALIVE_CMD,
	WMI_TAG_FORCE_FW_HANG_CMD,
	WMI_TAG_GPIO_CONFIG_CMD,
	WMI_TAG_GPIO_OUTPUT_CMD,
	WMI_TAG_PEER_ADD_WDS_ENTRY_CMD,
	WMI_TAG_PEER_REMOVE_WDS_ENTRY_CMD,
	WMI_TAG_BCN_TX_HDR,
	WMI_TAG_BCN_SEND_FROM_HOST_CMD,
	WMI_TAG_MGMT_TX_HDR,
	WMI_TAG_ADDBA_CLEAR_RESP_CMD,
	WMI_TAG_ADDBA_SEND_CMD,
	WMI_TAG_DELBA_SEND_CMD,
	WMI_TAG_ADDBA_SETRESPONSE_CMD,
	WMI_TAG_SEND_SINGLEAMSDU_CMD,
	WMI_TAG_PDEV_PKTLOG_ENABLE_CMD,
	WMI_TAG_PDEV_PKTLOG_DISABLE_CMD,
	WMI_TAG_PDEV_SET_HT_IE_CMD,
	WMI_TAG_PDEV_SET_VHT_IE_CMD,
	WMI_TAG_PDEV_SET_DSCP_TID_MAP_CMD,
	WMI_TAG_PDEV_GREEN_AP_PS_ENABLE_CMD,
	WMI_TAG_PDEV_GET_TPC_CONFIG_CMD,
	WMI_TAG_PDEV_SET_BASE_MACADDR_CMD,
	WMI_TAG_PEER_MCAST_GROUP_CMD,
	WMI_TAG_ROAM_AP_PROFILE,
	WMI_TAG_AP_PROFILE,
	WMI_TAG_SCAN_SCH_PRIORITY_TABLE_CMD,
	WMI_TAG_PDEV_DFS_ENABLE_CMD,
	WMI_TAG_PDEV_DFS_DISABLE_CMD,
	WMI_TAG_WOW_ADD_PATTERN_CMD,
	WMI_TAG_WOW_BITMAP_PATTERN_T,
	WMI_TAG_WOW_IPV4_SYNC_PATTERN_T,
	WMI_TAG_WOW_IPV6_SYNC_PATTERN_T,
	WMI_TAG_WOW_MAGIC_PATTERN_CMD,
	WMI_TAG_SCAN_UPDATE_REQUEST_CMD,
	WMI_TAG_CHATTER_PKT_COALESCING_FILTER,
	WMI_TAG_CHATTER_COALESCING_ADD_FILTER_CMD,
	WMI_TAG_CHATTER_COALESCING_DELETE_FILTER_CMD,
	WMI_TAG_CHATTER_COALESCING_QUERY_CMD,
	WMI_TAG_TXBF_CMD,
	WMI_TAG_DEBUG_LOG_CONFIG_CMD,
	WMI_TAG_NLO_EVENT,
	WMI_TAG_CHATTER_QUERY_REPLY_EVENT,
	WMI_TAG_UPLOAD_H_HDR,
	WMI_TAG_CAPTURE_H_EVENT_HDR,
	WMI_TAG_VDEV_WNM_SLEEPMODE_CMD,
	WMI_TAG_VDEV_IPSEC_NATKEEPALIVE_FILTER_CMD,
	WMI_TAG_VDEV_WMM_ADDTS_CMD,
	WMI_TAG_VDEV_WMM_DELTS_CMD,
	WMI_TAG_VDEV_SET_WMM_PARAMS_CMD,
	WMI_TAG_TDLS_SET_STATE_CMD,
	WMI_TAG_TDLS_PEER_UPDATE_CMD,
	WMI_TAG_TDLS_PEER_EVENT,
	WMI_TAG_TDLS_PEER_CAPABILITIES,
	WMI_TAG_VDEV_MCC_SET_TBTT_MODE_CMD,
	WMI_TAG_ROAM_CHAN_LIST,
	WMI_TAG_VDEV_MCC_BCN_INTVL_CHANGE_EVENT,
	WMI_TAG_RESMGR_ADAPTIVE_OCS_ENABLE_DISABLE_CMD,
	WMI_TAG_RESMGR_SET_CHAN_TIME_QUOTA_CMD,
	WMI_TAG_RESMGR_SET_CHAN_LATENCY_CMD,
	WMI_TAG_BA_REQ_SSN_CMD,
	WMI_TAG_BA_RSP_SSN_EVENT,
	WMI_TAG_STA_SMPS_FORCE_MODE_CMD,
	WMI_TAG_SET_MCASTBCAST_FILTER_CMD,
	WMI_TAG_P2P_SET_OPPPS_CMD,
	WMI_TAG_P2P_SET_NOA_CMD,
	WMI_TAG_BA_REQ_SSN_CMD_SUB_STRUCT_PARAM,
	WMI_TAG_BA_REQ_SSN_EVENT_SUB_STRUCT_PARAM,
	WMI_TAG_STA_SMPS_PARAM_CMD,
	WMI_TAG_VDEV_SET_GTX_PARAMS_CMD,
	WMI_TAG_MCC_SCHED_TRAFFIC_STATS_CMD,
	WMI_TAG_MCC_SCHED_STA_TRAFFIC_STATS,
	WMI_TAG_OFFLOAD_BCN_TX_STATUS_EVENT,
	WMI_TAG_P2P_NOA_EVENT,
	WMI_TAG_HB_SET_ENABLE_CMD,
	WMI_TAG_HB_SET_TCP_PARAMS_CMD,
	WMI_TAG_HB_SET_TCP_PKT_FILTER_CMD,
	WMI_TAG_HB_SET_UDP_PARAMS_CMD,
	WMI_TAG_HB_SET_UDP_PKT_FILTER_CMD,
	WMI_TAG_HB_IND_EVENT,
	WMI_TAG_TX_PAUSE_EVENT,
	WMI_TAG_RFKILL_EVENT,
	WMI_TAG_DFS_RADAR_EVENT,
	WMI_TAG_DFS_PHYERR_FILTER_ENA_CMD,
	WMI_TAG_DFS_PHYERR_FILTER_DIS_CMD,
	WMI_TAG_BATCH_SCAN_RESULT_SCAN_LIST,
	WMI_TAG_BATCH_SCAN_RESULT_NETWORK_INFO,
	WMI_TAG_BATCH_SCAN_ENABLE_CMD,
	WMI_TAG_BATCH_SCAN_DISABLE_CMD,
	WMI_TAG_BATCH_SCAN_TRIGGER_RESULT_CMD,
	WMI_TAG_BATCH_SCAN_ENABLED_EVENT,
	WMI_TAG_BATCH_SCAN_RESULT_EVENT,
	WMI_TAG_VDEV_PLMREQ_START_CMD,
	WMI_TAG_VDEV_PLMREQ_STOP_CMD,
	WMI_TAG_THERMAL_MGMT_CMD,
	WMI_TAG_THERMAL_MGMT_EVENT,
	WMI_TAG_PEER_INFO_REQ_CMD,
	WMI_TAG_PEER_INFO_EVENT,
	WMI_TAG_PEER_INFO,
	WMI_TAG_PEER_TX_FAIL_CNT_THR_EVENT,
	WMI_TAG_RMC_SET_MODE_CMD,
	WMI_TAG_RMC_SET_ACTION_PERIOD_CMD,
	WMI_TAG_RMC_CONFIG_CMD,
	WMI_TAG_MHF_OFFLOAD_SET_MODE_CMD,
	WMI_TAG_MHF_OFFLOAD_PLUMB_ROUTING_TABLE_CMD,
	WMI_TAG_ADD_PROACTIVE_ARP_RSP_PATTERN_CMD,
	WMI_TAG_DEL_PROACTIVE_ARP_RSP_PATTERN_CMD,
	WMI_TAG_NAN_CMD_PARAM,
	WMI_TAG_NAN_EVENT_HDR,
	WMI_TAG_PDEV_L1SS_TRACK_EVENT,
	WMI_TAG_DIAG_DATA_CONTAINER_EVENT,
	WMI_TAG_MODEM_POWER_STATE_CMD_PARAM,
	WMI_TAG_PEER_GET_ESTIMATED_LINKSPEED_CMD,
	WMI_TAG_PEER_ESTIMATED_LINKSPEED_EVENT,
	WMI_TAG_AGGR_STATE_TRIG_EVENT,
	WMI_TAG_MHF_OFFLOAD_ROUTING_TABLE_ENTRY,
	WMI_TAG_ROAM_SCAN_CMD,
	WMI_TAG_REQ_STATS_EXT_CMD,
	WMI_TAG_STATS_EXT_EVENT,
	WMI_TAG_OBSS_SCAN_ENABLE_CMD,
	WMI_TAG_OBSS_SCAN_DISABLE_CMD,
	WMI_TAG_OFFLOAD_PRB_RSP_TX_STATUS_EVENT,
	WMI_TAG_PDEV_SET_LED_CONFIG_CMD,
	WMI_TAG_HOST_AUTO_SHUTDOWN_CFG_CMD,
	WMI_TAG_HOST_AUTO_SHUTDOWN_EVENT,
	WMI_TAG_UPDATE_WHAL_MIB_STATS_EVENT,
	WMI_TAG_CHAN_AVOID_UPDATE_CMD_PARAM,
	WMI_TAG_WOW_IOAC_PKT_PATTERN_T,
	WMI_TAG_WOW_IOAC_TMR_PATTERN_T,
	WMI_TAG_WOW_IOAC_ADD_KEEPALIVE_CMD,
	WMI_TAG_WOW_IOAC_DEL_KEEPALIVE_CMD,
	WMI_TAG_WOW_IOAC_KEEPALIVE_T,
	WMI_TAG_WOW_IOAC_ADD_PATTERN_CMD,
	WMI_TAG_WOW_IOAC_DEL_PATTERN_CMD,
	WMI_TAG_START_LINK_STATS_CMD,
	WMI_TAG_CLEAR_LINK_STATS_CMD,
	WMI_TAG_REQUEST_LINK_STATS_CMD,
	WMI_TAG_IFACE_LINK_STATS_EVENT,
	WMI_TAG_RADIO_LINK_STATS_EVENT,
	WMI_TAG_PEER_STATS_EVENT,
	WMI_TAG_CHANNEL_STATS,
	WMI_TAG_RADIO_LINK_STATS,
	WMI_TAG_RATE_STATS,
	WMI_TAG_PEER_LINK_STATS,
	WMI_TAG_WMM_AC_STATS,
	WMI_TAG_IFACE_LINK_STATS,
	WMI_TAG_LPI_MGMT_SNOOPING_CONFIG_CMD,
	WMI_TAG_LPI_START_SCAN_CMD,
	WMI_TAG_LPI_STOP_SCAN_CMD,
	WMI_TAG_LPI_RESULT_EVENT,
	WMI_TAG_PEER_STATE_EVENT,
	WMI_TAG_EXTSCAN_BUCKET_CMD,
	WMI_TAG_EXTSCAN_BUCKET_CHANNEL_EVENT,
	WMI_TAG_EXTSCAN_START_CMD,
	WMI_TAG_EXTSCAN_STOP_CMD,
	WMI_TAG_EXTSCAN_CONFIGURE_WLAN_CHANGE_MONITOR_CMD,
	WMI_TAG_EXTSCAN_WLAN_CHANGE_BSSID_PARAM_CMD,
	WMI_TAG_EXTSCAN_CONFIGURE_HOTLIST_MONITOR_CMD,
	WMI_TAG_EXTSCAN_GET_CACHED_RESULTS_CMD,
	WMI_TAG_EXTSCAN_GET_WLAN_CHANGE_RESULTS_CMD,
	WMI_TAG_EXTSCAN_SET_CAPABILITIES_CMD,
	WMI_TAG_EXTSCAN_GET_CAPABILITIES_CMD,
	WMI_TAG_EXTSCAN_OPERATION_EVENT,
	WMI_TAG_EXTSCAN_START_STOP_EVENT,
	WMI_TAG_EXTSCAN_TABLE_USAGE_EVENT,
	WMI_TAG_EXTSCAN_WLAN_DESCRIPTOR_EVENT,
	WMI_TAG_EXTSCAN_RSSI_INFO_EVENT,
	WMI_TAG_EXTSCAN_CACHED_RESULTS_EVENT,
	WMI_TAG_EXTSCAN_WLAN_CHANGE_RESULTS_EVENT,
	WMI_TAG_EXTSCAN_WLAN_CHANGE_RESULT_BSSID_EVENT,
	WMI_TAG_EXTSCAN_HOTLIST_MATCH_EVENT,
	WMI_TAG_EXTSCAN_CAPABILITIES_EVENT,
	WMI_TAG_EXTSCAN_CACHE_CAPABILITIES_EVENT,
	WMI_TAG_EXTSCAN_WLAN_CHANGE_MONITOR_CAPABILITIES_EVENT,
	WMI_TAG_EXTSCAN_HOTLIST_MONITOR_CAPABILITIES_EVENT,
	WMI_TAG_D0_WOW_ENABLE_DISABLE_CMD,
	WMI_TAG_D0_WOW_DISABLE_ACK_EVENT,
	WMI_TAG_UNIT_TEST_CMD,
	WMI_TAG_ROAM_OFFLOAD_TLV_PARAM,
	WMI_TAG_ROAM_11I_OFFLOAD_TLV_PARAM,
	WMI_TAG_ROAM_11R_OFFLOAD_TLV_PARAM,
	WMI_TAG_ROAM_ESE_OFFLOAD_TLV_PARAM,
	WMI_TAG_ROAM_SYNCH_EVENT,
	WMI_TAG_ROAM_SYNCH_COMPLETE,
	WMI_TAG_EXTWOW_ENABLE_CMD,
	WMI_TAG_EXTWOW_SET_APP_TYPE1_PARAMS_CMD,
	WMI_TAG_EXTWOW_SET_APP_TYPE2_PARAMS_CMD,
	WMI_TAG_LPI_STATUS_EVENT,
	WMI_TAG_LPI_HANDOFF_EVENT,
	WMI_TAG_VDEV_RATE_STATS_EVENT,
	WMI_TAG_VDEV_RATE_HT_INFO,
	WMI_TAG_RIC_REQUEST,
	WMI_TAG_PDEV_GET_TEMPERATURE_CMD,
	WMI_TAG_PDEV_TEMPERATURE_EVENT,
	WMI_TAG_SET_DHCP_SERVER_OFFLOAD_CMD,
	WMI_TAG_TPC_CHAINMASK_CONFIG_CMD,
	WMI_TAG_RIC_TSPEC,
	WMI_TAG_TPC_CHAINMASK_CONFIG,
	WMI_TAG_IPA_OFFLOAD_ENABLE_DISABLE_CMD,
	WMI_TAG_SCAN_PROB_REQ_OUI_CMD,
	WMI_TAG_KEY_MATERIAL,
	WMI_TAG_TDLS_SET_OFFCHAN_MODE_CMD,
	WMI_TAG_SET_LED_FLASHING_CMD,
	WMI_TAG_MDNS_OFFLOAD_CMD,
	WMI_TAG_MDNS_SET_FQDN_CMD,
	WMI_TAG_MDNS_SET_RESP_CMD,
	WMI_TAG_MDNS_GET_STATS_CMD,
	WMI_TAG_MDNS_STATS_EVENT,
	WMI_TAG_ROAM_INVOKE_CMD,
	WMI_TAG_PDEV_RESUME_EVENT,
	WMI_TAG_PDEV_SET_ANTENNA_DIVERSITY_CMD,
	WMI_TAG_SAP_OFL_ENABLE_CMD,
	WMI_TAG_SAP_OFL_ADD_STA_EVENT,
	WMI_TAG_SAP_OFL_DEL_STA_EVENT,
	WMI_TAG_APFIND_CMD_PARAM,
	WMI_TAG_APFIND_EVENT_HDR,
	WMI_TAG_OCB_SET_SCHED_CMD,
	WMI_TAG_OCB_SET_SCHED_EVENT,
	WMI_TAG_OCB_SET_CONFIG_CMD,
	WMI_TAG_OCB_SET_CONFIG_RESP_EVENT,
	WMI_TAG_OCB_SET_UTC_TIME_CMD,
	WMI_TAG_OCB_START_TIMING_ADVERT_CMD,
	WMI_TAG_OCB_STOP_TIMING_ADVERT_CMD,
	WMI_TAG_OCB_GET_TSF_TIMER_CMD,
	WMI_TAG_OCB_GET_TSF_TIMER_RESP_EVENT,
	WMI_TAG_DCC_GET_STATS_CMD,
	WMI_TAG_DCC_CHANNEL_STATS_REQUEST,
	WMI_TAG_DCC_GET_STATS_RESP_EVENT,
	WMI_TAG_DCC_CLEAR_STATS_CMD,
	WMI_TAG_DCC_UPDATE_NDL_CMD,
	WMI_TAG_DCC_UPDATE_NDL_RESP_EVENT,
	WMI_TAG_DCC_STATS_EVENT,
	WMI_TAG_OCB_CHANNEL,
	WMI_TAG_OCB_SCHEDULE_ELEMENT,
	WMI_TAG_DCC_NDL_STATS_PER_CHANNEL,
	WMI_TAG_DCC_NDL_CHAN,
	WMI_TAG_QOS_PARAMETER,
	WMI_TAG_DCC_NDL_ACTIVE_STATE_CONFIG,
	WMI_TAG_ROAM_SCAN_EXTENDED_THRESHOLD_PARAM,
	WMI_TAG_ROAM_FILTER,
	WMI_TAG_PASSPOINT_CONFIG_CMD,
	WMI_TAG_PASSPOINT_EVENT_HDR,
	WMI_TAG_EXTSCAN_CONFIGURE_HOTLIST_SSID_MONITOR_CMD,
	WMI_TAG_EXTSCAN_HOTLIST_SSID_MATCH_EVENT,
	WMI_TAG_VDEV_TSF_TSTAMP_ACTION_CMD,
	WMI_TAG_VDEV_TSF_REPORT_EVENT,
	WMI_TAG_GET_FW_MEM_DUMP,
	WMI_TAG_UPDATE_FW_MEM_DUMP,
	WMI_TAG_FW_MEM_DUMP_PARAMS,
	WMI_TAG_DEBUG_MESG_FLUSH,
	WMI_TAG_DEBUG_MESG_FLUSH_COMPLETE,
	WMI_TAG_PEER_SET_RATE_REPORT_CONDITION,
	WMI_TAG_ROAM_SUBNET_CHANGE_CONFIG,
	WMI_TAG_VDEV_SET_IE_CMD,
	WMI_TAG_RSSI_BREACH_MONITOR_CONFIG,
	WMI_TAG_RSSI_BREACH_EVENT,
	WMI_TAG_WOW_EVENT_INITIAL_WAKEUP,
	WMI_TAG_SOC_SET_PCL_CMD,
	WMI_TAG_SOC_SET_HW_MODE_CMD,
	WMI_TAG_SOC_SET_HW_MODE_RESPONSE_EVENT,
	WMI_TAG_SOC_HW_MODE_TRANSITION_EVENT,
	WMI_TAG_VDEV_TXRX_STREAMS,
	WMI_TAG_SOC_SET_HW_MODE_RESPONSE_VDEV_MAC_ENTRY,
	WMI_TAG_SOC_SET_DUAL_MAC_CONFIG_CMD,
	WMI_TAG_SOC_SET_DUAL_MAC_CONFIG_RESPONSE_EVENT,
	WMI_TAG_WOW_IOAC_SOCK_PATTERN_T,
	WMI_TAG_WOW_ENABLE_ICMPV6_NA_FLT_CMD,
	WMI_TAG_DIAG_EVENT_LOG_CONFIG,
	WMI_TAG_DIAG_EVENT_LOG_SUPPORTED_EVENT_FIXED_PARAMS,
	WMI_TAG_PACKET_FILTER_CONFIG,
	WMI_TAG_PACKET_FILTER_ENABLE,
	WMI_TAG_SAP_SET_BLACKLIST_PARAM_CMD,
	WMI_TAG_MGMT_TX_SEND_CMD,
	WMI_TAG_MGMT_TX_COMPL_EVENT,
	WMI_TAG_SOC_SET_ANTENNA_MODE_CMD,
	WMI_TAG_WOW_UDP_SVC_OFLD_CMD,
	WMI_TAG_LRO_INFO_CMD,
	WMI_TAG_ROAM_EARLYSTOP_RSSI_THRES_PARAM,
	WMI_TAG_SERVICE_READY_EXT_EVENT,
	WMI_TAG_MAWC_SENSOR_REPORT_IND_CMD,
	WMI_TAG_MAWC_ENABLE_SENSOR_EVENT,
	WMI_TAG_ROAM_CONFIGURE_MAWC_CMD,
	WMI_TAG_NLO_CONFIGURE_MAWC_CMD,
	WMI_TAG_EXTSCAN_CONFIGURE_MAWC_CMD,
	WMI_TAG_PEER_ASSOC_CONF_EVENT,
	WMI_TAG_WOW_HOSTWAKEUP_GPIO_PIN_PATTERN_CONFIG_CMD,
	WMI_TAG_AP_PS_EGAP_PARAM_CMD,
	WMI_TAG_AP_PS_EGAP_INFO_EVENT,
	WMI_TAG_PMF_OFFLOAD_SET_SA_QUERY_CMD,
	WMI_TAG_TRANSFER_DATA_TO_FLASH_CMD,
	WMI_TAG_TRANSFER_DATA_TO_FLASH_COMPLETE_EVENT,
	WMI_TAG_SCPC_EVENT,
	WMI_TAG_AP_PS_EGAP_INFO_CHAINMASK_LIST,
	WMI_TAG_STA_SMPS_FORCE_MODE_COMPLETE_EVENT,
	WMI_TAG_BPF_GET_CAPABILITY_CMD,
	WMI_TAG_BPF_CAPABILITY_INFO_EVT,
	WMI_TAG_BPF_GET_VDEV_STATS_CMD,
	WMI_TAG_BPF_VDEV_STATS_INFO_EVT,
	WMI_TAG_BPF_SET_VDEV_INSTRUCTIONS_CMD,
	WMI_TAG_BPF_DEL_VDEV_INSTRUCTIONS_CMD,
	WMI_TAG_VDEV_DELETE_RESP_EVENT,
	WMI_TAG_PEER_DELETE_RESP_EVENT,
	WMI_TAG_ROAM_DENSE_THRES_PARAM,
	WMI_TAG_ENLO_CANDIDATE_SCORE_PARAM,
	WMI_TAG_PEER_UPDATE_WDS_ENTRY_CMD,
	WMI_TAG_VDEV_CONFIG_RATEMASK,
	WMI_TAG_PDEV_FIPS_CMD,
	WMI_TAG_PDEV_SMART_ANT_ENABLE_CMD,
	WMI_TAG_PDEV_SMART_ANT_SET_RX_ANTENNA_CMD,
	WMI_TAG_PEER_SMART_ANT_SET_TX_ANTENNA_CMD,
	WMI_TAG_PEER_SMART_ANT_SET_TRAIN_ANTENNA_CMD,
	WMI_TAG_PEER_SMART_ANT_SET_NODE_CONFIG_OPS_CMD,
	WMI_TAG_PDEV_SET_ANT_SWITCH_TBL_CMD,
	WMI_TAG_PDEV_SET_CTL_TABLE_CMD,
	WMI_TAG_PDEV_SET_MIMOGAIN_TABLE_CMD,
	WMI_TAG_FWTEST_SET_PARAM_CMD,
	WMI_TAG_PEER_ATF_REQUEST,
	WMI_TAG_VDEV_ATF_REQUEST,
	WMI_TAG_PDEV_GET_ANI_CCK_CONFIG_CMD,
	WMI_TAG_PDEV_GET_ANI_OFDM_CONFIG_CMD,
	WMI_TAG_INST_RSSI_STATS_RESP,
	WMI_TAG_MED_UTIL_REPORT_EVENT,
	WMI_TAG_PEER_STA_PS_STATECHANGE_EVENT,
	WMI_TAG_WDS_ADDR_EVENT,
	WMI_TAG_PEER_RATECODE_LIST_EVENT,
	WMI_TAG_PDEV_NFCAL_POWER_ALL_CHANNELS_EVENT,
	WMI_TAG_PDEV_TPC_EVENT,
	WMI_TAG_ANI_OFDM_EVENT,
	WMI_TAG_ANI_CCK_EVENT,
	WMI_TAG_PDEV_CHANNEL_HOPPING_EVENT,
	WMI_TAG_PDEV_FIPS_EVENT,
	WMI_TAG_ATF_PEER_INFO,
	WMI_TAG_PDEV_GET_TPC_CMD,
	WMI_TAG_VDEV_FILTER_NRP_CONFIG_CMD,
	WMI_TAG_QBOOST_CFG_CMD,
	WMI_TAG_PDEV_SMART_ANT_GPIO_HANDLE,
	WMI_TAG_PEER_SMART_ANT_SET_TX_ANTENNA_SERIES,
	WMI_TAG_PEER_SMART_ANT_SET_TRAIN_ANTENNA_PARAM,
	WMI_TAG_PDEV_SET_ANT_CTRL_CHAIN,
	WMI_TAG_PEER_CCK_OFDM_RATE_INFO,
	WMI_TAG_PEER_MCS_RATE_INFO,
	WMI_TAG_PDEV_NFCAL_POWER_ALL_CHANNELS_NFDBR,
	WMI_TAG_PDEV_NFCAL_POWER_ALL_CHANNELS_NFDBM,
	WMI_TAG_PDEV_NFCAL_POWER_ALL_CHANNELS_FREQNUM,
	WMI_TAG_MU_REPORT_TOTAL_MU,
	WMI_TAG_VDEV_SET_DSCP_TID_MAP_CMD,
	WMI_TAG_ROAM_SET_MBO,
	WMI_TAG_MIB_STATS_ENABLE_CMD,
	WMI_TAG_NAN_DISC_IFACE_CREATED_EVENT,
	WMI_TAG_NAN_DISC_IFACE_DELETED_EVENT,
	WMI_TAG_NAN_STARTED_CLUSTER_EVENT,
	WMI_TAG_NAN_JOINED_CLUSTER_EVENT,
	WMI_TAG_NDI_GET_CAP_REQ,
	WMI_TAG_NDP_INITIATOR_REQ,
	WMI_TAG_NDP_RESPONDER_REQ,
	WMI_TAG_NDP_END_REQ,
	WMI_TAG_NDI_CAP_RSP_EVENT,
	WMI_TAG_NDP_INITIATOR_RSP_EVENT,
	WMI_TAG_NDP_RESPONDER_RSP_EVENT,
	WMI_TAG_NDP_END_RSP_EVENT,
	WMI_TAG_NDP_INDICATION_EVENT,
	WMI_TAG_NDP_CONFIRM_EVENT,
	WMI_TAG_NDP_END_INDICATION_EVENT,
	WMI_TAG_VDEV_SET_QUIET_CMD,
	WMI_TAG_PDEV_SET_PCL_CMD,
	WMI_TAG_PDEV_SET_HW_MODE_CMD,
	WMI_TAG_PDEV_SET_MAC_CONFIG_CMD,
	WMI_TAG_PDEV_SET_ANTENNA_MODE_CMD,
	WMI_TAG_PDEV_SET_HW_MODE_RESPONSE_EVENT,
	WMI_TAG_PDEV_HW_MODE_TRANSITION_EVENT,
	WMI_TAG_PDEV_SET_HW_MODE_RESPONSE_VDEV_MAC_ENTRY,
	WMI_TAG_PDEV_SET_MAC_CONFIG_RESPONSE_EVENT,
	WMI_TAG_COEX_CONFIG_CMD,
	WMI_TAG_CONFIG_ENHANCED_MCAST_FILTER,
	WMI_TAG_CHAN_AVOID_RPT_ALLOW_CMD,
	WMI_TAG_SET_PERIODIC_CHANNEL_STATS_CONFIG,
	WMI_TAG_VDEV_SET_CUSTOM_AGGR_SIZE_CMD,
	WMI_TAG_PDEV_WAL_POWER_DEBUG_CMD,
	WMI_TAG_MAC_PHY_CAPABILITIES,
	WMI_TAG_HW_MODE_CAPABILITIES,
	WMI_TAG_SOC_MAC_PHY_HW_MODE_CAPS,
	WMI_TAG_HAL_REG_CAPABILITIES_EXT,
	WMI_TAG_SOC_HAL_REG_CAPABILITIES,
	WMI_TAG_VDEV_WISA_CMD,
	WMI_TAG_TX_POWER_LEVEL_STATS_EVT,
	WMI_TAG_SCAN_ADAPTIVE_DWELL_PARAMETERS_TLV,
	WMI_TAG_SCAN_ADAPTIVE_DWELL_CONFIG,
	WMI_TAG_WOW_SET_ACTION_WAKE_UP_CMD,
	WMI_TAG_NDP_END_RSP_PER_NDI,
	WMI_TAG_PEER_BWF_REQUEST,
	WMI_TAG_BWF_PEER_INFO,
	WMI_TAG_DBGLOG_TIME_STAMP_SYNC_CMD,
	WMI_TAG_RMC_SET_LEADER_CMD,
	WMI_TAG_RMC_MANUAL_LEADER_EVENT,
	WMI_TAG_PER_CHAIN_RSSI_STATS,
	WMI_TAG_RSSI_STATS,
	WMI_TAG_P2P_LO_START_CMD,
	WMI_TAG_P2P_LO_STOP_CMD,
	WMI_TAG_P2P_LO_STOPPED_EVENT,
	WMI_TAG_REORDER_QUEUE_SETUP_CMD,
	WMI_TAG_REORDER_QUEUE_REMOVE_CMD,
	WMI_TAG_SET_MULTIPLE_MCAST_FILTER_CMD,
	WMI_TAG_MGMT_TX_COMPL_BUNDLE_EVENT,
	WMI_TAG_READ_DATA_FROM_FLASH_CMD,
	WMI_TAG_READ_DATA_FROM_FLASH_EVENT,
	WMI_TAG_PDEV_SET_REORDER_TIMEOUT_VAL_CMD,
	WMI_TAG_PEER_SET_RX_BLOCKSIZE_CMD,
	WMI_TAG_PDEV_SET_WAKEUP_CONFIG_CMDID,
	WMI_TAG_TLV_BUF_LEN_PARAM,
	WMI_TAG_SERVICE_AVAILABLE_EVENT,
	WMI_TAG_PEER_ANTDIV_INFO_REQ_CMD,
	WMI_TAG_PEER_ANTDIV_INFO_EVENT,
	WMI_TAG_PEER_ANTDIV_INFO,
	WMI_TAG_PDEV_GET_ANTDIV_STATUS_CMD,
	WMI_TAG_PDEV_ANTDIV_STATUS_EVENT,
	WMI_TAG_MNT_FILTER_CMD,
	WMI_TAG_GET_CHIP_POWER_STATS_CMD,
	WMI_TAG_PDEV_CHIP_POWER_STATS_EVENT,
	WMI_TAG_COEX_GET_ANTENNA_ISOLATION_CMD,
	WMI_TAG_COEX_REPORT_ISOLATION_EVENT,
	WMI_TAG_CHAN_CCA_STATS,
	WMI_TAG_PEER_SIGNAL_STATS,
	WMI_TAG_TX_STATS,
	WMI_TAG_PEER_AC_TX_STATS,
	WMI_TAG_RX_STATS,
	WMI_TAG_PEER_AC_RX_STATS,
	WMI_TAG_REPORT_STATS_EVENT,
	WMI_TAG_CHAN_CCA_STATS_THRESH,
	WMI_TAG_PEER_SIGNAL_STATS_THRESH,
	WMI_TAG_TX_STATS_THRESH,
	WMI_TAG_RX_STATS_THRESH,
	WMI_TAG_PDEV_SET_STATS_THRESHOLD_CMD,
	WMI_TAG_REQUEST_WLAN_STATS_CMD,
	WMI_TAG_RX_AGGR_FAILURE_EVENT,
	WMI_TAG_RX_AGGR_FAILURE_INFO,
	WMI_TAG_VDEV_ENCRYPT_DECRYPT_DATA_REQ_CMD,
	WMI_TAG_VDEV_ENCRYPT_DECRYPT_DATA_RESP_EVENT,
	WMI_TAG_PDEV_BAND_TO_MAC,
	WMI_TAG_TBTT_OFFSET_INFO,
	WMI_TAG_TBTT_OFFSET_EXT_EVENT,
	WMI_TAG_SAR_LIMITS_CMD,
	WMI_TAG_SAR_LIMIT_CMD_ROW,
	WMI_TAG_PDEV_DFS_PHYERR_OFFLOAD_ENABLE_CMD,
	WMI_TAG_PDEV_DFS_PHYERR_OFFLOAD_DISABLE_CMD,
	WMI_TAG_VDEV_ADFS_CH_CFG_CMD,
	WMI_TAG_VDEV_ADFS_OCAC_ABORT_CMD,
	WMI_TAG_PDEV_DFS_RADAR_DETECTION_EVENT,
	WMI_TAG_VDEV_ADFS_OCAC_COMPLETE_EVENT,
	WMI_TAG_VDEV_DFS_CAC_COMPLETE_EVENT,
	WMI_TAG_VENDOR_OUI,
	WMI_TAG_REQUEST_RCPI_CMD,
	WMI_TAG_UPDATE_RCPI_EVENT,
	WMI_TAG_REQUEST_PEER_STATS_INFO_CMD,
	WMI_TAG_PEER_STATS_INFO,
	WMI_TAG_PEER_STATS_INFO_EVENT,
	WMI_TAG_PKGID_EVENT,
	WMI_TAG_CONNECTED_NLO_RSSI_PARAMS,
	WMI_TAG_SET_CURRENT_COUNTRY_CMD,
	WMI_TAG_REGULATORY_RULE_STRUCT,
	WMI_TAG_REG_CHAN_LIST_CC_EVENT,
	WMI_TAG_11D_SCAN_START_CMD,
	WMI_TAG_11D_SCAN_STOP_CMD,
	WMI_TAG_11D_NEW_COUNTRY_EVENT,
	WMI_TAG_REQUEST_RADIO_CHAN_STATS_CMD,
	WMI_TAG_RADIO_CHAN_STATS,
	WMI_TAG_RADIO_CHAN_STATS_EVENT,
	WMI_TAG_ROAM_PER_CONFIG,
	WMI_TAG_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_CMD,
	WMI_TAG_VDEV_ADD_MAC_ADDR_TO_RX_FILTER_STATUS_EVENT,
	WMI_TAG_BPF_SET_VDEV_ACTIVE_MODE_CMD,
	WMI_TAG_HW_DATA_FILTER_CMD,
	WMI_TAG_CONNECTED_NLO_BSS_BAND_RSSI_PREF,
	WMI_TAG_PEER_OPER_MODE_CHANGE_EVENT,
	WMI_TAG_CHIP_POWER_SAVE_FAILURE_DETECTED,
	WMI_TAG_PDEV_MULTIPLE_VDEV_RESTART_REQUEST_CMD,
	WMI_TAG_PDEV_CSA_SWITCH_COUNT_STATUS_EVENT,
	WMI_TAG_PDEV_UPDATE_PKT_ROUTING_CMD,
	WMI_TAG_PDEV_CHECK_CAL_VERSION_CMD,
	WMI_TAG_PDEV_CHECK_CAL_VERSION_EVENT,
	WMI_TAG_PDEV_SET_DIVERSITY_GAIN_CMD,
	WMI_TAG_MAC_PHY_CHAINMASK_COMBO,
	WMI_TAG_MAC_PHY_CHAINMASK_CAPABILITY,
	WMI_TAG_VDEV_SET_ARP_STATS_CMD,
	WMI_TAG_VDEV_GET_ARP_STATS_CMD,
	WMI_TAG_VDEV_GET_ARP_STATS_EVENT,
	WMI_TAG_IFACE_OFFLOAD_STATS,
	WMI_TAG_REQUEST_STATS_CMD_SUB_STRUCT_PARAM,
	WMI_TAG_RSSI_CTL_EXT,
	WMI_TAG_SINGLE_PHYERR_EXT_RX_HDR,
	WMI_TAG_COEX_BT_ACTIVITY_EVENT,
	WMI_TAG_VDEV_GET_TX_POWER_CMD,
	WMI_TAG_VDEV_TX_POWER_EVENT,
	WMI_TAG_OFFCHAN_DATA_TX_COMPL_EVENT,
	WMI_TAG_OFFCHAN_DATA_TX_SEND_CMD,
	WMI_TAG_TX_SEND_PARAMS,
	WMI_TAG_HE_RATE_SET,
	WMI_TAG_CONGESTION_STATS,
	WMI_TAG_SET_INIT_COUNTRY_CMD,
	WMI_TAG_SCAN_DBS_DUTY_CYCLE,
	WMI_TAG_SCAN_DBS_DUTY_CYCLE_PARAM_TLV,
	WMI_TAG_PDEV_DIV_GET_RSSI_ANTID,
	WMI_TAG_THERM_THROT_CONFIG_REQUEST,
	WMI_TAG_THERM_THROT_LEVEL_CONFIG_INFO,
	WMI_TAG_THERM_THROT_STATS_EVENT,
	WMI_TAG_THERM_THROT_LEVEL_STATS_INFO,
	WMI_TAG_PDEV_DIV_RSSI_ANTID_EVENT,
	WMI_TAG_OEM_DMA_RING_CAPABILITIES,
	WMI_TAG_OEM_DMA_RING_CFG_REQ,
	WMI_TAG_OEM_DMA_RING_CFG_RSP,
	WMI_TAG_OEM_INDIRECT_DATA,
	WMI_TAG_OEM_DMA_BUF_RELEASE,
	WMI_TAG_OEM_DMA_BUF_RELEASE_ENTRY,
	WMI_TAG_PDEV_BSS_CHAN_INFO_REQUEST,
	WMI_TAG_PDEV_BSS_CHAN_INFO_EVENT,
	WMI_TAG_ROAM_LCA_DISALLOW_CONFIG,
	WMI_TAG_VDEV_LIMIT_OFFCHAN_CMD,
	WMI_TAG_ROAM_RSSI_REJECTION_OCE_CONFIG,
	WMI_TAG_UNIT_TEST_EVENT,
	WMI_TAG_ROAM_FILS_OFFLOAD,
	WMI_TAG_PDEV_UPDATE_PMK_CACHE_CMD,
	WMI_TAG_PMK_CACHE,
	WMI_TAG_PDEV_UPDATE_FILS_HLP_PKT_CMD,
	WMI_TAG_ROAM_FILS_SYNCH,
	WMI_TAG_GTK_OFFLOAD_EXTENDED,
	WMI_TAG_ROAM_BG_SCAN_ROAMING,
	WMI_TAG_OIC_PING_OFFLOAD_PARAMS_CMD,
	WMI_TAG_OIC_PING_OFFLOAD_SET_ENABLE_CMD,
	WMI_TAG_OIC_PING_HANDOFF_EVENT,
	WMI_TAG_DHCP_LEASE_RENEW_OFFLOAD_CMD,
	WMI_TAG_DHCP_LEASE_RENEW_EVENT,
	WMI_TAG_BTM_CONFIG,
	WMI_TAG_DEBUG_MESG_FW_DATA_STALL,
	WMI_TAG_WLM_CONFIG_CMD,
	WMI_TAG_PDEV_UPDATE_CTLTABLE_REQUEST,
	WMI_TAG_PDEV_UPDATE_CTLTABLE_EVENT,
	WMI_TAG_ROAM_CND_SCORING_PARAM,
	WMI_TAG_PDEV_CONFIG_VENDOR_OUI_ACTION,
	WMI_TAG_VENDOR_OUI_EXT,
	WMI_TAG_ROAM_SYNCH_FRAME_EVENT,
	WMI_TAG_FD_SEND_FROM_HOST_CMD,
	WMI_TAG_ENABLE_FILS_CMD,
	WMI_TAG_HOST_SWFDA_EVENT,
	WMI_TAG_BCN_OFFLOAD_CTRL_CMD,
	WMI_TAG_PDEV_SET_AC_TX_QUEUE_OPTIMIZED_CMD,
	WMI_TAG_STATS_PERIOD,
	WMI_TAG_NDL_SCHEDULE_UPDATE,
	WMI_TAG_PEER_TID_MSDUQ_QDEPTH_THRESH_UPDATE_CMD,
	WMI_TAG_MSDUQ_QDEPTH_THRESH_UPDATE,
	WMI_TAG_PDEV_SET_RX_FILTER_PROMISCUOUS_CMD,
	WMI_TAG_SAR2_RESULT_EVENT,
	WMI_TAG_SAR_CAPABILITIES,
	WMI_TAG_SAP_OBSS_DETECTION_CFG_CMD,
	WMI_TAG_SAP_OBSS_DETECTION_INFO_EVT,
	WMI_TAG_DMA_RING_CAPABILITIES,
	WMI_TAG_DMA_RING_CFG_REQ,
	WMI_TAG_DMA_RING_CFG_RSP,
	WMI_TAG_DMA_BUF_RELEASE,
	WMI_TAG_DMA_BUF_RELEASE_ENTRY,
	WMI_TAG_SAR_GET_LIMITS_CMD,
	WMI_TAG_SAR_GET_LIMITS_EVENT,
	WMI_TAG_SAR_GET_LIMITS_EVENT_ROW,
	WMI_TAG_OFFLOAD_11K_REPORT,
	WMI_TAG_INVOKE_NEIGHBOR_REPORT,
	WMI_TAG_NEIGHBOR_REPORT_OFFLOAD,
	WMI_TAG_VDEV_SET_CONNECTIVITY_CHECK_STATS,
	WMI_TAG_VDEV_GET_CONNECTIVITY_CHECK_STATS,
	WMI_TAG_BPF_SET_VDEV_ENABLE_CMD,
	WMI_TAG_BPF_SET_VDEV_WORK_MEMORY_CMD,
	WMI_TAG_BPF_GET_VDEV_WORK_MEMORY_CMD,
	WMI_TAG_BPF_GET_VDEV_WORK_MEMORY_RESP_EVT,
	WMI_TAG_PDEV_GET_NFCAL_POWER,
	WMI_TAG_BSS_COLOR_CHANGE_ENABLE,
	WMI_TAG_OBSS_COLOR_COLLISION_DET_CONFIG,
	WMI_TAG_OBSS_COLOR_COLLISION_EVT,
	WMI_TAG_RUNTIME_DPD_RECAL_CMD,
	WMI_TAG_TWT_ENABLE_CMD,
	WMI_TAG_TWT_DISABLE_CMD,
	WMI_TAG_TWT_ADD_DIALOG_CMD,
	WMI_TAG_TWT_DEL_DIALOG_CMD,
	WMI_TAG_TWT_PAUSE_DIALOG_CMD,
	WMI_TAG_TWT_RESUME_DIALOG_CMD,
	WMI_TAG_TWT_ENABLE_COMPLETE_EVENT,
	WMI_TAG_TWT_DISABLE_COMPLETE_EVENT,
	WMI_TAG_TWT_ADD_DIALOG_COMPLETE_EVENT,
	WMI_TAG_TWT_DEL_DIALOG_COMPLETE_EVENT,
	WMI_TAG_TWT_PAUSE_DIALOG_COMPLETE_EVENT,
	WMI_TAG_TWT_RESUME_DIALOG_COMPLETE_EVENT,
	WMI_TAG_REQUEST_ROAM_SCAN_STATS_CMD,
	WMI_TAG_ROAM_SCAN_STATS_EVENT,
	WMI_TAG_PEER_TID_CONFIGURATIONS_CMD,
	WMI_TAG_VDEV_SET_CUSTOM_SW_RETRY_TH_CMD,
	WMI_TAG_GET_TPC_POWER_CMD,
	WMI_TAG_GET_TPC_POWER_EVENT,
	WMI_TAG_DMA_BUF_RELEASE_SPECTRAL_META_DATA,
	WMI_TAG_MOTION_DET_CONFIG_PARAMS_CMD,
	WMI_TAG_MOTION_DET_BASE_LINE_CONFIG_PARAMS_CMD,
	WMI_TAG_MOTION_DET_START_STOP_CMD,
	WMI_TAG_MOTION_DET_BASE_LINE_START_STOP_CMD,
	WMI_TAG_MOTION_DET_EVENT,
	WMI_TAG_MOTION_DET_BASE_LINE_EVENT,
	WMI_TAG_NDP_TRANSPORT_IP,
	WMI_TAG_OBSS_SPATIAL_REUSE_SET_CMD,
	WMI_TAG_ESP_ESTIMATE_EVENT,
	WMI_TAG_NAN_HOST_CONFIG,
	WMI_TAG_SPECTRAL_BIN_SCALING_PARAMS,
	WMI_TAG_PEER_CFR_CAPTURE_CMD,
	WMI_TAG_PEER_CHAN_WIDTH_SWITCH_CMD,
	WMI_TAG_CHAN_WIDTH_PEER_LIST,
	WMI_TAG_OBSS_SPATIAL_REUSE_SET_DEF_OBSS_THRESH_CMD,
	WMI_TAG_PDEV_HE_TB_ACTION_FRM_CMD,
	WMI_TAG_PEER_EXTD2_STATS,
	WMI_TAG_HPCS_PULSE_START_CMD,
	WMI_TAG_PDEV_CTL_FAILSAFE_CHECK_EVENT,
	WMI_TAG_VDEV_CHAINMASK_CONFIG_CMD,
	WMI_TAG_VDEV_BCN_OFFLOAD_QUIET_CONFIG_CMD,
	WMI_TAG_NAN_EVENT_INFO,
	WMI_TAG_NDP_CHANNEL_INFO,
	WMI_TAG_NDP_CMD,
	WMI_TAG_NDP_EVENT,
	WMI_TAG_PDEV_PEER_PKTLOG_FILTER_CMD = 0x301,
	WMI_TAG_PDEV_PEER_PKTLOG_FILTER_INFO,
	WMI_TAG_FILS_DISCOVERY_TMPL_CMD = 0x344,
	WMI_TAG_PDEV_SRG_BSS_COLOR_BITMAP_CMD = 0x37b,
	WMI_TAG_PDEV_SRG_PARTIAL_BSSID_BITMAP_CMD,
	WMI_TAG_PDEV_SRG_OBSS_COLOR_ENABLE_BITMAP_CMD = 0x381,
	WMI_TAG_PDEV_SRG_OBSS_BSSID_ENABLE_BITMAP_CMD,
	WMI_TAG_PDEV_NON_SRG_OBSS_COLOR_ENABLE_BITMAP_CMD,
	WMI_TAG_PDEV_NON_SRG_OBSS_BSSID_ENABLE_BITMAP_CMD,
	WMI_TAG_REGULATORY_RULE_EXT_STRUCT = 0x3A9,
	WMI_TAG_REG_CHAN_LIST_CC_EXT_EVENT,
	WMI_TAG_VDEV_SET_TPC_POWER_CMD = 0x3B5,
	WMI_TAG_VDEV_CH_POWER_INFO,
	WMI_TAG_PDEV_SET_BIOS_SAR_TABLE_CMD = 0x3D8,
	WMI_TAG_PDEV_SET_BIOS_GEO_TABLE_CMD,
	WMI_TAG_MAX
};

enum wmi_tlv_service {
	WMI_TLV_SERVICE_BEACON_OFFLOAD = 0,
	WMI_TLV_SERVICE_SCAN_OFFLOAD = 1,
	WMI_TLV_SERVICE_ROAM_SCAN_OFFLOAD = 2,
	WMI_TLV_SERVICE_BCN_MISS_OFFLOAD = 3,
	WMI_TLV_SERVICE_STA_PWRSAVE = 4,
	WMI_TLV_SERVICE_STA_ADVANCED_PWRSAVE = 5,
	WMI_TLV_SERVICE_AP_UAPSD = 6,
	WMI_TLV_SERVICE_AP_DFS = 7,
	WMI_TLV_SERVICE_11AC = 8,
	WMI_TLV_SERVICE_BLOCKACK = 9,
	WMI_TLV_SERVICE_PHYERR = 10,
	WMI_TLV_SERVICE_BCN_FILTER = 11,
	WMI_TLV_SERVICE_RTT = 12,
	WMI_TLV_SERVICE_WOW = 13,
	WMI_TLV_SERVICE_RATECTRL_CACHE = 14,
	WMI_TLV_SERVICE_IRAM_TIDS = 15,
	WMI_TLV_SERVICE_ARPNS_OFFLOAD = 16,
	WMI_TLV_SERVICE_NLO = 17,
	WMI_TLV_SERVICE_GTK_OFFLOAD = 18,
	WMI_TLV_SERVICE_SCAN_SCH = 19,
	WMI_TLV_SERVICE_CSA_OFFLOAD = 20,
	WMI_TLV_SERVICE_CHATTER = 21,
	WMI_TLV_SERVICE_COEX_FREQAVOID = 22,
	WMI_TLV_SERVICE_PACKET_POWER_SAVE = 23,
	WMI_TLV_SERVICE_FORCE_FW_HANG = 24,
	WMI_TLV_SERVICE_GPIO = 25,
	WMI_TLV_SERVICE_STA_DTIM_PS_MODULATED_DTIM = 26,
	WMI_STA_UAPSD_BASIC_AUTO_TRIG = 27,
	WMI_STA_UAPSD_VAR_AUTO_TRIG = 28,
	WMI_TLV_SERVICE_STA_KEEP_ALIVE = 29,
	WMI_TLV_SERVICE_TX_ENCAP = 30,
	WMI_TLV_SERVICE_AP_PS_DETECT_OUT_OF_SYNC = 31,
	WMI_TLV_SERVICE_EARLY_RX = 32,
	WMI_TLV_SERVICE_STA_SMPS = 33,
	WMI_TLV_SERVICE_FWTEST = 34,
	WMI_TLV_SERVICE_STA_WMMAC = 35,
	WMI_TLV_SERVICE_TDLS = 36,
	WMI_TLV_SERVICE_BURST = 37,
	WMI_TLV_SERVICE_MCC_BCN_INTERVAL_CHANGE = 38,
	WMI_TLV_SERVICE_ADAPTIVE_OCS = 39,
	WMI_TLV_SERVICE_BA_SSN_SUPPORT = 40,
	WMI_TLV_SERVICE_FILTER_IPSEC_NATKEEPALIVE = 41,
	WMI_TLV_SERVICE_WLAN_HB = 42,
	WMI_TLV_SERVICE_LTE_ANT_SHARE_SUPPORT = 43,
	WMI_TLV_SERVICE_BATCH_SCAN = 44,
	WMI_TLV_SERVICE_QPOWER = 45,
	WMI_TLV_SERVICE_PLMREQ = 46,
	WMI_TLV_SERVICE_THERMAL_MGMT = 47,
	WMI_TLV_SERVICE_RMC = 48,
	WMI_TLV_SERVICE_MHF_OFFLOAD = 49,
	WMI_TLV_SERVICE_COEX_SAR = 50,
	WMI_TLV_SERVICE_BCN_TXRATE_OVERRIDE = 51,
	WMI_TLV_SERVICE_NAN = 52,
	WMI_TLV_SERVICE_L1SS_STAT = 53,
	WMI_TLV_SERVICE_ESTIMATE_LINKSPEED = 54,
	WMI_TLV_SERVICE_OBSS_SCAN = 55,
	WMI_TLV_SERVICE_TDLS_OFFCHAN = 56,
	WMI_TLV_SERVICE_TDLS_UAPSD_BUFFER_STA = 57,
	WMI_TLV_SERVICE_TDLS_UAPSD_SLEEP_STA = 58,
	WMI_TLV_SERVICE_IBSS_PWRSAVE = 59,
	WMI_TLV_SERVICE_LPASS = 60,
	WMI_TLV_SERVICE_EXTSCAN = 61,
	WMI_TLV_SERVICE_D0WOW = 62,
	WMI_TLV_SERVICE_HSOFFLOAD = 63,
	WMI_TLV_SERVICE_ROAM_HO_OFFLOAD = 64,
	WMI_TLV_SERVICE_RX_FULL_REORDER = 65,
	WMI_TLV_SERVICE_DHCP_OFFLOAD = 66,
	WMI_TLV_SERVICE_STA_RX_IPA_OFFLOAD_SUPPORT = 67,
	WMI_TLV_SERVICE_MDNS_OFFLOAD = 68,
	WMI_TLV_SERVICE_SAP_AUTH_OFFLOAD = 69,
	WMI_TLV_SERVICE_DUAL_BAND_SIMULTANEOUS_SUPPORT = 70,
	WMI_TLV_SERVICE_OCB = 71,
	WMI_TLV_SERVICE_AP_ARPNS_OFFLOAD = 72,
	WMI_TLV_SERVICE_PER_BAND_CHAINMASK_SUPPORT = 73,
	WMI_TLV_SERVICE_PACKET_FILTER_OFFLOAD = 74,
	WMI_TLV_SERVICE_MGMT_TX_HTT = 75,
	WMI_TLV_SERVICE_MGMT_TX_WMI = 76,
	WMI_TLV_SERVICE_EXT_MSG = 77,
	WMI_TLV_SERVICE_MAWC = 78,
	WMI_TLV_SERVICE_PEER_ASSOC_CONF = 79,
	WMI_TLV_SERVICE_EGAP = 80,
	WMI_TLV_SERVICE_STA_PMF_OFFLOAD = 81,
	WMI_TLV_SERVICE_UNIFIED_WOW_CAPABILITY = 82,
	WMI_TLV_SERVICE_ENHANCED_PROXY_STA = 83,
	WMI_TLV_SERVICE_ATF = 84,
	WMI_TLV_SERVICE_COEX_GPIO = 85,
	WMI_TLV_SERVICE_AUX_SPECTRAL_INTF = 86,
	WMI_TLV_SERVICE_AUX_CHAN_LOAD_INTF = 87,
	WMI_TLV_SERVICE_BSS_CHANNEL_INFO_64 = 88,
	WMI_TLV_SERVICE_ENTERPRISE_MESH = 89,
	WMI_TLV_SERVICE_RESTRT_CHNL_SUPPORT = 90,
	WMI_TLV_SERVICE_BPF_OFFLOAD = 91,
	WMI_TLV_SERVICE_SYNC_DELETE_CMDS = 92,
	WMI_TLV_SERVICE_SMART_ANTENNA_SW_SUPPORT = 93,
	WMI_TLV_SERVICE_SMART_ANTENNA_HW_SUPPORT = 94,
	WMI_TLV_SERVICE_RATECTRL_LIMIT_MAX_MIN_RATES = 95,
	WMI_TLV_SERVICE_NAN_DATA = 96,
	WMI_TLV_SERVICE_NAN_RTT = 97,
	WMI_TLV_SERVICE_11AX = 98,
	WMI_TLV_SERVICE_DEPRECATED_REPLACE = 99,
	WMI_TLV_SERVICE_TDLS_CONN_TRACKER_IN_HOST_MODE = 100,
	WMI_TLV_SERVICE_ENHANCED_MCAST_FILTER = 101,
	WMI_TLV_SERVICE_PERIODIC_CHAN_STAT_SUPPORT = 102,
	WMI_TLV_SERVICE_MESH_11S = 103,
	WMI_TLV_SERVICE_HALF_RATE_QUARTER_RATE_SUPPORT = 104,
	WMI_TLV_SERVICE_VDEV_RX_FILTER = 105,
	WMI_TLV_SERVICE_P2P_LISTEN_OFFLOAD_SUPPORT = 106,
	WMI_TLV_SERVICE_MARK_FIRST_WAKEUP_PACKET = 107,
	WMI_TLV_SERVICE_MULTIPLE_MCAST_FILTER_SET = 108,
	WMI_TLV_SERVICE_HOST_MANAGED_RX_REORDER = 109,
	WMI_TLV_SERVICE_FLASH_RDWR_SUPPORT = 110,
	WMI_TLV_SERVICE_WLAN_STATS_REPORT = 111,
	WMI_TLV_SERVICE_TX_MSDU_ID_NEW_PARTITION_SUPPORT = 112,
	WMI_TLV_SERVICE_DFS_PHYERR_OFFLOAD = 113,
	WMI_TLV_SERVICE_RCPI_SUPPORT = 114,
	WMI_TLV_SERVICE_FW_MEM_DUMP_SUPPORT = 115,
	WMI_TLV_SERVICE_PEER_STATS_INFO = 116,
	WMI_TLV_SERVICE_REGULATORY_DB = 117,
	WMI_TLV_SERVICE_11D_OFFLOAD = 118,
	WMI_TLV_SERVICE_HW_DATA_FILTERING = 119,
	WMI_TLV_SERVICE_MULTIPLE_VDEV_RESTART = 120,
	WMI_TLV_SERVICE_PKT_ROUTING = 121,
	WMI_TLV_SERVICE_CHECK_CAL_VERSION = 122,
	WMI_TLV_SERVICE_OFFCHAN_TX_WMI = 123,
	WMI_TLV_SERVICE_8SS_TX_BFEE  =  124,
	WMI_TLV_SERVICE_EXTENDED_NSS_SUPPORT = 125,
	WMI_TLV_SERVICE_ACK_TIMEOUT = 126,
	WMI_TLV_SERVICE_PDEV_BSS_CHANNEL_INFO_64 = 127,

	/* The first 128 bits */
	WMI_MAX_SERVICE = 128,

	WMI_TLV_SERVICE_CHAN_LOAD_INFO = 128,
	WMI_TLV_SERVICE_TX_PPDU_INFO_STATS_SUPPORT = 129,
	WMI_TLV_SERVICE_VDEV_LIMIT_OFFCHAN_SUPPORT = 130,
	WMI_TLV_SERVICE_FILS_SUPPORT = 131,
	WMI_TLV_SERVICE_WLAN_OIC_PING_OFFLOAD = 132,
	WMI_TLV_SERVICE_WLAN_DHCP_RENEW = 133,
	WMI_TLV_SERVICE_MAWC_SUPPORT = 134,
	WMI_TLV_SERVICE_VDEV_LATENCY_CONFIG = 135,
	WMI_TLV_SERVICE_PDEV_UPDATE_CTLTABLE_SUPPORT = 136,
	WMI_TLV_SERVICE_PKTLOG_SUPPORT_OVER_HTT = 137,
	WMI_TLV_SERVICE_VDEV_MULTI_GROUP_KEY_SUPPORT = 138,
	WMI_TLV_SERVICE_SCAN_PHYMODE_SUPPORT = 139,
	WMI_TLV_SERVICE_THERM_THROT = 140,
	WMI_TLV_SERVICE_BCN_OFFLOAD_START_STOP_SUPPORT = 141,
	WMI_TLV_SERVICE_WOW_WAKEUP_BY_TIMER_PATTERN = 142,
	WMI_TLV_SERVICE_PEER_MAP_UNMAP_V2_SUPPORT = 143,
	WMI_TLV_SERVICE_OFFCHAN_DATA_TID_SUPPORT = 144,
	WMI_TLV_SERVICE_RX_PROMISC_ENABLE_SUPPORT = 145,
	WMI_TLV_SERVICE_SUPPORT_DIRECT_DMA = 146,
	WMI_TLV_SERVICE_AP_OBSS_DETECTION_OFFLOAD = 147,
	WMI_TLV_SERVICE_11K_NEIGHBOUR_REPORT_SUPPORT = 148,
	WMI_TLV_SERVICE_LISTEN_INTERVAL_OFFLOAD_SUPPORT = 149,
	WMI_TLV_SERVICE_BSS_COLOR_OFFLOAD = 150,
	WMI_TLV_SERVICE_RUNTIME_DPD_RECAL = 151,
	WMI_TLV_SERVICE_STA_TWT = 152,
	WMI_TLV_SERVICE_AP_TWT = 153,
	WMI_TLV_SERVICE_GMAC_OFFLOAD_SUPPORT = 154,
	WMI_TLV_SERVICE_SPOOF_MAC_SUPPORT = 155,
	WMI_TLV_SERVICE_PEER_TID_CONFIGS_SUPPORT = 156,
	WMI_TLV_SERVICE_VDEV_SWRETRY_PER_AC_CONFIG_SUPPORT = 157,
	WMI_TLV_SERVICE_DUAL_BEACON_ON_SINGLE_MAC_SCC_SUPPORT = 158,
	WMI_TLV_SERVICE_DUAL_BEACON_ON_SINGLE_MAC_MCC_SUPPORT = 159,
	WMI_TLV_SERVICE_MOTION_DET = 160,
	WMI_TLV_SERVICE_INFRA_MBSSID = 161,
	WMI_TLV_SERVICE_OBSS_SPATIAL_REUSE = 162,
	WMI_TLV_SERVICE_VDEV_DIFFERENT_BEACON_INTERVAL_SUPPORT = 163,
	WMI_TLV_SERVICE_NAN_DBS_SUPPORT = 164,
	WMI_TLV_SERVICE_NDI_DBS_SUPPORT = 165,
	WMI_TLV_SERVICE_NAN_SAP_SUPPORT = 166,
	WMI_TLV_SERVICE_NDI_SAP_SUPPORT = 167,
	WMI_TLV_SERVICE_CFR_CAPTURE_SUPPORT = 168,
	WMI_TLV_SERVICE_CFR_CAPTURE_IND_MSG_TYPE_1 = 169,
	WMI_TLV_SERVICE_ESP_SUPPORT = 170,
	WMI_TLV_SERVICE_PEER_CHWIDTH_CHANGE = 171,
	WMI_TLV_SERVICE_WLAN_HPCS_PULSE = 172,
	WMI_TLV_SERVICE_PER_VDEV_CHAINMASK_CONFIG_SUPPORT = 173,
	WMI_TLV_SERVICE_TX_DATA_MGMT_ACK_RSSI = 174,
	WMI_TLV_SERVICE_NAN_DISABLE_SUPPORT = 175,
	WMI_TLV_SERVICE_HTT_H2T_NO_HTC_HDR_LEN_IN_MSG_LEN = 176,
	WMI_TLV_SERVICE_COEX_SUPPORT_UNEQUAL_ISOLATION = 177,
	WMI_TLV_SERVICE_HW_DB2DBM_CONVERSION_SUPPORT = 178,
	WMI_TLV_SERVICE_SUPPORT_EXTEND_ADDRESS = 179,
	WMI_TLV_SERVICE_BEACON_RECEPTION_STATS = 180,
	WMI_TLV_SERVICE_FETCH_TX_PN = 181,
	WMI_TLV_SERVICE_PEER_UNMAP_RESPONSE_SUPPORT = 182,
	WMI_TLV_SERVICE_TX_PER_PEER_AMPDU_SIZE = 183,
	WMI_TLV_SERVICE_BSS_COLOR_SWITCH_COUNT = 184,
	WMI_TLV_SERVICE_HTT_PEER_STATS_SUPPORT = 185,
	WMI_TLV_SERVICE_UL_RU26_ALLOWED = 186,
	WMI_TLV_SERVICE_GET_MWS_COEX_STATE = 187,
	WMI_TLV_SERVICE_GET_MWS_DPWB_STATE = 188,
	WMI_TLV_SERVICE_GET_MWS_TDM_STATE = 189,
	WMI_TLV_SERVICE_GET_MWS_IDRX_STATE = 190,
	WMI_TLV_SERVICE_GET_MWS_ANTENNA_SHARING_STATE = 191,
	WMI_TLV_SERVICE_ENHANCED_TPC_CONFIG_EVENT = 192,
	WMI_TLV_SERVICE_WLM_STATS_REQUEST = 193,
	WMI_TLV_SERVICE_EXT_PEER_TID_CONFIGS_SUPPORT = 194,
	WMI_TLV_SERVICE_WPA3_FT_SAE_SUPPORT = 195,
	WMI_TLV_SERVICE_WPA3_FT_SUITE_B_SUPPORT = 196,
	WMI_TLV_SERVICE_VOW_ENABLE = 197,
	WMI_TLV_SERVICE_CFR_CAPTURE_IND_EVT_TYPE_1 = 198,
	WMI_TLV_SERVICE_BROADCAST_TWT = 199,
	WMI_TLV_SERVICE_RAP_DETECTION_SUPPORT = 200,
	WMI_TLV_SERVICE_PS_TDCC = 201,
	WMI_TLV_SERVICE_THREE_WAY_COEX_CONFIG_LEGACY   = 202,
	WMI_TLV_SERVICE_THREE_WAY_COEX_CONFIG_OVERRIDE = 203,
	WMI_TLV_SERVICE_TX_PWR_PER_PEER = 204,
	WMI_TLV_SERVICE_STA_PLUS_STA_SUPPORT = 205,
	WMI_TLV_SERVICE_WPA3_FT_FILS = 206,
	WMI_TLV_SERVICE_ADAPTIVE_11R_ROAM = 207,
	WMI_TLV_SERVICE_CHAN_RF_CHARACTERIZATION_INFO = 208,
	WMI_TLV_SERVICE_FW_IFACE_COMBINATION_SUPPORT = 209,
	WMI_TLV_SERVICE_TX_COMPL_TSF64 = 210,
	WMI_TLV_SERVICE_DSM_ROAM_FILTER = 211,
	WMI_TLV_SERVICE_PACKET_CAPTURE_SUPPORT = 212,
	WMI_TLV_SERVICE_PER_PEER_HTT_STATS_RESET = 213,
	WMI_TLV_SERVICE_FREQINFO_IN_METADATA = 219,
	WMI_TLV_SERVICE_EXT2_MSG = 220,
	WMI_TLV_SERVICE_PEER_POWER_SAVE_DURATION_SUPPORT = 246,
	WMI_TLV_SERVICE_SRG_SRP_SPATIAL_REUSE_SUPPORT = 249,
	WMI_TLV_SERVICE_MBSS_PARAM_IN_VDEV_START_SUPPORT = 253,
	WMI_TLV_SERVICE_PASSIVE_SCAN_START_TIME_ENHANCE = 263,

	/* The second 128 bits */
	WMI_MAX_EXT_SERVICE = 256,
	WMI_TLV_SERVICE_SCAN_CONFIG_PER_CHANNEL = 265,
	WMI_TLV_SERVICE_EXT_TPC_REG_SUPPORT = 280,
	WMI_TLV_SERVICE_REG_CC_EXT_EVENT_SUPPORT = 281,
	WMI_TLV_SERVICE_BIOS_SAR_SUPPORT = 326,
	WMI_TLV_SERVICE_SUPPORT_11D_FOR_HOST_SCAN = 357,

	/* The third 128 bits */
	WMI_MAX_EXT2_SERVICE = 384
};

enum {
	WMI_SMPS_FORCED_MODE_NONE = 0,
	WMI_SMPS_FORCED_MODE_DISABLED,
	WMI_SMPS_FORCED_MODE_STATIC,
	WMI_SMPS_FORCED_MODE_DYNAMIC
};

#define WMI_TPC_CHAINMASK_CONFIG_BAND_2G      0
#define WMI_TPC_CHAINMASK_CONFIG_BAND_5G      1
#define WMI_NUM_SUPPORTED_BAND_MAX 2

#define WMI_PEER_MIMO_PS_STATE                          0x1
#define WMI_PEER_AMPDU                                  0x2
#define WMI_PEER_AUTHORIZE                              0x3
#define WMI_PEER_CHWIDTH                                0x4
#define WMI_PEER_NSS                                    0x5
#define WMI_PEER_USE_4ADDR                              0x6
#define WMI_PEER_MEMBERSHIP                             0x7
#define WMI_PEER_USERPOS                                0x8
#define WMI_PEER_CRIT_PROTO_HINT_ENABLED                0x9
#define WMI_PEER_TX_FAIL_CNT_THR                        0xA
#define WMI_PEER_SET_HW_RETRY_CTS2S                     0xB
#define WMI_PEER_IBSS_ATIM_WINDOW_LENGTH                0xC
#define WMI_PEER_PHYMODE                                0xD
#define WMI_PEER_USE_FIXED_PWR                          0xE
#define WMI_PEER_PARAM_FIXED_RATE                       0xF
#define WMI_PEER_SET_MU_WHITELIST                       0x10
#define WMI_PEER_SET_MAX_TX_RATE                        0x11
#define WMI_PEER_SET_MIN_TX_RATE                        0x12
#define WMI_PEER_SET_DEFAULT_ROUTING                    0x13

/* slot time long */
#define WMI_VDEV_SLOT_TIME_LONG         0x1
/* slot time short */
#define WMI_VDEV_SLOT_TIME_SHORT        0x2
/* preablbe long */
#define WMI_VDEV_PREAMBLE_LONG          0x1
/* preablbe short */
#define WMI_VDEV_PREAMBLE_SHORT         0x2

enum wmi_peer_smps_state {
	WMI_PEER_SMPS_PS_NONE = 0x0,
	WMI_PEER_SMPS_STATIC  = 0x1,
	WMI_PEER_SMPS_DYNAMIC = 0x2
};

enum wmi_peer_chwidth {
	WMI_PEER_CHWIDTH_20MHZ = 0,
	WMI_PEER_CHWIDTH_40MHZ = 1,
	WMI_PEER_CHWIDTH_80MHZ = 2,
	WMI_PEER_CHWIDTH_160MHZ = 3,
};

enum wmi_beacon_gen_mode {
	WMI_BEACON_STAGGERED_MODE = 0,
	WMI_BEACON_BURST_MODE = 1
};

enum wmi_direct_buffer_module {
	WMI_DIRECT_BUF_SPECTRAL = 0,
	WMI_DIRECT_BUF_CFR = 1,

	/* keep it last */
	WMI_DIRECT_BUF_MAX
};

/* enum wmi_nss_ratio - NSS ratio received from FW during service ready ext
 *			event
 * WMI_NSS_RATIO_1BY2_NSS -Max nss of 160MHz is equals to half of the max nss
 *			   of 80MHz
 * WMI_NSS_RATIO_3BY4_NSS - Max nss of 160MHz is equals to 3/4 of the max nss
 *			    of 80MHz
 * WMI_NSS_RATIO_1_NSS - Max nss of 160MHz is equals to the max nss of 80MHz
 * WMI_NSS_RATIO_2_NSS - Max nss of 160MHz is equals to two times the max
 *			 nss of 80MHz
 */

enum wmi_nss_ratio {
	WMI_NSS_RATIO_1BY2_NSS = 0x0,
	WMI_NSS_RATIO_3BY4_NSS = 0x1,
	WMI_NSS_RATIO_1_NSS = 0x2,
	WMI_NSS_RATIO_2_NSS = 0x3,
};

enum wmi_dtim_policy {
	WMI_DTIM_POLICY_IGNORE = 1,
	WMI_DTIM_POLICY_NORMAL = 2,
	WMI_DTIM_POLICY_STICK  = 3,
	WMI_DTIM_POLICY_AUTO   = 4,
};

struct wmi_host_pdev_band_to_mac {
	u32 pdev_id;
	u32 start_freq;
	u32 end_freq;
};

struct ath11k_ppe_threshold {
	u32 numss_m1;
	u32 ru_bit_mask;
	u32 ppet16_ppet8_ru3_ru0[PSOC_HOST_MAX_NUM_SS];
};

struct ath11k_service_ext_param {
	u32 default_conc_scan_config_bits;
	u32 default_fw_config_bits;
	struct ath11k_ppe_threshold ppet;
	u32 he_cap_info;
	u32 mpdu_density;
	u32 max_bssid_rx_filters;
	u32 num_hw_modes;
	u32 num_phy;
};

struct ath11k_hw_mode_caps {
	u32 hw_mode_id;
	u32 phy_id_map;
	u32 hw_mode_config_type;
};

#define PSOC_HOST_MAX_PHY_SIZE (3)
#define ATH11K_11B_SUPPORT                 BIT(0)
#define ATH11K_11G_SUPPORT                 BIT(1)
#define ATH11K_11A_SUPPORT                 BIT(2)
#define ATH11K_11N_SUPPORT                 BIT(3)
#define ATH11K_11AC_SUPPORT                BIT(4)
#define ATH11K_11AX_SUPPORT                BIT(5)

struct ath11k_hal_reg_capabilities_ext {
	u32 phy_id;
	u32 eeprom_reg_domain;
	u32 eeprom_reg_domain_ext;
	u32 regcap1;
	u32 regcap2;
	u32 wireless_modes;
	u32 low_2ghz_chan;
	u32 high_2ghz_chan;
	u32 low_5ghz_chan;
	u32 high_5ghz_chan;
};

#define WMI_HOST_MAX_PDEV 3

struct wlan_host_mem_chunk {
	u32 tlv_header;
	u32 req_id;
	u32 ptr;
	u32 size;
} __packed;

struct wmi_host_mem_chunk {
	void *vaddr;
	dma_addr_t paddr;
	u32 len;
	u32 req_id;
};

struct wmi_init_cmd_param {
	u32 tlv_header;
	struct target_resource_config *res_cfg;
	u8 num_mem_chunks;
	struct wmi_host_mem_chunk *mem_chunks;
	u32 hw_mode_id;
	u32 num_band_to_mac;
	struct wmi_host_pdev_band_to_mac band_to_mac[WMI_HOST_MAX_PDEV];
};

struct wmi_pdev_band_to_mac {
	u32 tlv_header;
	u32 pdev_id;
	u32 start_freq;
	u32 end_freq;
} __packed;

struct wmi_pdev_set_hw_mode_cmd_param {
	u32 tlv_header;
	u32 pdev_id;
	u32 hw_mode_index;
	u32 num_band_to_mac;
} __packed;

struct wmi_ppe_threshold {
	u32 numss_m1; /** NSS - 1*/
	union {
		u32 ru_count;
		u32 ru_mask;
	} __packed;
	u32 ppet16_ppet8_ru3_ru0[WMI_MAX_NUM_SS];
} __packed;

#define HW_BD_INFO_SIZE       5

struct wmi_abi_version {
	u32 abi_version_0;
	u32 abi_version_1;
	u32 abi_version_ns_0;
	u32 abi_version_ns_1;
	u32 abi_version_ns_2;
	u32 abi_version_ns_3;
} __packed;

struct wmi_init_cmd {
	u32 tlv_header;
	struct wmi_abi_version host_abi_vers;
	u32 num_host_mem_chunks;
} __packed;

#define WMI_RSRC_CFG_FLAG1_BSS_CHANNEL_INFO_64 BIT(5)
#define WMI_RSRC_CFG_FLAG2_CALC_NEXT_DTIM_COUNT_SET BIT(9)
#define WMI_RSRC_CFG_FLAG1_ACK_RSSI BIT(18)

#define WMI_CFG_HOST_SERVICE_FLAG_REG_CC_EXT 4

struct wmi_resource_config {
	u32 tlv_header;
	u32 num_vdevs;
	u32 num_peers;
	u32 num_offload_peers;
	u32 num_offload_reorder_buffs;
	u32 num_peer_keys;
	u32 num_tids;
	u32 ast_skid_limit;
	u32 tx_chain_mask;
	u32 rx_chain_mask;
	u32 rx_timeout_pri[4];
	u32 rx_decap_mode;
	u32 scan_max_pending_req;
	u32 bmiss_offload_max_vdev;
	u32 roam_offload_max_vdev;
	u32 roam_offload_max_ap_profiles;
	u32 num_mcast_groups;
	u32 num_mcast_table_elems;
	u32 mcast2ucast_mode;
	u32 tx_dbg_log_size;
	u32 num_wds_entries;
	u32 dma_burst_size;
	u32 mac_aggr_delim;
	u32 rx_skip_defrag_timeout_dup_detection_check;
	u32 vow_config;
	u32 gtk_offload_max_vdev;
	u32 num_msdu_desc;
	u32 max_frag_entries;
	u32 num_tdls_vdevs;
	u32 num_tdls_conn_table_entries;
	u32 beacon_tx_offload_max_vdev;
	u32 num_multicast_filter_entries;
	u32 num_wow_filters;
	u32 num_keep_alive_pattern;
	u32 keep_alive_pattern_size;
	u32 max_tdls_concurrent_sleep_sta;
	u32 max_tdls_concurrent_buffer_sta;
	u32 wmi_send_separate;
	u32 num_ocb_vdevs;
	u32 num_ocb_channels;
	u32 num_ocb_schedules;
	u32 flag1;
	u32 smart_ant_cap;
	u32 bk_minfree;
	u32 be_minfree;
	u32 vi_minfree;
	u32 vo_minfree;
	u32 alloc_frag_desc_for_data_pkt;
	u32 num_ns_ext_tuples_cfg;
	u32 bpf_instruction_size;
	u32 max_bssid_rx_filters;
	u32 use_pdev_id;
	u32 max_num_dbs_scan_duty_cycle;
	u32 max_num_group_keys;
	u32 peer_map_unmap_v2_support;
	u32 sched_params;
	u32 twt_ap_pdev_count;
	u32 twt_ap_sta_count;
	u32 max_nlo_ssids;
	u32 num_pkt_filters;
	u32 num_max_sta_vdevs;
	u32 max_bssid_indicator;
	u32 ul_resp_config;
	u32 msdu_flow_override_config0;
	u32 msdu_flow_override_config1;
	u32 flags2;
	u32 host_service_flags;
	u32 max_rnr_neighbours;
	u32 ema_max_vap_cnt;
	u32 ema_max_profile_period;
} __packed;

struct wmi_service_ready_event {
	u32 fw_build_vers;
	struct wmi_abi_version fw_abi_vers;
	u32 phy_capability;
	u32 max_frag_entry;
	u32 num_rf_chains;
	u32 ht_cap_info;
	u32 vht_cap_info;
	u32 vht_supp_mcs;
	u32 hw_min_tx_power;
	u32 hw_max_tx_power;
	u32 sys_cap_info;
	u32 min_pkt_size_enable;
	u32 max_bcn_ie_size;
	u32 num_mem_reqs;
	u32 max_num_scan_channels;
	u32 hw_bd_id;
	u32 hw_bd_info[HW_BD_INFO_SIZE];
	u32 max_supported_macs;
	u32 wmi_fw_sub_feat_caps;
	u32 num_dbs_hw_modes;
	/* txrx_chainmask
	 *    [7:0]   - 2G band tx chain mask
	 *    [15:8]  - 2G band rx chain mask
	 *    [23:16] - 5G band tx chain mask
	 *    [31:24] - 5G band rx chain mask
	 */
	u32 txrx_chainmask;
	u32 default_dbs_hw_mode_index;
	u32 num_msdu_desc;
} __packed;

#define WMI_SERVICE_BM_SIZE	((WMI_MAX_SERVICE + sizeof(u32) - 1) / sizeof(u32))

#define WMI_SERVICE_SEGMENT_BM_SIZE32 4 /* 4x u32 = 128 bits */
#define WMI_SERVICE_EXT_BM_SIZE (WMI_SERVICE_SEGMENT_BM_SIZE32 * sizeof(u32))
#define WMI_AVAIL_SERVICE_BITS_IN_SIZE32 32
#define WMI_SERVICE_BITS_IN_SIZE32 4

struct wmi_service_ready_ext_event {
	u32 default_conc_scan_config_bits;
	u32 default_fw_config_bits;
	struct wmi_ppe_threshold ppet;
	u32 he_cap_info;
	u32 mpdu_density;
	u32 max_bssid_rx_filters;
	u32 fw_build_vers_ext;
	u32 max_nlo_ssids;
	u32 max_bssid_indicator;
	u32 he_cap_info_ext;
} __packed;

struct wmi_soc_mac_phy_hw_mode_caps {
	u32 num_hw_modes;
	u32 num_chainmask_tables;
} __packed;

struct wmi_hw_mode_capabilities {
	u32 tlv_header;
	u32 hw_mode_id;
	u32 phy_id_map;
	u32 hw_mode_config_type;
} __packed;

#define WMI_MAX_HECAP_PHY_SIZE                 (3)
#define WMI_NSS_RATIO_ENABLE_DISABLE_BITPOS    BIT(0)
#define WMI_NSS_RATIO_ENABLE_DISABLE_GET(_val) \
	FIELD_GET(WMI_NSS_RATIO_ENABLE_DISABLE_BITPOS, _val)
#define WMI_NSS_RATIO_INFO_BITPOS              GENMASK(4, 1)
#define WMI_NSS_RATIO_INFO_GET(_val) \
	FIELD_GET(WMI_NSS_RATIO_INFO_BITPOS, _val)

struct wmi_mac_phy_capabilities {
	u32 hw_mode_id;
	u32 pdev_id;
	u32 phy_id;
	u32 supported_flags;
	u32 supported_bands;
	u32 ampdu_density;
	u32 max_bw_supported_2g;
	u32 ht_cap_info_2g;
	u32 vht_cap_info_2g;
	u32 vht_supp_mcs_2g;
	u32 he_cap_info_2g;
	u32 he_supp_mcs_2g;
	u32 tx_chain_mask_2g;
	u32 rx_chain_mask_2g;
	u32 max_bw_supported_5g;
	u32 ht_cap_info_5g;
	u32 vht_cap_info_5g;
	u32 vht_supp_mcs_5g;
	u32 he_cap_info_5g;
	u32 he_supp_mcs_5g;
	u32 tx_chain_mask_5g;
	u32 rx_chain_mask_5g;
	u32 he_cap_phy_info_2g[WMI_MAX_HECAP_PHY_SIZE];
	u32 he_cap_phy_info_5g[WMI_MAX_HECAP_PHY_SIZE];
	struct wmi_ppe_threshold he_ppet2g;
	struct wmi_ppe_threshold he_ppet5g;
	u32 chainmask_table_id;
	u32 lmac_id;
	u32 he_cap_info_2g_ext;
	u32 he_cap_info_5g_ext;
	u32 he_cap_info_internal;
	u32 wireless_modes;
	u32 low_2ghz_chan_freq;
	u32 high_2ghz_chan_freq;
	u32 low_5ghz_chan_freq;
	u32 high_5ghz_chan_freq;
	u32 nss_ratio;
} __packed;

struct wmi_hal_reg_capabilities_ext {
	u32 tlv_header;
	u32 phy_id;
	u32 eeprom_reg_domain;
	u32 eeprom_reg_domain_ext;
	u32 regcap1;
	u32 regcap2;
	u32 wireless_modes;
	u32 low_2ghz_chan;
	u32 high_2ghz_chan;
	u32 low_5ghz_chan;
	u32 high_5ghz_chan;
} __packed;

struct wmi_soc_hal_reg_capabilities {
	u32 num_phy;
} __packed;

/* 2 word representation of MAC addr */
struct wmi_mac_addr {
	union {
		u8 addr[6];
		struct {
			u32 word0;
			u32 word1;
		} __packed;
	} __packed;
} __packed;

struct wmi_dma_ring_capabilities {
	u32 tlv_header;
	u32 pdev_id;
	u32 module_id;
	u32 min_elem;
	u32 min_buf_sz;
	u32 min_buf_align;
} __packed;

struct wmi_ready_event_min {
	struct wmi_abi_version fw_abi_vers;
	struct wmi_mac_addr mac_addr;
	u32 status;
	u32 num_dscp_table;
	u32 num_extra_mac_addr;
	u32 num_total_peers;
	u32 num_extra_peers;
} __packed;

struct wmi_ready_event {
	struct wmi_ready_event_min ready_event_min;
	u32 max_ast_index;
	u32 pktlog_defs_checksum;
} __packed;

struct wmi_service_available_event {
	u32 wmi_service_segment_offset;
	u32 wmi_service_segment_bitmap[WMI_SERVICE_SEGMENT_BM_SIZE32];
} __packed;

struct ath11k_pdev_wmi {
	struct ath11k_wmi_base *wmi_ab;
	enum ath11k_htc_ep_id eid;
	u32 rx_decap_mode;
	wait_queue_head_t tx_ce_desc_wq;
};

struct vdev_create_params {
	u8 if_id;
	u32 type;
	u32 subtype;
	struct {
		u8 tx;
		u8 rx;
	} chains[NUM_NL80211_BANDS];
	u32 pdev_id;
	u32 mbssid_flags;
	u32 mbssid_tx_vdev_id;
};

struct wmi_vdev_create_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 vdev_type;
	u32 vdev_subtype;
	struct wmi_mac_addr vdev_macaddr;
	u32 num_cfg_txrx_streams;
	u32 pdev_id;
	u32 mbssid_flags;
	u32 mbssid_tx_vdev_id;
} __packed;

struct wmi_vdev_txrx_streams {
	u32 tlv_header;
	u32 band;
	u32 supported_tx_streams;
	u32 supported_rx_streams;
} __packed;

struct wmi_vdev_delete_cmd {
	u32 tlv_header;
	u32 vdev_id;
} __packed;

struct wmi_vdev_up_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 vdev_assoc_id;
	struct wmi_mac_addr vdev_bssid;
	struct wmi_mac_addr tx_vdev_bssid;
	u32 nontx_profile_idx;
	u32 nontx_profile_cnt;
} __packed;

struct wmi_vdev_stop_cmd {
	u32 tlv_header;
	u32 vdev_id;
} __packed;

struct wmi_vdev_down_cmd {
	u32 tlv_header;
	u32 vdev_id;
} __packed;

#define WMI_VDEV_START_HIDDEN_SSID  BIT(0)
#define WMI_VDEV_START_PMF_ENABLED  BIT(1)
#define WMI_VDEV_START_LDPC_RX_ENABLED BIT(3)
#define WMI_VDEV_START_HW_ENCRYPTION_DISABLED BIT(4)

struct wmi_ssid {
	u32 ssid_len;
	u32 ssid[8];
} __packed;

#define ATH11K_VDEV_SETUP_TIMEOUT_HZ (1 * HZ)

struct wmi_vdev_start_request_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 requestor_id;
	u32 beacon_interval;
	u32 dtim_period;
	u32 flags;
	struct wmi_ssid ssid;
	u32 bcn_tx_rate;
	u32 bcn_txpower;
	u32 num_noa_descriptors;
	u32 disable_hw_ack;
	u32 preferred_tx_streams;
	u32 preferred_rx_streams;
	u32 he_ops;
	u32 cac_duration_ms;
	u32 regdomain;
	u32 min_data_rate;
	u32 mbssid_flags;
	u32 mbssid_tx_vdev_id;
} __packed;

#define MGMT_TX_DL_FRM_LEN		     64
#define WMI_MAC_MAX_SSID_LENGTH              32
struct mac_ssid {
	u8 length;
	u8 mac_ssid[WMI_MAC_MAX_SSID_LENGTH];
} __packed;

struct wmi_p2p_noa_descriptor {
	u32 type_count;
	u32 duration;
	u32 interval;
	u32 start_time;
};

struct channel_param {
	u8 chan_id;
	u8 pwr;
	u32 mhz;
	u32 half_rate:1,
	    quarter_rate:1,
	    dfs_set:1,
	    dfs_set_cfreq2:1,
	    is_chan_passive:1,
	    allow_ht:1,
	    allow_vht:1,
	    allow_he:1,
	    set_agile:1,
	    psc_channel:1;
	u32 phy_mode;
	u32 cfreq1;
	u32 cfreq2;
	char   maxpower;
	char   minpower;
	char   maxregpower;
	u8  antennamax;
	u8  reg_class_id;
} __packed;

enum wmi_phy_mode {
	MODE_11A        = 0,
	MODE_11G        = 1,   /* 11b/g Mode */
	MODE_11B        = 2,   /* 11b Mode */
	MODE_11GONLY    = 3,   /* 11g only Mode */
	MODE_11NA_HT20   = 4,
	MODE_11NG_HT20   = 5,
	MODE_11NA_HT40   = 6,
	MODE_11NG_HT40   = 7,
	MODE_11AC_VHT20 = 8,
	MODE_11AC_VHT40 = 9,
	MODE_11AC_VHT80 = 10,
	MODE_11AC_VHT20_2G = 11,
	MODE_11AC_VHT40_2G = 12,
	MODE_11AC_VHT80_2G = 13,
	MODE_11AC_VHT80_80 = 14,
	MODE_11AC_VHT160 = 15,
	MODE_11AX_HE20 = 16,
	MODE_11AX_HE40 = 17,
	MODE_11AX_HE80 = 18,
	MODE_11AX_HE80_80 = 19,
	MODE_11AX_HE160 = 20,
	MODE_11AX_HE20_2G = 21,
	MODE_11AX_HE40_2G = 22,
	MODE_11AX_HE80_2G = 23,
	MODE_UNKNOWN = 24,
	MODE_MAX = 24
};

static inline const char *ath11k_wmi_phymode_str(enum wmi_phy_mode mode)
{
	switch (mode) {
	case MODE_11A:
		return "11a";
	case MODE_11G:
		return "11g";
	case MODE_11B:
		return "11b";
	case MODE_11GONLY:
		return "11gonly";
	case MODE_11NA_HT20:
		return "11na-ht20";
	case MODE_11NG_HT20:
		return "11ng-ht20";
	case MODE_11NA_HT40:
		return "11na-ht40";
	case MODE_11NG_HT40:
		return "11ng-ht40";
	case MODE_11AC_VHT20:
		return "11ac-vht20";
	case MODE_11AC_VHT40:
		return "11ac-vht40";
	case MODE_11AC_VHT80:
		return "11ac-vht80";
	case MODE_11AC_VHT160:
		return "11ac-vht160";
	case MODE_11AC_VHT80_80:
		return "11ac-vht80+80";
	case MODE_11AC_VHT20_2G:
		return "11ac-vht20-2g";
	case MODE_11AC_VHT40_2G:
		return "11ac-vht40-2g";
	case MODE_11AC_VHT80_2G:
		return "11ac-vht80-2g";
	case MODE_11AX_HE20:
		return "11ax-he20";
	case MODE_11AX_HE40:
		return "11ax-he40";
	case MODE_11AX_HE80:
		return "11ax-he80";
	case MODE_11AX_HE80_80:
		return "11ax-he80+80";
	case MODE_11AX_HE160:
		return "11ax-he160";
	case MODE_11AX_HE20_2G:
		return "11ax-he20-2g";
	case MODE_11AX_HE40_2G:
		return "11ax-he40-2g";
	case MODE_11AX_HE80_2G:
		return "11ax-he80-2g";
	case MODE_UNKNOWN:
		/* skip */
		break;

		/* no default handler to allow compiler to check that the
		 * enum is fully handled
		 */
	}

	return "<unknown>";
}

struct wmi_channel_arg {
	u32 freq;
	u32 band_center_freq1;
	u32 band_center_freq2;
	bool passive;
	bool allow_ibss;
	bool allow_ht;
	bool allow_vht;
	bool ht40plus;
	bool chan_radar;
	bool freq2_radar;
	bool allow_he;
	u32 min_power;
	u32 max_power;
	u32 max_reg_power;
	u32 max_antenna_gain;
	enum wmi_phy_mode mode;
};

struct wmi_vdev_start_req_arg {
	u32 vdev_id;
	struct wmi_channel_arg channel;
	u32 bcn_intval;
	u32 dtim_period;
	u8 *ssid;
	u32 ssid_len;
	u32 bcn_tx_rate;
	u32 bcn_tx_power;
	bool disable_hw_ack;
	bool hidden_ssid;
	bool pmf_enabled;
	u32 he_ops;
	u32 cac_duration_ms;
	u32 regdomain;
	u32 pref_rx_streams;
	u32 pref_tx_streams;
	u32 num_noa_descriptors;
	u32 min_data_rate;
	u32 mbssid_flags;
	u32 mbssid_tx_vdev_id;
};

struct peer_create_params {
	const u8 *peer_addr;
	u32 peer_type;
	u32 vdev_id;
};

struct peer_delete_params {
	u8 vdev_id;
};

struct peer_flush_params {
	u32 peer_tid_bitmap;
	u8 vdev_id;
};

struct pdev_set_regdomain_params {
	u16 current_rd_in_use;
	u16 current_rd_2g;
	u16 current_rd_5g;
	u32 ctl_2g;
	u32 ctl_5g;
	u8 dfs_domain;
	u32 pdev_id;
};

struct rx_reorder_queue_remove_params {
	u8 *peer_macaddr;
	u16 vdev_id;
	u32 peer_tid_bitmap;
};

#define WMI_HOST_PDEV_ID_SOC 0xFF
#define WMI_HOST_PDEV_ID_0   0
#define WMI_HOST_PDEV_ID_1   1
#define WMI_HOST_PDEV_ID_2   2

#define WMI_PDEV_ID_SOC         0
#define WMI_PDEV_ID_1ST         1
#define WMI_PDEV_ID_2ND         2
#define WMI_PDEV_ID_3RD         3

/* Freq units in MHz */
#define REG_RULE_START_FREQ			0x0000ffff
#define REG_RULE_END_FREQ			0xffff0000
#define REG_RULE_FLAGS				0x0000ffff
#define REG_RULE_MAX_BW				0x0000ffff
#define REG_RULE_REG_PWR			0x00ff0000
#define REG_RULE_ANT_GAIN			0xff000000
#define REG_RULE_PSD_INFO			BIT(0)
#define REG_RULE_PSD_EIRP			0xff0000

#define WMI_VDEV_PARAM_TXBF_SU_TX_BFEE BIT(0)
#define WMI_VDEV_PARAM_TXBF_MU_TX_BFEE BIT(1)
#define WMI_VDEV_PARAM_TXBF_SU_TX_BFER BIT(2)
#define WMI_VDEV_PARAM_TXBF_MU_TX_BFER BIT(3)

#define HE_PHYCAP_BYTE_0	0
#define HE_PHYCAP_BYTE_1	1
#define HE_PHYCAP_BYTE_2	2
#define HE_PHYCAP_BYTE_3	3
#define HE_PHYCAP_BYTE_4	4

#define HECAP_PHY_SU_BFER		BIT(7)
#define HECAP_PHY_SU_BFEE		BIT(0)
#define HECAP_PHY_MU_BFER		BIT(1)
#define HECAP_PHY_UL_MUMIMO		BIT(6)
#define HECAP_PHY_UL_MUOFDMA		BIT(7)

#define HECAP_PHY_SUBFMR_GET(hecap_phy) \
	FIELD_GET(HECAP_PHY_SU_BFER, hecap_phy[HE_PHYCAP_BYTE_3])

#define HECAP_PHY_SUBFME_GET(hecap_phy) \
	FIELD_GET(HECAP_PHY_SU_BFEE, hecap_phy[HE_PHYCAP_BYTE_4])

#define HECAP_PHY_MUBFMR_GET(hecap_phy) \
	FIELD_GET(HECAP_PHY_MU_BFER, hecap_phy[HE_PHYCAP_BYTE_4])

#define HECAP_PHY_ULMUMIMO_GET(hecap_phy) \
	FIELD_GET(HECAP_PHY_UL_MUMIMO, hecap_phy[HE_PHYCAP_BYTE_2])

#define HECAP_PHY_ULOFDMA_GET(hecap_phy) \
	FIELD_GET(HECAP_PHY_UL_MUOFDMA, hecap_phy[HE_PHYCAP_BYTE_2])

#define HE_MODE_SU_TX_BFEE	BIT(0)
#define HE_MODE_SU_TX_BFER	BIT(1)
#define HE_MODE_MU_TX_BFEE	BIT(2)
#define HE_MODE_MU_TX_BFER	BIT(3)
#define HE_MODE_DL_OFDMA	BIT(4)
#define HE_MODE_UL_OFDMA	BIT(5)
#define HE_MODE_UL_MUMIMO	BIT(6)

#define HE_DL_MUOFDMA_ENABLE	1
#define HE_UL_MUOFDMA_ENABLE	1
#define HE_DL_MUMIMO_ENABLE	1
#define HE_UL_MUMIMO_ENABLE	1
#define HE_MU_BFEE_ENABLE	1
#define HE_SU_BFEE_ENABLE	1
#define HE_MU_BFER_ENABLE	1
#define HE_SU_BFER_ENABLE	1

#define HE_VHT_SOUNDING_MODE_ENABLE		1
#define HE_SU_MU_SOUNDING_MODE_ENABLE		1
#define HE_TRIG_NONTRIG_SOUNDING_MODE_ENABLE	1

/* HE or VHT Sounding */
#define HE_VHT_SOUNDING_MODE		BIT(0)
/* SU or MU Sounding */
#define HE_SU_MU_SOUNDING_MODE		BIT(2)
/* Trig or Non-Trig Sounding */
#define HE_TRIG_NONTRIG_SOUNDING_MODE	BIT(3)

#define WMI_TXBF_STS_CAP_OFFSET_LSB	4
#define WMI_TXBF_STS_CAP_OFFSET_MASK	0x70
#define WMI_BF_SOUND_DIM_OFFSET_LSB	8
#define WMI_BF_SOUND_DIM_OFFSET_MASK	0x700

struct pdev_params {
	u32 param_id;
	u32 param_value;
};

enum wmi_peer_type {
	WMI_PEER_TYPE_DEFAULT = 0,
	WMI_PEER_TYPE_BSS = 1,
	WMI_PEER_TYPE_TDLS = 2,
};

struct wmi_peer_create_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 peer_type;
} __packed;

struct wmi_peer_delete_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
} __packed;

struct wmi_peer_reorder_queue_setup_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 tid;
	u32 queue_ptr_lo;
	u32 queue_ptr_hi;
	u32 queue_no;
	u32 ba_window_size_valid;
	u32 ba_window_size;
} __packed;

struct wmi_peer_reorder_queue_remove_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 tid_mask;
} __packed;

struct gpio_config_params {
	u32 gpio_num;
	u32 input;
	u32 pull_type;
	u32 intr_mode;
};

enum wmi_gpio_type {
	WMI_GPIO_PULL_NONE,
	WMI_GPIO_PULL_UP,
	WMI_GPIO_PULL_DOWN
};

enum wmi_gpio_intr_type {
	WMI_GPIO_INTTYPE_DISABLE,
	WMI_GPIO_INTTYPE_RISING_EDGE,
	WMI_GPIO_INTTYPE_FALLING_EDGE,
	WMI_GPIO_INTTYPE_BOTH_EDGE,
	WMI_GPIO_INTTYPE_LEVEL_LOW,
	WMI_GPIO_INTTYPE_LEVEL_HIGH
};

enum wmi_bss_chan_info_req_type {
	WMI_BSS_SURVEY_REQ_TYPE_READ = 1,
	WMI_BSS_SURVEY_REQ_TYPE_READ_CLEAR,
};

struct wmi_gpio_config_cmd_param {
	u32 tlv_header;
	u32 gpio_num;
	u32 input;
	u32 pull_type;
	u32 intr_mode;
};

struct gpio_output_params {
	u32 gpio_num;
	u32 set;
};

struct wmi_gpio_output_cmd_param {
	u32 tlv_header;
	u32 gpio_num;
	u32 set;
};

struct set_fwtest_params {
	u32 arg;
	u32 value;
};

struct wmi_fwtest_set_param_cmd_param {
	u32 tlv_header;
	u32 param_id;
	u32 param_value;
};

struct wmi_pdev_set_param_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 param_id;
	u32 param_value;
} __packed;

struct wmi_pdev_set_ps_mode_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 sta_ps_mode;
} __packed;

struct wmi_pdev_suspend_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 suspend_opt;
} __packed;

struct wmi_pdev_resume_cmd {
	u32 tlv_header;
	u32 pdev_id;
} __packed;

struct wmi_pdev_bss_chan_info_req_cmd {
	u32 tlv_header;
	/* ref wmi_bss_chan_info_req_type */
	u32 req_type;
	u32 pdev_id;
} __packed;

struct wmi_ap_ps_peer_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 param;
	u32 value;
} __packed;

struct wmi_sta_powersave_param_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 param;
	u32 value;
} __packed;

struct wmi_pdev_set_regdomain_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 reg_domain;
	u32 reg_domain_2g;
	u32 reg_domain_5g;
	u32 conformance_test_limit_2g;
	u32 conformance_test_limit_5g;
	u32 dfs_domain;
} __packed;

struct wmi_peer_set_param_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 param_id;
	u32 param_value;
} __packed;

struct wmi_peer_flush_tids_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 peer_tid_bitmap;
} __packed;

struct wmi_dfs_phyerr_offload_cmd {
	u32 tlv_header;
	u32 pdev_id;
} __packed;

struct wmi_bcn_offload_ctrl_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 bcn_ctrl_op;
} __packed;

enum scan_dwelltime_adaptive_mode {
	SCAN_DWELL_MODE_DEFAULT = 0,
	SCAN_DWELL_MODE_CONSERVATIVE = 1,
	SCAN_DWELL_MODE_MODERATE = 2,
	SCAN_DWELL_MODE_AGGRESSIVE = 3,
	SCAN_DWELL_MODE_STATIC = 4
};

#define WLAN_SSID_MAX_LEN 32

struct element_info {
	u32 len;
	u8 *ptr;
};

struct wlan_ssid {
	u8 length;
	u8 ssid[WLAN_SSID_MAX_LEN];
};

struct wmi_vdev_ch_power_info {
	u32 tlv_header;

	/* Channel center frequency (MHz) */
	u32 chan_cfreq;

	/* Unit: dBm, either PSD/EIRP power for this frequency or
	 * incremental for non-PSD BW
	 */
	u32 tx_power;
} __packed;

struct wmi_vdev_set_tpc_power_cmd {
	u32 tlv_header;
	u32 vdev_id;

	/* Value: 0 or 1, is PSD power or not */
	u32 psd_power;

	 /* Maximum EIRP power (dBm units), valid only if power is PSD */
	u32 eirp_power;

	/* Type: WMI_6GHZ_REG_TYPE, used for halphy CTL lookup */
	u32 power_type_6ghz;

	/* This fixed_param TLV is followed by the below TLVs:
	 * num_pwr_levels of wmi_vdev_ch_power_info
	 * For PSD power, it is the PSD/EIRP power of the frequency (20 MHz chunks).
	 * For non-PSD power, the power values are for 20, 40, and till
	 * BSS BW power levels.
	 * The num_pwr_levels will be checked by sw how many elements present
	 * in the variable-length array.
	 */
} __packed;

#define WMI_IE_BITMAP_SIZE             8

/* prefix used by scan requestor ids on the host */
#define WMI_HOST_SCAN_REQUESTOR_ID_PREFIX 0xA000

/* prefix used by scan request ids generated on the host */
/* host cycles through the lower 12 bits to generate ids */
#define WMI_HOST_SCAN_REQ_ID_PREFIX 0xA000

/* Values lower than this may be refused by some firmware revisions with a scan
 * completion with a timedout reason.
 */
#define WMI_SCAN_CHAN_MIN_TIME_MSEC 40

/* Scan priority numbers must be sequential, starting with 0 */
enum wmi_scan_priority {
	WMI_SCAN_PRIORITY_VERY_LOW = 0,
	WMI_SCAN_PRIORITY_LOW,
	WMI_SCAN_PRIORITY_MEDIUM,
	WMI_SCAN_PRIORITY_HIGH,
	WMI_SCAN_PRIORITY_VERY_HIGH,
	WMI_SCAN_PRIORITY_COUNT   /* number of priorities supported */
};

enum wmi_scan_event_type {
	WMI_SCAN_EVENT_STARTED              = BIT(0),
	WMI_SCAN_EVENT_COMPLETED            = BIT(1),
	WMI_SCAN_EVENT_BSS_CHANNEL          = BIT(2),
	WMI_SCAN_EVENT_FOREIGN_CHAN         = BIT(3),
	WMI_SCAN_EVENT_DEQUEUED             = BIT(4),
	/* possibly by high-prio scan */
	WMI_SCAN_EVENT_PREEMPTED            = BIT(5),
	WMI_SCAN_EVENT_START_FAILED         = BIT(6),
	WMI_SCAN_EVENT_RESTARTED            = BIT(7),
	WMI_SCAN_EVENT_FOREIGN_CHAN_EXIT    = BIT(8),
	WMI_SCAN_EVENT_SUSPENDED            = BIT(9),
	WMI_SCAN_EVENT_RESUMED              = BIT(10),
	WMI_SCAN_EVENT_MAX                  = BIT(15),
};

enum wmi_scan_completion_reason {
	WMI_SCAN_REASON_COMPLETED,
	WMI_SCAN_REASON_CANCELLED,
	WMI_SCAN_REASON_PREEMPTED,
	WMI_SCAN_REASON_TIMEDOUT,
	WMI_SCAN_REASON_INTERNAL_FAILURE,
	WMI_SCAN_REASON_MAX,
};

struct  wmi_start_scan_cmd {
	u32 tlv_header;
	u32 scan_id;
	u32 scan_req_id;
	u32 vdev_id;
	u32 scan_priority;
	u32 notify_scan_events;
	u32 dwell_time_active;
	u32 dwell_time_passive;
	u32 min_rest_time;
	u32 max_rest_time;
	u32 repeat_probe_time;
	u32 probe_spacing_time;
	u32 idle_time;
	u32 max_scan_time;
	u32 probe_delay;
	u32 scan_ctrl_flags;
	u32 burst_duration;
	u32 num_chan;
	u32 num_bssid;
	u32 num_ssids;
	u32 ie_len;
	u32 n_probes;
	struct wmi_mac_addr mac_addr;
	struct wmi_mac_addr mac_mask;
	u32 ie_bitmap[WMI_IE_BITMAP_SIZE];
	u32 num_vendor_oui;
	u32 scan_ctrl_flags_ext;
	u32 dwell_time_active_2g;
	u32 dwell_time_active_6g;
	u32 dwell_time_passive_6g;
	u32 scan_start_offset;
} __packed;

#define WMI_SCAN_FLAG_PASSIVE        0x1
#define WMI_SCAN_ADD_BCAST_PROBE_REQ 0x2
#define WMI_SCAN_ADD_CCK_RATES       0x4
#define WMI_SCAN_ADD_OFDM_RATES      0x8
#define WMI_SCAN_CHAN_STAT_EVENT     0x10
#define WMI_SCAN_FILTER_PROBE_REQ    0x20
#define WMI_SCAN_BYPASS_DFS_CHN      0x40
#define WMI_SCAN_CONTINUE_ON_ERROR   0x80
#define WMI_SCAN_FILTER_PROMISCUOS   0x100
#define WMI_SCAN_FLAG_FORCE_ACTIVE_ON_DFS 0x200
#define WMI_SCAN_ADD_TPC_IE_IN_PROBE_REQ  0x400
#define WMI_SCAN_ADD_DS_IE_IN_PROBE_REQ   0x800
#define WMI_SCAN_ADD_SPOOF_MAC_IN_PROBE_REQ   0x1000
#define WMI_SCAN_OFFCHAN_MGMT_TX    0x2000
#define WMI_SCAN_OFFCHAN_DATA_TX    0x4000
#define WMI_SCAN_CAPTURE_PHY_ERROR  0x8000
#define WMI_SCAN_FLAG_STRICT_PASSIVE_ON_PCHN 0x10000
#define WMI_SCAN_FLAG_HALF_RATE_SUPPORT      0x20000
#define WMI_SCAN_FLAG_QUARTER_RATE_SUPPORT   0x40000
#define WMI_SCAN_RANDOM_SEQ_NO_IN_PROBE_REQ 0x80000
#define WMI_SCAN_ENABLE_IE_WHTELIST_IN_PROBE_REQ 0x100000

#define WMI_SCAN_DWELL_MODE_MASK 0x00E00000
#define WMI_SCAN_DWELL_MODE_SHIFT        21
#define WMI_SCAN_FLAG_EXT_PASSIVE_SCAN_START_TIME_ENHANCE   0x00000800

#define WMI_SCAN_CONFIG_PER_CHANNEL_MASK	GENMASK(19, 0)
#define WMI_SCAN_CH_FLAG_SCAN_ONLY_IF_RNR_FOUND	BIT(20)

enum {
	WMI_SCAN_DWELL_MODE_DEFAULT      = 0,
	WMI_SCAN_DWELL_MODE_CONSERVATIVE = 1,
	WMI_SCAN_DWELL_MODE_MODERATE     = 2,
	WMI_SCAN_DWELL_MODE_AGGRESSIVE   = 3,
	WMI_SCAN_DWELL_MODE_STATIC       = 4,
};

#define WMI_SCAN_SET_DWELL_MODE(flag, mode) \
	((flag) |= (((mode) << WMI_SCAN_DWELL_MODE_SHIFT) & \
		    WMI_SCAN_DWELL_MODE_MASK))

struct hint_short_ssid {
	u32 freq_flags;
	u32 short_ssid;
};

struct hint_bssid {
	u32 freq_flags;
	struct wmi_mac_addr bssid;
};

struct scan_req_params {
	u32 scan_id;
	u32 scan_req_id;
	u32 vdev_id;
	u32 pdev_id;
	enum wmi_scan_priority scan_priority;
	u32 scan_ev_started:1,
	    scan_ev_completed:1,
	    scan_ev_bss_chan:1,
	    scan_ev_foreign_chan:1,
	    scan_ev_dequeued:1,
	    scan_ev_preempted:1,
	    scan_ev_start_failed:1,
	    scan_ev_restarted:1,
	    scan_ev_foreign_chn_exit:1,
	    scan_ev_invalid:1,
	    scan_ev_gpio_timeout:1,
	    scan_ev_suspended:1,
	    scan_ev_resumed:1;
	u32 scan_ctrl_flags_ext;
	u32 dwell_time_active;
	u32 dwell_time_active_2g;
	u32 dwell_time_passive;
	u32 dwell_time_active_6g;
	u32 dwell_time_passive_6g;
	u32 min_rest_time;
	u32 max_rest_time;
	u32 repeat_probe_time;
	u32 probe_spacing_time;
	u32 idle_time;
	u32 max_scan_time;
	u32 probe_delay;
	u32 scan_f_passive:1,
	    scan_f_bcast_probe:1,
	    scan_f_cck_rates:1,
	    scan_f_ofdm_rates:1,
	    scan_f_chan_stat_evnt:1,
	    scan_f_filter_prb_req:1,
	    scan_f_bypass_dfs_chn:1,
	    scan_f_continue_on_err:1,
	    scan_f_offchan_mgmt_tx:1,
	    scan_f_offchan_data_tx:1,
	    scan_f_promisc_mode:1,
	    scan_f_capture_phy_err:1,
	    scan_f_strict_passive_pch:1,
	    scan_f_half_rate:1,
	    scan_f_quarter_rate:1,
	    scan_f_force_active_dfs_chn:1,
	    scan_f_add_tpc_ie_in_probe:1,
	    scan_f_add_ds_ie_in_probe:1,
	    scan_f_add_spoofed_mac_in_probe:1,
	    scan_f_add_rand_seq_in_probe:1,
	    scan_f_en_ie_whitelist_in_probe:1,
	    scan_f_forced:1,
	    scan_f_2ghz:1,
	    scan_f_5ghz:1,
	    scan_f_80mhz:1;
	enum scan_dwelltime_adaptive_mode adaptive_dwell_time_mode;
	u32 burst_duration;
	u32 num_chan;
	u32 num_bssid;
	u32 num_ssids;
	u32 n_probes;
	u32 *chan_list;
	u32 notify_scan_events;
	struct wlan_ssid ssid[WLAN_SCAN_PARAMS_MAX_SSID];
	struct wmi_mac_addr bssid_list[WLAN_SCAN_PARAMS_MAX_BSSID];
	struct element_info extraie;
	struct element_info htcap;
	struct element_info vhtcap;
	u32 num_hint_s_ssid;
	u32 num_hint_bssid;
	struct hint_short_ssid hint_s_ssid[WLAN_SCAN_MAX_HINT_S_SSID];
	struct hint_bssid hint_bssid[WLAN_SCAN_MAX_HINT_BSSID];
	struct wmi_mac_addr mac_addr;
	struct wmi_mac_addr mac_mask;
};

struct wmi_ssid_arg {
	int len;
	const u8 *ssid;
};

struct wmi_bssid_arg {
	const u8 *bssid;
};

#define WMI_SCAN_STOP_ONE       0x00000000
#define WMI_SCN_STOP_VAP_ALL    0x01000000
#define WMI_SCAN_STOP_ALL       0x04000000

/* Prefix 0xA000 indicates that the scan request
 * is trigger by HOST
 */
#define ATH11K_SCAN_ID          0xA000

enum scan_cancel_req_type {
	WLAN_SCAN_CANCEL_SINGLE = 1,
	WLAN_SCAN_CANCEL_VDEV_ALL,
	WLAN_SCAN_CANCEL_PDEV_ALL,
};

struct scan_cancel_param {
	u32 requester;
	u32 scan_id;
	enum scan_cancel_req_type req_type;
	u32 vdev_id;
	u32 pdev_id;
};

struct  wmi_bcn_send_from_host_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 data_len;
	union {
		u32 frag_ptr;
		u32 frag_ptr_lo;
	};
	u32 frame_ctrl;
	u32 dtim_flag;
	u32 bcn_antenna;
	u32 frag_ptr_hi;
};

#define WMI_CHAN_INFO_MODE		GENMASK(5, 0)
#define WMI_CHAN_INFO_HT40_PLUS		BIT(6)
#define WMI_CHAN_INFO_PASSIVE		BIT(7)
#define WMI_CHAN_INFO_ADHOC_ALLOWED	BIT(8)
#define WMI_CHAN_INFO_AP_DISABLED	BIT(9)
#define WMI_CHAN_INFO_DFS		BIT(10)
#define WMI_CHAN_INFO_ALLOW_HT		BIT(11)
#define WMI_CHAN_INFO_ALLOW_VHT		BIT(12)
#define WMI_CHAN_INFO_CHAN_CHANGE_CAUSE_CSA	BIT(13)
#define WMI_CHAN_INFO_HALF_RATE		BIT(14)
#define WMI_CHAN_INFO_QUARTER_RATE	BIT(15)
#define WMI_CHAN_INFO_DFS_FREQ2		BIT(16)
#define WMI_CHAN_INFO_ALLOW_HE		BIT(17)
#define WMI_CHAN_INFO_PSC		BIT(18)

#define WMI_CHAN_REG_INFO1_MIN_PWR	GENMASK(7, 0)
#define WMI_CHAN_REG_INFO1_MAX_PWR	GENMASK(15, 8)
#define WMI_CHAN_REG_INFO1_MAX_REG_PWR	GENMASK(23, 16)
#define WMI_CHAN_REG_INFO1_REG_CLS	GENMASK(31, 24)

#define WMI_CHAN_REG_INFO2_ANT_MAX	GENMASK(7, 0)
#define WMI_CHAN_REG_INFO2_MAX_TX_PWR	GENMASK(15, 8)

struct wmi_channel {
	u32 tlv_header;
	u32 mhz;
	u32 band_center_freq1;
	u32 band_center_freq2;
	u32 info;
	u32 reg_info_1;
	u32 reg_info_2;
} __packed;

struct wmi_mgmt_params {
	void *tx_frame;
	u16 frm_len;
	u8 vdev_id;
	u16 chanfreq;
	void *pdata;
	u16 desc_id;
	u8 *macaddr;
};

enum wmi_sta_ps_mode {
	WMI_STA_PS_MODE_DISABLED = 0,
	WMI_STA_PS_MODE_ENABLED = 1,
};

#define WMI_SMPS_MASK_LOWER_16BITS 0xFF
#define WMI_SMPS_MASK_UPPER_3BITS 0x7
#define WMI_SMPS_PARAM_VALUE_SHIFT 29

#define ATH11K_WMI_FW_HANG_ASSERT_TYPE 1
#define ATH11K_WMI_FW_HANG_DELAY 0

/* type, 0:unused 1: ASSERT 2: not respond detect command
 * delay_time_ms, the simulate will delay time
 */

struct wmi_force_fw_hang_cmd {
	u32 tlv_header;
	u32 type;
	u32 delay_time_ms;
};

struct wmi_vdev_set_param_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 param_id;
	u32 param_value;
} __packed;

enum wmi_stats_id {
	WMI_REQUEST_PEER_STAT			= BIT(0),
	WMI_REQUEST_AP_STAT			= BIT(1),
	WMI_REQUEST_PDEV_STAT			= BIT(2),
	WMI_REQUEST_VDEV_STAT			= BIT(3),
	WMI_REQUEST_BCNFLT_STAT			= BIT(4),
	WMI_REQUEST_VDEV_RATE_STAT		= BIT(5),
	WMI_REQUEST_INST_STAT			= BIT(6),
	WMI_REQUEST_MIB_STAT			= BIT(7),
	WMI_REQUEST_RSSI_PER_CHAIN_STAT		= BIT(8),
	WMI_REQUEST_CONGESTION_STAT		= BIT(9),
	WMI_REQUEST_PEER_EXTD_STAT		= BIT(10),
	WMI_REQUEST_BCN_STAT			= BIT(11),
	WMI_REQUEST_BCN_STAT_RESET		= BIT(12),
	WMI_REQUEST_PEER_EXTD2_STAT		= BIT(13),
};

struct wmi_request_stats_cmd {
	u32 tlv_header;
	enum wmi_stats_id stats_id;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 pdev_id;
} __packed;

struct wmi_get_pdev_temperature_cmd {
	u32 tlv_header;
	u32 param;
	u32 pdev_id;
} __packed;

struct wmi_ftm_seg_hdr {
	u32 len;
	u32 msgref;
	u32 segmentinfo;
	u32 pdev_id;
} __packed;

struct wmi_ftm_cmd {
	u32 tlv_header;
	struct wmi_ftm_seg_hdr seg_hdr;
	u8 data[];
} __packed;

struct wmi_ftm_event_msg {
	struct wmi_ftm_seg_hdr seg_hdr;
	u8 data[];
} __packed;

#define WMI_P2P_MAX_NOA_DESCRIPTORS		4

struct wmi_p2p_noa_event {
	u32 vdev_id;
} __packed;

struct ath11k_wmi_p2p_noa_descriptor {
	u32 type_count; /* 255: continuous schedule, 0: reserved */
	u32 duration;  /* Absent period duration in micro seconds */
	u32 interval;   /* Absent period interval in micro seconds */
	u32 start_time; /* 32 bit tsf time when in starts */
} __packed;

#define WMI_P2P_NOA_INFO_CHANGED_FLAG		BIT(0)
#define WMI_P2P_NOA_INFO_INDEX			GENMASK(15, 8)
#define WMI_P2P_NOA_INFO_OPP_PS			BIT(16)
#define WMI_P2P_NOA_INFO_CTWIN_TU		GENMASK(23, 17)
#define WMI_P2P_NOA_INFO_DESC_NUM		GENMASK(31, 24)

struct ath11k_wmi_p2p_noa_info {
	/* Bit 0 - Flag to indicate an update in NOA schedule
	 * Bits 7-1 - Reserved
	 * Bits 15-8 - Index (identifies the instance of NOA sub element)
	 * Bit  16 - Opp PS state of the AP
	 * Bits 23-17 -  Ctwindow in TUs
	 * Bits 31-24 -  Number of NOA descriptors
	 */
	u32 noa_attr;
	struct ath11k_wmi_p2p_noa_descriptor descriptors[WMI_P2P_MAX_NOA_DESCRIPTORS];
} __packed;

#define WMI_BEACON_TX_BUFFER_SIZE	512

#define WMI_EMA_TMPL_IDX_SHIFT            8
#define WMI_EMA_FIRST_TMPL_SHIFT          16
#define WMI_EMA_LAST_TMPL_SHIFT           24

struct wmi_bcn_tmpl_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 tim_ie_offset;
	u32 buf_len;
	u32 csa_switch_count_offset;
	u32 ext_csa_switch_count_offset;
	u32 csa_event_bitmap;
	u32 mbssid_ie_offset;
	u32 esp_ie_offset;
	u32 csc_switch_count_offset;
	u32 csc_event_bitmap;
	u32 mu_edca_ie_offset;
	u32 feature_enable_bitmap;
	u32 ema_params;
} __packed;

struct wmi_p2p_go_set_beacon_ie_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 ie_buf_len;
	u8 tlv[];
} __packed;

struct wmi_key_seq_counter {
	u32 key_seq_counter_l;
	u32 key_seq_counter_h;
} __packed;

struct wmi_vdev_install_key_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 key_idx;
	u32 key_flags;
	u32 key_cipher;
	struct wmi_key_seq_counter key_rsc_counter;
	struct wmi_key_seq_counter key_global_rsc_counter;
	struct wmi_key_seq_counter key_tsc_counter;
	u8 wpi_key_rsc_counter[16];
	u8 wpi_key_tsc_counter[16];
	u32 key_len;
	u32 key_txmic_len;
	u32 key_rxmic_len;
	u32 is_group_key_id_valid;
	u32 group_key_id;

	/* Followed by key_data containing key followed by
	 * tx mic and then rx mic
	 */
} __packed;

struct wmi_vdev_install_key_arg {
	u32 vdev_id;
	const u8 *macaddr;
	u32 key_idx;
	u32 key_flags;
	u32 key_cipher;
	u32 key_len;
	u32 key_txmic_len;
	u32 key_rxmic_len;
	u64 key_rsc_counter;
	const void *key_data;
};

#define WMI_MAX_SUPPORTED_RATES			128
#define WMI_HOST_MAX_HECAP_PHY_SIZE		3
#define WMI_HOST_MAX_HE_RATE_SET		3
#define WMI_HECAP_TXRX_MCS_NSS_IDX_80		0
#define WMI_HECAP_TXRX_MCS_NSS_IDX_160		1
#define WMI_HECAP_TXRX_MCS_NSS_IDX_80_80	2

struct wmi_rate_set_arg {
	u32 num_rates;
	u8 rates[WMI_MAX_SUPPORTED_RATES];
};

struct peer_assoc_params {
	struct wmi_mac_addr peer_macaddr;
	u32 vdev_id;
	u32 peer_new_assoc;
	u32 peer_associd;
	u32 peer_flags;
	u32 peer_caps;
	u32 peer_listen_intval;
	u32 peer_ht_caps;
	u32 peer_max_mpdu;
	u32 peer_mpdu_density;
	u32 peer_rate_caps;
	u32 peer_nss;
	u32 peer_vht_caps;
	u32 peer_phymode;
	u32 peer_ht_info[2];
	struct wmi_rate_set_arg peer_legacy_rates;
	struct wmi_rate_set_arg peer_ht_rates;
	u32 rx_max_rate;
	u32 rx_mcs_set;
	u32 tx_max_rate;
	u32 tx_mcs_set;
	u8 vht_capable;
	u8 min_data_rate;
	u32 tx_max_mcs_nss;
	u32 peer_bw_rxnss_override;
	bool is_pmf_enabled;
	bool is_wme_set;
	bool qos_flag;
	bool apsd_flag;
	bool ht_flag;
	bool bw_40;
	bool bw_80;
	bool bw_160;
	bool stbc_flag;
	bool ldpc_flag;
	bool static_mimops_flag;
	bool dynamic_mimops_flag;
	bool spatial_mux_flag;
	bool vht_flag;
	bool vht_ng_flag;
	bool need_ptk_4_way;
	bool need_gtk_2_way;
	bool auth_flag;
	bool safe_mode_enabled;
	bool amsdu_disable;
	/* Use common structure */
	u8 peer_mac[ETH_ALEN];

	bool he_flag;
	u32 peer_he_cap_macinfo[2];
	u32 peer_he_cap_macinfo_internal;
	u32 peer_he_caps_6ghz;
	u32 peer_he_ops;
	u32 peer_he_cap_phyinfo[WMI_HOST_MAX_HECAP_PHY_SIZE];
	u32 peer_he_mcs_count;
	u32 peer_he_rx_mcs_set[WMI_HOST_MAX_HE_RATE_SET];
	u32 peer_he_tx_mcs_set[WMI_HOST_MAX_HE_RATE_SET];
	bool twt_responder;
	bool twt_requester;
	bool is_assoc;
	struct ath11k_ppe_threshold peer_ppet;
};

struct  wmi_peer_assoc_complete_cmd {
	u32 tlv_header;
	struct wmi_mac_addr peer_macaddr;
	u32 vdev_id;
	u32 peer_new_assoc;
	u32 peer_associd;
	u32 peer_flags;
	u32 peer_caps;
	u32 peer_listen_intval;
	u32 peer_ht_caps;
	u32 peer_max_mpdu;
	u32 peer_mpdu_density;
	u32 peer_rate_caps;
	u32 peer_nss;
	u32 peer_vht_caps;
	u32 peer_phymode;
	u32 peer_ht_info[2];
	u32 num_peer_legacy_rates;
	u32 num_peer_ht_rates;
	u32 peer_bw_rxnss_override;
	struct  wmi_ppe_threshold peer_ppet;
	u32 peer_he_cap_info;
	u32 peer_he_ops;
	u32 peer_he_cap_phy[WMI_MAX_HECAP_PHY_SIZE];
	u32 peer_he_mcs;
	u32 peer_he_cap_info_ext;
	u32 peer_he_cap_info_internal;
	u32 min_data_rate;
	u32 peer_he_caps_6ghz;
} __packed;

struct wmi_stop_scan_cmd {
	u32 tlv_header;
	u32 requestor;
	u32 scan_id;
	u32 req_type;
	u32 vdev_id;
	u32 pdev_id;
};

struct scan_chan_list_params {
	struct list_head list;
	u32 pdev_id;
	u16 nallchans;
	struct channel_param ch_param[];
};

struct wmi_scan_chan_list_cmd {
	u32 tlv_header;
	u32 num_scan_chans;
	u32 flags;
	u32 pdev_id;
} __packed;

struct wmi_scan_prob_req_oui_cmd {
	u32 tlv_header;
	u32 prob_req_oui;
}  __packed;

#define WMI_MGMT_SEND_DOWNLD_LEN	64

#define WMI_TX_PARAMS_DWORD0_POWER		GENMASK(7, 0)
#define WMI_TX_PARAMS_DWORD0_MCS_MASK		GENMASK(19, 8)
#define WMI_TX_PARAMS_DWORD0_NSS_MASK		GENMASK(27, 20)
#define WMI_TX_PARAMS_DWORD0_RETRY_LIMIT	GENMASK(31, 28)

#define WMI_TX_PARAMS_DWORD1_CHAIN_MASK		GENMASK(7, 0)
#define WMI_TX_PARAMS_DWORD1_BW_MASK		GENMASK(14, 8)
#define WMI_TX_PARAMS_DWORD1_PREAMBLE_TYPE	GENMASK(19, 15)
#define WMI_TX_PARAMS_DWORD1_FRAME_TYPE		BIT(20)
#define WMI_TX_PARAMS_DWORD1_RSVD		GENMASK(31, 21)

struct wmi_mgmt_send_params {
	u32 tlv_header;
	u32 tx_params_dword0;
	u32 tx_params_dword1;
};

struct wmi_mgmt_send_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 desc_id;
	u32 chanfreq;
	u32 paddr_lo;
	u32 paddr_hi;
	u32 frame_len;
	u32 buf_len;
	u32 tx_params_valid;

	/* This TLV is followed by struct wmi_mgmt_frame */

	/* Followed by struct wmi_mgmt_send_params */
} __packed;

struct wmi_sta_powersave_mode_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 sta_ps_mode;
};

struct wmi_sta_smps_force_mode_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 forced_mode;
};

struct wmi_sta_smps_param_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 param;
	u32 value;
};

struct wmi_bcn_prb_info {
	u32 tlv_header;
	u32 caps;
	u32 erp;
} __packed;

enum {
	WMI_PDEV_SUSPEND,
	WMI_PDEV_SUSPEND_AND_DISABLE_INTR,
};

struct green_ap_ps_params {
	u32 value;
};

struct wmi_pdev_green_ap_ps_enable_cmd_param {
	u32 tlv_header;
	u32 pdev_id;
	u32 enable;
};

struct ap_ps_params {
	u32 vdev_id;
	u32 param;
	u32 value;
};

struct vdev_set_params {
	u32 if_id;
	u32 param_id;
	u32 param_value;
};

struct stats_request_params {
	u32 stats_id;
	u32 vdev_id;
	u32 pdev_id;
};

struct wmi_set_current_country_params {
	u8 alpha2[3];
};

struct wmi_set_current_country_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 new_alpha2;
} __packed;

enum set_init_cc_type {
	WMI_COUNTRY_INFO_TYPE_ALPHA,
	WMI_COUNTRY_INFO_TYPE_COUNTRY_CODE,
	WMI_COUNTRY_INFO_TYPE_REGDOMAIN,
};

enum set_init_cc_flags {
	INVALID_CC,
	CC_IS_SET,
	REGDMN_IS_SET,
	ALPHA_IS_SET,
};

struct wmi_init_country_params {
	union {
		u16 country_code;
		u16 regdom_id;
		u8 alpha2[3];
	} cc_info;
	enum set_init_cc_flags flags;
};

struct wmi_init_country_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 init_cc_type;
	union {
		u32 country_code;
		u32 regdom_id;
		u32 alpha2;
	} cc_info;
} __packed;

struct wmi_11d_scan_start_params {
	u32 vdev_id;
	u32 scan_period_msec;
	u32 start_interval_msec;
};

struct wmi_11d_scan_start_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 scan_period_msec;
	u32 start_interval_msec;
} __packed;

struct wmi_11d_scan_stop_cmd {
	u32 tlv_header;
	u32 vdev_id;
} __packed;

struct wmi_11d_new_cc_ev {
	u32 new_alpha2;
} __packed;

#define THERMAL_LEVELS  1
struct tt_level_config {
	u32 tmplwm;
	u32 tmphwm;
	u32 dcoffpercent;
	u32 priority;
};

struct thermal_mitigation_params {
	u32 pdev_id;
	u32 enable;
	u32 dc;
	u32 dc_per_event;
	struct tt_level_config levelconf[THERMAL_LEVELS];
};

struct wmi_therm_throt_config_request_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 enable;
	u32 dc;
	u32 dc_per_event;
	u32 therm_throt_levels;
} __packed;

struct wmi_therm_throt_level_config_info {
	u32 tlv_header;
	u32 temp_lwm;
	u32 temp_hwm;
	u32 dc_off_percent;
	u32 prio;
} __packed;

struct wmi_delba_send_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 tid;
	u32 initiator;
	u32 reasoncode;
} __packed;

struct wmi_addba_setresponse_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 tid;
	u32 statuscode;
} __packed;

struct wmi_addba_send_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 tid;
	u32 buffersize;
} __packed;

struct wmi_addba_clear_resp_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
} __packed;

struct wmi_pdev_pktlog_filter_info {
	u32 tlv_header;
	struct wmi_mac_addr peer_macaddr;
} __packed;

struct wmi_pdev_pktlog_filter_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 enable;
	u32 filter_type;
	u32 num_mac;
} __packed;

enum ath11k_wmi_pktlog_enable {
	ATH11K_WMI_PKTLOG_ENABLE_AUTO  = 0,
	ATH11K_WMI_PKTLOG_ENABLE_FORCE = 1,
};

struct wmi_pktlog_enable_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 evlist; /* WMI_PKTLOG_EVENT */
	u32 enable;
} __packed;

struct wmi_pktlog_disable_cmd {
	u32 tlv_header;
	u32 pdev_id;
} __packed;

#define DFS_PHYERR_UNIT_TEST_CMD 0
#define DFS_UNIT_TEST_MODULE	0x2b
#define DFS_UNIT_TEST_TOKEN	0xAA

enum dfs_test_args_idx {
	DFS_TEST_CMDID = 0,
	DFS_TEST_PDEV_ID,
	DFS_TEST_RADAR_PARAM,
	DFS_MAX_TEST_ARGS,
};

struct wmi_dfs_unit_test_arg {
	u32 cmd_id;
	u32 pdev_id;
	u32 radar_param;
};

struct wmi_unit_test_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 module_id;
	u32 num_args;
	u32 diag_token;
	/* Followed by test args*/
} __packed;

#define MAX_SUPPORTED_RATES 128

struct beacon_tmpl_params {
	u8 vdev_id;
	u32 tim_ie_offset;
	u32 tmpl_len;
	u32 tmpl_len_aligned;
	u32 csa_switch_count_offset;
	u32 ext_csa_switch_count_offset;
	u8 *frm;
};

struct wmi_rate_set {
	u32 num_rates;
	u32 rates[(MAX_SUPPORTED_RATES / 4) + 1];
};

struct wmi_vht_rate_set {
	u32 tlv_header;
	u32 rx_max_rate;
	u32 rx_mcs_set;
	u32 tx_max_rate;
	u32 tx_mcs_set;
	u32 tx_max_mcs_nss;
} __packed;

struct wmi_he_rate_set {
	u32 tlv_header;

	/* MCS at which the peer can receive */
	u32 rx_mcs_set;

	/* MCS at which the peer can transmit */
	u32 tx_mcs_set;
} __packed;

#define MAX_REG_RULES 10
#define REG_ALPHA2_LEN 2
#define MAX_6GHZ_REG_RULES 5

enum wmi_start_event_param {
	WMI_VDEV_START_RESP_EVENT = 0,
	WMI_VDEV_RESTART_RESP_EVENT,
};

struct wmi_vdev_start_resp_event {
	u32 vdev_id;
	u32 requestor_id;
	enum wmi_start_event_param resp_type;
	u32 status;
	u32 chain_mask;
	u32 smps_mode;
	union {
		u32 mac_id;
		u32 pdev_id;
	};
	u32 cfgd_tx_streams;
	u32 cfgd_rx_streams;
	s32 max_allowed_tx_power;
} __packed;

/* VDEV start response status codes */
enum wmi_vdev_start_resp_status_code {
	WMI_VDEV_START_RESPONSE_STATUS_SUCCESS = 0,
	WMI_VDEV_START_RESPONSE_INVALID_VDEVID = 1,
	WMI_VDEV_START_RESPONSE_NOT_SUPPORTED = 2,
	WMI_VDEV_START_RESPONSE_DFS_VIOLATION = 3,
	WMI_VDEV_START_RESPONSE_INVALID_REGDOMAIN = 4,
};

/* Regaulatory Rule Flags Passed by FW */
#define REGULATORY_CHAN_DISABLED     BIT(0)
#define REGULATORY_CHAN_NO_IR        BIT(1)
#define REGULATORY_CHAN_RADAR        BIT(3)
#define REGULATORY_CHAN_NO_OFDM      BIT(6)
#define REGULATORY_CHAN_INDOOR_ONLY  BIT(9)

#define REGULATORY_CHAN_NO_HT40      BIT(4)
#define REGULATORY_CHAN_NO_80MHZ     BIT(7)
#define REGULATORY_CHAN_NO_160MHZ    BIT(8)
#define REGULATORY_CHAN_NO_20MHZ     BIT(11)
#define REGULATORY_CHAN_NO_10MHZ     BIT(12)

enum wmi_reg_chan_list_cmd_type {
	WMI_REG_CHAN_LIST_CC_ID = 0,
	WMI_REG_CHAN_LIST_CC_EXT_ID = 1,
};

enum wmi_reg_cc_setting_code {
	WMI_REG_SET_CC_STATUS_PASS = 0,
	WMI_REG_CURRENT_ALPHA2_NOT_FOUND = 1,
	WMI_REG_INIT_ALPHA2_NOT_FOUND = 2,
	WMI_REG_SET_CC_CHANGE_NOT_ALLOWED = 3,
	WMI_REG_SET_CC_STATUS_NO_MEMORY = 4,
	WMI_REG_SET_CC_STATUS_FAIL = 5,

	/* add new setting code above, update in
	 * @enum cc_setting_code as well.
	 * Also handle it in ath11k_wmi_cc_setting_code_to_reg()
	 */
};

enum cc_setting_code {
	REG_SET_CC_STATUS_PASS = 0,
	REG_CURRENT_ALPHA2_NOT_FOUND = 1,
	REG_INIT_ALPHA2_NOT_FOUND = 2,
	REG_SET_CC_CHANGE_NOT_ALLOWED = 3,
	REG_SET_CC_STATUS_NO_MEMORY = 4,
	REG_SET_CC_STATUS_FAIL = 5,

	/* add new setting code above, update in
	 * @enum wmi_reg_cc_setting_code as well.
	 * Also handle it in ath11k_cc_status_to_str()
	 */
};

static inline enum cc_setting_code
ath11k_wmi_cc_setting_code_to_reg(enum wmi_reg_cc_setting_code status_code)
{
	switch (status_code) {
	case WMI_REG_SET_CC_STATUS_PASS:
		return REG_SET_CC_STATUS_PASS;
	case WMI_REG_CURRENT_ALPHA2_NOT_FOUND:
		return REG_CURRENT_ALPHA2_NOT_FOUND;
	case WMI_REG_INIT_ALPHA2_NOT_FOUND:
		return REG_INIT_ALPHA2_NOT_FOUND;
	case WMI_REG_SET_CC_CHANGE_NOT_ALLOWED:
		return REG_SET_CC_CHANGE_NOT_ALLOWED;
	case WMI_REG_SET_CC_STATUS_NO_MEMORY:
		return REG_SET_CC_STATUS_NO_MEMORY;
	case WMI_REG_SET_CC_STATUS_FAIL:
		return REG_SET_CC_STATUS_FAIL;
	}

	return REG_SET_CC_STATUS_FAIL;
}

static inline const char *ath11k_cc_status_to_str(enum cc_setting_code code)
{
	switch (code) {
	case REG_SET_CC_STATUS_PASS:
		return "REG_SET_CC_STATUS_PASS";
	case REG_CURRENT_ALPHA2_NOT_FOUND:
		return "REG_CURRENT_ALPHA2_NOT_FOUND";
	case REG_INIT_ALPHA2_NOT_FOUND:
		return "REG_INIT_ALPHA2_NOT_FOUND";
	case REG_SET_CC_CHANGE_NOT_ALLOWED:
		return "REG_SET_CC_CHANGE_NOT_ALLOWED";
	case REG_SET_CC_STATUS_NO_MEMORY:
		return "REG_SET_CC_STATUS_NO_MEMORY";
	case REG_SET_CC_STATUS_FAIL:
		return "REG_SET_CC_STATUS_FAIL";
	}

	return "Unknown CC status";
}

enum wmi_reg_6ghz_ap_type {
	WMI_REG_INDOOR_AP = 0,
	WMI_REG_STANDARD_POWER_AP = 1,
	WMI_REG_VERY_LOW_POWER_AP = 2,

	/* add AP type above, handle in ath11k_6ghz_ap_type_to_str()
	 */
	WMI_REG_CURRENT_MAX_AP_TYPE,
	WMI_REG_MAX_AP_TYPE = 7,
};

static inline const char *
ath11k_6ghz_ap_type_to_str(enum wmi_reg_6ghz_ap_type type)
{
	switch (type) {
	case WMI_REG_INDOOR_AP:
		return "INDOOR AP";
	case WMI_REG_STANDARD_POWER_AP:
		return "STANDARD POWER AP";
	case WMI_REG_VERY_LOW_POWER_AP:
		return "VERY LOW POWER AP";
	case WMI_REG_CURRENT_MAX_AP_TYPE:
		return "CURRENT_MAX_AP_TYPE";
	case WMI_REG_MAX_AP_TYPE:
		return "MAX_AP_TYPE";
	}

	return "unknown 6 GHz AP type";
}

enum wmi_reg_6ghz_client_type {
	WMI_REG_DEFAULT_CLIENT = 0,
	WMI_REG_SUBORDINATE_CLIENT = 1,
	WMI_REG_MAX_CLIENT_TYPE = 2,

	/* add client type above, handle it in
	 * ath11k_6ghz_client_type_to_str()
	 */
};

static inline const char *
ath11k_6ghz_client_type_to_str(enum wmi_reg_6ghz_client_type type)
{
	switch (type) {
	case WMI_REG_DEFAULT_CLIENT:
		return "DEFAULT CLIENT";
	case WMI_REG_SUBORDINATE_CLIENT:
		return "SUBORDINATE CLIENT";
	case WMI_REG_MAX_CLIENT_TYPE:
		return "MAX_CLIENT_TYPE";
	}

	return "unknown 6 GHz client type";
}

enum reg_subdomains_6ghz {
	EMPTY_6GHZ = 0x0,
	FCC1_CLIENT_LPI_REGULAR_6GHZ = 0x01,
	FCC1_CLIENT_SP_6GHZ = 0x02,
	FCC1_AP_LPI_6GHZ = 0x03,
	FCC1_CLIENT_LPI_SUBORDINATE = FCC1_AP_LPI_6GHZ,
	FCC1_AP_SP_6GHZ = 0x04,
	ETSI1_LPI_6GHZ = 0x10,
	ETSI1_VLP_6GHZ = 0x11,
	ETSI2_LPI_6GHZ = 0x12,
	ETSI2_VLP_6GHZ = 0x13,
	APL1_LPI_6GHZ = 0x20,
	APL1_VLP_6GHZ = 0x21,

	/* add sub-domain above, handle it in
	 * ath11k_sub_reg_6ghz_to_str()
	 */
};

static inline const char *
ath11k_sub_reg_6ghz_to_str(enum reg_subdomains_6ghz sub_id)
{
	switch (sub_id) {
	case EMPTY_6GHZ:
		return "N/A";
	case FCC1_CLIENT_LPI_REGULAR_6GHZ:
		return "FCC1_CLIENT_LPI_REGULAR_6GHZ";
	case FCC1_CLIENT_SP_6GHZ:
		return "FCC1_CLIENT_SP_6GHZ";
	case FCC1_AP_LPI_6GHZ:
		return "FCC1_AP_LPI_6GHZ/FCC1_CLIENT_LPI_SUBORDINATE";
	case FCC1_AP_SP_6GHZ:
		return "FCC1_AP_SP_6GHZ";
	case ETSI1_LPI_6GHZ:
		return "ETSI1_LPI_6GHZ";
	case ETSI1_VLP_6GHZ:
		return "ETSI1_VLP_6GHZ";
	case ETSI2_LPI_6GHZ:
		return "ETSI2_LPI_6GHZ";
	case ETSI2_VLP_6GHZ:
		return "ETSI2_VLP_6GHZ";
	case APL1_LPI_6GHZ:
		return "APL1_LPI_6GHZ";
	case APL1_VLP_6GHZ:
		return "APL1_VLP_6GHZ";
	}

	return "unknown sub reg id";
}

enum reg_super_domain_6ghz {
	FCC1_6GHZ = 0x01,
	ETSI1_6GHZ = 0x02,
	ETSI2_6GHZ = 0x03,
	APL1_6GHZ = 0x04,
	FCC1_6GHZ_CL = 0x05,

	/* add super domain above, handle it in
	 * ath11k_super_reg_6ghz_to_str()
	 */
};

static inline const char *
ath11k_super_reg_6ghz_to_str(enum reg_super_domain_6ghz domain_id)
{
	switch (domain_id) {
	case FCC1_6GHZ:
		return "FCC1_6GHZ";
	case ETSI1_6GHZ:
		return "ETSI1_6GHZ";
	case ETSI2_6GHZ:
		return "ETSI2_6GHZ";
	case APL1_6GHZ:
		return "APL1_6GHZ";
	case FCC1_6GHZ_CL:
		return "FCC1_6GHZ_CL";
	}

	return "unknown domain id";
}

struct cur_reg_rule {
	u16 start_freq;
	u16 end_freq;
	u16 max_bw;
	u8 reg_power;
	u8 ant_gain;
	u16 flags;
	bool psd_flag;
	s8 psd_eirp;
};

struct cur_regulatory_info {
	enum cc_setting_code status_code;
	u8 num_phy;
	u8 phy_id;
	u16 reg_dmn_pair;
	u16 ctry_code;
	u8 alpha2[REG_ALPHA2_LEN + 1];
	u32 dfs_region;
	u32 phybitmap;
	u32 min_bw_2ghz;
	u32 max_bw_2ghz;
	u32 min_bw_5ghz;
	u32 max_bw_5ghz;
	u32 num_2ghz_reg_rules;
	u32 num_5ghz_reg_rules;
	struct cur_reg_rule *reg_rules_2ghz_ptr;
	struct cur_reg_rule *reg_rules_5ghz_ptr;
	bool is_ext_reg_event;
	enum wmi_reg_6ghz_client_type client_type;
	bool rnr_tpe_usable;
	bool unspecified_ap_usable;
	u8 domain_code_6ghz_ap[WMI_REG_CURRENT_MAX_AP_TYPE];
	u8 domain_code_6ghz_client[WMI_REG_CURRENT_MAX_AP_TYPE][WMI_REG_MAX_CLIENT_TYPE];
	u32 domain_code_6ghz_super_id;
	u32 min_bw_6ghz_ap[WMI_REG_CURRENT_MAX_AP_TYPE];
	u32 max_bw_6ghz_ap[WMI_REG_CURRENT_MAX_AP_TYPE];
	u32 min_bw_6ghz_client[WMI_REG_CURRENT_MAX_AP_TYPE][WMI_REG_MAX_CLIENT_TYPE];
	u32 max_bw_6ghz_client[WMI_REG_CURRENT_MAX_AP_TYPE][WMI_REG_MAX_CLIENT_TYPE];
	u32 num_6ghz_rules_ap[WMI_REG_CURRENT_MAX_AP_TYPE];
	u32 num_6ghz_rules_client[WMI_REG_CURRENT_MAX_AP_TYPE][WMI_REG_MAX_CLIENT_TYPE];
	struct cur_reg_rule *reg_rules_6ghz_ap_ptr[WMI_REG_CURRENT_MAX_AP_TYPE];
	struct cur_reg_rule *reg_rules_6ghz_client_ptr
		[WMI_REG_CURRENT_MAX_AP_TYPE][WMI_REG_MAX_CLIENT_TYPE];
};

struct wmi_reg_chan_list_cc_event {
	u32 status_code;
	u32 phy_id;
	u32 alpha2;
	u32 num_phy;
	u32 country_id;
	u32 domain_code;
	u32 dfs_region;
	u32 phybitmap;
	u32 min_bw_2ghz;
	u32 max_bw_2ghz;
	u32 min_bw_5ghz;
	u32 max_bw_5ghz;
	u32 num_2ghz_reg_rules;
	u32 num_5ghz_reg_rules;
} __packed;

struct wmi_regulatory_rule_struct {
	u32  tlv_header;
	u32  freq_info;
	u32  bw_pwr_info;
	u32  flag_info;
};

#define WMI_REG_CLIENT_MAX 4

struct wmi_reg_chan_list_cc_ext_event {
	u32 status_code;
	u32 phy_id;
	u32 alpha2;
	u32 num_phy;
	u32 country_id;
	u32 domain_code;
	u32 dfs_region;
	u32 phybitmap;
	u32 min_bw_2ghz;
	u32 max_bw_2ghz;
	u32 min_bw_5ghz;
	u32 max_bw_5ghz;
	u32 num_2ghz_reg_rules;
	u32 num_5ghz_reg_rules;
	u32 client_type;
	u32 rnr_tpe_usable;
	u32 unspecified_ap_usable;
	u32 domain_code_6ghz_ap_lpi;
	u32 domain_code_6ghz_ap_sp;
	u32 domain_code_6ghz_ap_vlp;
	u32 domain_code_6ghz_client_lpi[WMI_REG_CLIENT_MAX];
	u32 domain_code_6ghz_client_sp[WMI_REG_CLIENT_MAX];
	u32 domain_code_6ghz_client_vlp[WMI_REG_CLIENT_MAX];
	u32 domain_code_6ghz_super_id;
	u32 min_bw_6ghz_ap_sp;
	u32 max_bw_6ghz_ap_sp;
	u32 min_bw_6ghz_ap_lpi;
	u32 max_bw_6ghz_ap_lpi;
	u32 min_bw_6ghz_ap_vlp;
	u32 max_bw_6ghz_ap_vlp;
	u32 min_bw_6ghz_client_sp[WMI_REG_CLIENT_MAX];
	u32 max_bw_6ghz_client_sp[WMI_REG_CLIENT_MAX];
	u32 min_bw_6ghz_client_lpi[WMI_REG_CLIENT_MAX];
	u32 max_bw_6ghz_client_lpi[WMI_REG_CLIENT_MAX];
	u32 min_bw_6ghz_client_vlp[WMI_REG_CLIENT_MAX];
	u32 max_bw_6ghz_client_vlp[WMI_REG_CLIENT_MAX];
	u32 num_6ghz_reg_rules_ap_sp;
	u32 num_6ghz_reg_rules_ap_lpi;
	u32 num_6ghz_reg_rules_ap_vlp;
	u32 num_6ghz_reg_rules_client_sp[WMI_REG_CLIENT_MAX];
	u32 num_6ghz_reg_rules_client_lpi[WMI_REG_CLIENT_MAX];
	u32 num_6ghz_reg_rules_client_vlp[WMI_REG_CLIENT_MAX];
} __packed;

struct wmi_regulatory_ext_rule {
	u32 tlv_header;
	u32 freq_info;
	u32 bw_pwr_info;
	u32 flag_info;
	u32 psd_power_info;
} __packed;

struct wmi_vdev_delete_resp_event {
	u32 vdev_id;
} __packed;

struct wmi_peer_delete_resp_event {
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
} __packed;

struct wmi_bcn_tx_status_event {
	u32 vdev_id;
	u32 tx_status;
} __packed;

struct wmi_vdev_stopped_event {
	u32 vdev_id;
} __packed;

struct wmi_pdev_bss_chan_info_event {
	u32 freq;	/* Units in MHz */
	u32 noise_floor;	/* units are dBm */
	/* rx clear - how often the channel was unused */
	u32 rx_clear_count_low;
	u32 rx_clear_count_high;
	/* cycle count - elapsed time during measured period, in clock ticks */
	u32 cycle_count_low;
	u32 cycle_count_high;
	/* tx cycle count - elapsed time spent in tx, in clock ticks */
	u32 tx_cycle_count_low;
	u32 tx_cycle_count_high;
	/* rx cycle count - elapsed time spent in rx, in clock ticks */
	u32 rx_cycle_count_low;
	u32 rx_cycle_count_high;
	/*rx_cycle cnt for my bss in 64bits format */
	u32 rx_bss_cycle_count_low;
	u32 rx_bss_cycle_count_high;
	u32 pdev_id;
} __packed;

#define WMI_VDEV_INSTALL_KEY_COMPL_STATUS_SUCCESS 0

struct wmi_vdev_install_key_compl_event {
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 key_idx;
	u32 key_flags;
	u32 status;
} __packed;

struct wmi_vdev_install_key_complete_arg {
	u32 vdev_id;
	const u8 *macaddr;
	u32 key_idx;
	u32 key_flags;
	u32 status;
};

struct wmi_peer_assoc_conf_event {
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
} __packed;

struct wmi_peer_assoc_conf_arg {
	u32 vdev_id;
	const u8 *macaddr;
};

struct wmi_fils_discovery_event {
	u32 vdev_id;
	u32 fils_tt;
	u32 tbtt;
} __packed;

struct wmi_probe_resp_tx_status_event {
	u32 vdev_id;
	u32 tx_status;
} __packed;

/*
 * PDEV statistics
 */
struct wmi_pdev_stats_base {
	s32 chan_nf;
	u32 tx_frame_count; /* Cycles spent transmitting frames */
	u32 rx_frame_count; /* Cycles spent receiving frames */
	u32 rx_clear_count; /* Total channel busy time, evidently */
	u32 cycle_count; /* Total on-channel time */
	u32 phy_err_count;
	u32 chan_tx_pwr;
} __packed;

struct wmi_pdev_stats_extra {
	u32 ack_rx_bad;
	u32 rts_bad;
	u32 rts_good;
	u32 fcs_bad;
	u32 no_beacons;
	u32 mib_int_count;
} __packed;

struct wmi_pdev_stats_tx {
	/* Num HTT cookies queued to dispatch list */
	s32 comp_queued;

	/* Num HTT cookies dispatched */
	s32 comp_delivered;

	/* Num MSDU queued to WAL */
	s32 msdu_enqued;

	/* Num MPDU queue to WAL */
	s32 mpdu_enqued;

	/* Num MSDUs dropped by WMM limit */
	s32 wmm_drop;

	/* Num Local frames queued */
	s32 local_enqued;

	/* Num Local frames done */
	s32 local_freed;

	/* Num queued to HW */
	s32 hw_queued;

	/* Num PPDU reaped from HW */
	s32 hw_reaped;

	/* Num underruns */
	s32 underrun;

	/* Num hw paused */
	u32 hw_paused;

	/* Num PPDUs cleaned up in TX abort */
	s32 tx_abort;

	/* Num MPDUs requeued by SW */
	s32 mpdus_requeued;

	/* excessive retries */
	u32 tx_ko;

	u32 tx_xretry;

	/* data hw rate code */
	u32 data_rc;

	/* Scheduler self triggers */
	u32 self_triggers;

	/* frames dropped due to excessive sw retries */
	u32 sw_retry_failure;

	/* illegal rate phy errors  */
	u32 illgl_rate_phy_err;

	/* wal pdev continuous xretry */
	u32 pdev_cont_xretry;

	/* wal pdev tx timeouts */
	u32 pdev_tx_timeout;

	/* wal pdev resets  */
	u32 pdev_resets;

	/* frames dropped due to non-availability of stateless TIDs */
	u32 stateless_tid_alloc_failure;

	/* PhY/BB underrun */
	u32 phy_underrun;

	/* MPDU is more than txop limit */
	u32 txop_ovf;

	/* Num sequences posted */
	u32 seq_posted;

	/* Num sequences failed in queueing */
	u32 seq_failed_queueing;

	/* Num sequences completed */
	u32 seq_completed;

	/* Num sequences restarted */
	u32 seq_restarted;

	/* Num of MU sequences posted */
	u32 mu_seq_posted;

	/* Num MPDUs flushed by SW, HWPAUSED, SW TXABORT
	 * (Reset,channel change)
	 */
	s32 mpdus_sw_flush;

	/* Num MPDUs filtered by HW, all filter condition (TTL expired) */
	s32 mpdus_hw_filter;

	/* Num MPDUs truncated by PDG (TXOP, TBTT,
	 * PPDU_duration based on rate, dyn_bw)
	 */
	s32 mpdus_truncated;

	/* Num MPDUs that was tried but didn't receive ACK or BA */
	s32 mpdus_ack_failed;

	/* Num MPDUs that was dropped du to expiry. */
	s32 mpdus_expired;
} __packed;

struct wmi_pdev_stats_rx {
	/* Cnts any change in ring routing mid-ppdu */
	s32 mid_ppdu_route_change;

	/* Total number of statuses processed */
	s32 status_rcvd;

	/* Extra frags on rings 0-3 */
	s32 r0_frags;
	s32 r1_frags;
	s32 r2_frags;
	s32 r3_frags;

	/* MSDUs / MPDUs delivered to HTT */
	s32 htt_msdus;
	s32 htt_mpdus;

	/* MSDUs / MPDUs delivered to local stack */
	s32 loc_msdus;
	s32 loc_mpdus;

	/* AMSDUs that have more MSDUs than the status ring size */
	s32 oversize_amsdu;

	/* Number of PHY errors */
	s32 phy_errs;

	/* Number of PHY errors drops */
	s32 phy_err_drop;

	/* Number of mpdu errors - FCS, MIC, ENC etc. */
	s32 mpdu_errs;

	/* Num overflow errors */
	s32 rx_ovfl_errs;
} __packed;

struct wmi_pdev_stats {
	struct wmi_pdev_stats_base base;
	struct wmi_pdev_stats_tx tx;
	struct wmi_pdev_stats_rx rx;
} __packed;

#define WLAN_MAX_AC 4
#define MAX_TX_RATE_VALUES 10
#define MAX_TX_RATE_VALUES 10

struct wmi_vdev_stats {
	u32 vdev_id;
	u32 beacon_snr;
	u32 data_snr;
	u32 num_tx_frames[WLAN_MAX_AC];
	u32 num_rx_frames;
	u32 num_tx_frames_retries[WLAN_MAX_AC];
	u32 num_tx_frames_failures[WLAN_MAX_AC];
	u32 num_rts_fail;
	u32 num_rts_success;
	u32 num_rx_err;
	u32 num_rx_discard;
	u32 num_tx_not_acked;
	u32 tx_rate_history[MAX_TX_RATE_VALUES];
	u32 beacon_rssi_history[MAX_TX_RATE_VALUES];
} __packed;

struct wmi_bcn_stats {
	u32 vdev_id;
	u32 tx_bcn_succ_cnt;
	u32 tx_bcn_outage_cnt;
} __packed;

struct wmi_stats_event {
	u32 stats_id;
	u32 num_pdev_stats;
	u32 num_vdev_stats;
	u32 num_peer_stats;
	u32 num_bcnflt_stats;
	u32 num_chan_stats;
	u32 num_mib_stats;
	u32 pdev_id;
	u32 num_bcn_stats;
	u32 num_peer_extd_stats;
	u32 num_peer_extd2_stats;
} __packed;

struct wmi_rssi_stats {
	u32 vdev_id;
	u32 rssi_avg_beacon[WMI_MAX_CHAINS];
	u32 rssi_avg_data[WMI_MAX_CHAINS];
	struct wmi_mac_addr peer_macaddr;
} __packed;

struct wmi_per_chain_rssi_stats {
	u32 num_per_chain_rssi_stats;
} __packed;

struct wmi_pdev_ctl_failsafe_chk_event {
	u32 pdev_id;
	u32 ctl_failsafe_status;
} __packed;

struct wmi_pdev_csa_switch_ev {
	u32 pdev_id;
	u32 current_switch_count;
	u32 num_vdevs;
} __packed;

struct wmi_pdev_radar_ev {
	u32 pdev_id;
	u32 detection_mode;
	u32 chan_freq;
	u32 chan_width;
	u32 detector_id;
	u32 segment_id;
	u32 timestamp;
	u32 is_chirp;
	s32 freq_offset;
	s32 sidx;
} __packed;

struct wmi_pdev_temperature_event {
	/* temperature value in Celsius degree */
	s32 temp;
	u32 pdev_id;
} __packed;

#define WMI_RX_STATUS_OK			0x00
#define WMI_RX_STATUS_ERR_CRC			0x01
#define WMI_RX_STATUS_ERR_DECRYPT		0x08
#define WMI_RX_STATUS_ERR_MIC			0x10
#define WMI_RX_STATUS_ERR_KEY_CACHE_MISS	0x20

#define WLAN_MGMT_TXRX_HOST_MAX_ANTENNA 4

struct mgmt_rx_event_params {
	u32 chan_freq;
	u32 channel;
	u32 snr;
	u8 rssi_ctl[WLAN_MGMT_TXRX_HOST_MAX_ANTENNA];
	u32 rate;
	enum wmi_phy_mode phy_mode;
	u32 buf_len;
	int status;
	u32 flags;
	int rssi;
	u32 tsf_delta;
	u8 pdev_id;
};

#define ATH_MAX_ANTENNA 4

struct wmi_mgmt_rx_hdr {
	u32 channel;
	u32 snr;
	u32 rate;
	u32 phy_mode;
	u32 buf_len;
	u32 status;
	u32 rssi_ctl[ATH_MAX_ANTENNA];
	u32 flags;
	int rssi;
	u32 tsf_delta;
	u32 rx_tsf_l32;
	u32 rx_tsf_u32;
	u32 pdev_id;
	u32 chan_freq;
} __packed;

#define MAX_ANTENNA_EIGHT 8

struct wmi_rssi_ctl_ext {
	u32 tlv_header;
	u32 rssi_ctl_ext[MAX_ANTENNA_EIGHT - ATH_MAX_ANTENNA];
};

struct wmi_mgmt_tx_compl_event {
	u32 desc_id;
	u32 status;
	u32 pdev_id;
	u32 ppdu_id;
	u32 ack_rssi;
} __packed;

struct wmi_scan_event {
	u32 event_type; /* %WMI_SCAN_EVENT_ */
	u32 reason; /* %WMI_SCAN_REASON_ */
	u32 channel_freq; /* only valid for WMI_SCAN_EVENT_FOREIGN_CHANNEL */
	u32 scan_req_id;
	u32 scan_id;
	u32 vdev_id;
	/* TSF Timestamp when the scan event (%WMI_SCAN_EVENT_) is completed
	 * In case of AP it is TSF of the AP vdev
	 * In case of STA connected state, this is the TSF of the AP
	 * In case of STA not connected, it will be the free running HW timer
	 */
	u32 tsf_timestamp;
} __packed;

struct wmi_peer_sta_kickout_arg {
	const u8 *mac_addr;
};

struct wmi_peer_sta_kickout_event {
	struct wmi_mac_addr peer_macaddr;
} __packed;

enum wmi_roam_reason {
	WMI_ROAM_REASON_BETTER_AP = 1,
	WMI_ROAM_REASON_BEACON_MISS = 2,
	WMI_ROAM_REASON_LOW_RSSI = 3,
	WMI_ROAM_REASON_SUITABLE_AP_FOUND = 4,
	WMI_ROAM_REASON_HO_FAILED = 5,

	/* keep last */
	WMI_ROAM_REASON_MAX,
};

struct wmi_roam_event {
	u32 vdev_id;
	u32 reason;
	u32 rssi;
} __packed;

#define WMI_CHAN_INFO_START_RESP 0
#define WMI_CHAN_INFO_END_RESP 1

struct wmi_chan_info_event {
	u32 err_code;
	u32 freq;
	u32 cmd_flags;
	u32 noise_floor;
	u32 rx_clear_count;
	u32 cycle_count;
	u32 chan_tx_pwr_range;
	u32 chan_tx_pwr_tp;
	u32 rx_frame_count;
	u32 my_bss_rx_cycle_count;
	u32 rx_11b_mode_data_duration;
	u32 tx_frame_cnt;
	u32 mac_clk_mhz;
	u32 vdev_id;
} __packed;

struct ath11k_targ_cap {
	u32 phy_capability;
	u32 max_frag_entry;
	u32 num_rf_chains;
	u32 ht_cap_info;
	u32 vht_cap_info;
	u32 vht_supp_mcs;
	u32 hw_min_tx_power;
	u32 hw_max_tx_power;
	u32 sys_cap_info;
	u32 min_pkt_size_enable;
	u32 max_bcn_ie_size;
	u32 max_num_scan_channels;
	u32 max_supported_macs;
	u32 wmi_fw_sub_feat_caps;
	u32 txrx_chainmask;
	u32 default_dbs_hw_mode_index;
	u32 num_msdu_desc;
};

enum wmi_vdev_type {
	WMI_VDEV_TYPE_UNSPEC =  0,
	WMI_VDEV_TYPE_AP      = 1,
	WMI_VDEV_TYPE_STA     = 2,
	WMI_VDEV_TYPE_IBSS    = 3,
	WMI_VDEV_TYPE_MONITOR = 4,
};

enum wmi_vdev_subtype {
	WMI_VDEV_SUBTYPE_NONE,
	WMI_VDEV_SUBTYPE_P2P_DEVICE,
	WMI_VDEV_SUBTYPE_P2P_CLIENT,
	WMI_VDEV_SUBTYPE_P2P_GO,
	WMI_VDEV_SUBTYPE_PROXY_STA,
	WMI_VDEV_SUBTYPE_MESH_NON_11S,
	WMI_VDEV_SUBTYPE_MESH_11S,
};

enum wmi_sta_powersave_param {
	WMI_STA_PS_PARAM_RX_WAKE_POLICY = 0,
	WMI_STA_PS_PARAM_TX_WAKE_THRESHOLD = 1,
	WMI_STA_PS_PARAM_PSPOLL_COUNT = 2,
	WMI_STA_PS_PARAM_INACTIVITY_TIME = 3,
	WMI_STA_PS_PARAM_UAPSD = 4,
};

#define WMI_UAPSD_AC_TYPE_DELI 0
#define WMI_UAPSD_AC_TYPE_TRIG 1

#define WMI_UAPSD_AC_BIT_MASK(ac, type) \
	((type ==  WMI_UAPSD_AC_TYPE_DELI) ? \
	 (1 << (ac << 1)) : (1 << ((ac << 1) + 1)))

enum wmi_sta_ps_param_uapsd {
	WMI_STA_PS_UAPSD_AC0_DELIVERY_EN = (1 << 0),
	WMI_STA_PS_UAPSD_AC0_TRIGGER_EN  = (1 << 1),
	WMI_STA_PS_UAPSD_AC1_DELIVERY_EN = (1 << 2),
	WMI_STA_PS_UAPSD_AC1_TRIGGER_EN  = (1 << 3),
	WMI_STA_PS_UAPSD_AC2_DELIVERY_EN = (1 << 4),
	WMI_STA_PS_UAPSD_AC2_TRIGGER_EN  = (1 << 5),
	WMI_STA_PS_UAPSD_AC3_DELIVERY_EN = (1 << 6),
	WMI_STA_PS_UAPSD_AC3_TRIGGER_EN  = (1 << 7),
};

#define WMI_STA_UAPSD_MAX_INTERVAL_MSEC UINT_MAX

struct wmi_sta_uapsd_auto_trig_param {
	u32 wmm_ac;
	u32 user_priority;
	u32 service_interval;
	u32 suspend_interval;
	u32 delay_interval;
};

struct wmi_sta_uapsd_auto_trig_cmd_fixed_param {
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 num_ac;
};

struct wmi_sta_uapsd_auto_trig_arg {
	u32 wmm_ac;
	u32 user_priority;
	u32 service_interval;
	u32 suspend_interval;
	u32 delay_interval;
};

enum wmi_sta_ps_param_tx_wake_threshold {
	WMI_STA_PS_TX_WAKE_THRESHOLD_NEVER = 0,
	WMI_STA_PS_TX_WAKE_THRESHOLD_ALWAYS = 1,

	/* Values greater than one indicate that many TX attempts per beacon
	 * interval before the STA will wake up
	 */
};

/* The maximum number of PS-Poll frames the FW will send in response to
 * traffic advertised in TIM before waking up (by sending a null frame with PS
 * = 0). Value 0 has a special meaning: there is no maximum count and the FW
 * will send as many PS-Poll as are necessary to retrieve buffered BU. This
 * parameter is used when the RX wake policy is
 * WMI_STA_PS_RX_WAKE_POLICY_POLL_UAPSD and ignored when the RX wake
 * policy is WMI_STA_PS_RX_WAKE_POLICY_WAKE.
 */
enum wmi_sta_ps_param_pspoll_count {
	WMI_STA_PS_PSPOLL_COUNT_NO_MAX = 0,
	/* Values greater than 0 indicate the maximum number of PS-Poll frames
	 * FW will send before waking up.
	 */
};

/* U-APSD configuration of peer station from (re)assoc request and TSPECs */
enum wmi_ap_ps_param_uapsd {
	WMI_AP_PS_UAPSD_AC0_DELIVERY_EN = (1 << 0),
	WMI_AP_PS_UAPSD_AC0_TRIGGER_EN  = (1 << 1),
	WMI_AP_PS_UAPSD_AC1_DELIVERY_EN = (1 << 2),
	WMI_AP_PS_UAPSD_AC1_TRIGGER_EN  = (1 << 3),
	WMI_AP_PS_UAPSD_AC2_DELIVERY_EN = (1 << 4),
	WMI_AP_PS_UAPSD_AC2_TRIGGER_EN  = (1 << 5),
	WMI_AP_PS_UAPSD_AC3_DELIVERY_EN = (1 << 6),
	WMI_AP_PS_UAPSD_AC3_TRIGGER_EN  = (1 << 7),
};

/* U-APSD maximum service period of peer station */
enum wmi_ap_ps_peer_param_max_sp {
	WMI_AP_PS_PEER_PARAM_MAX_SP_UNLIMITED = 0,
	WMI_AP_PS_PEER_PARAM_MAX_SP_2 = 1,
	WMI_AP_PS_PEER_PARAM_MAX_SP_4 = 2,
	WMI_AP_PS_PEER_PARAM_MAX_SP_6 = 3,
	MAX_WMI_AP_PS_PEER_PARAM_MAX_SP,
};

enum wmi_ap_ps_peer_param {
	/** Set uapsd configuration for a given peer.
	 *
	 * This include the delivery and trigger enabled state for each AC.
	 * The host MLME needs to set this based on AP capability and stations
	 * request Set in the association request  received from the station.
	 *
	 * Lower 8 bits of the value specify the UAPSD configuration.
	 *
	 * (see enum wmi_ap_ps_param_uapsd)
	 * The default value is 0.
	 */
	WMI_AP_PS_PEER_PARAM_UAPSD = 0,

	/**
	 * Set the service period for a UAPSD capable station
	 *
	 * The service period from wme ie in the (re)assoc request frame.
	 *
	 * (see enum wmi_ap_ps_peer_param_max_sp)
	 */
	WMI_AP_PS_PEER_PARAM_MAX_SP = 1,

	/** Time in seconds for aging out buffered frames
	 * for STA in power save
	 */
	WMI_AP_PS_PEER_PARAM_AGEOUT_TIME = 2,

	/** Specify frame types that are considered SIFS
	 * RESP trigger frame
	 */
	WMI_AP_PS_PEER_PARAM_SIFS_RESP_FRMTYPE = 3,

	/** Specifies the trigger state of TID.
	 * Valid only for UAPSD frame type
	 */
	WMI_AP_PS_PEER_PARAM_SIFS_RESP_UAPSD = 4,

	/* Specifies the WNM sleep state of a STA */
	WMI_AP_PS_PEER_PARAM_WNM_SLEEP = 5,
};

#define DISABLE_SIFS_RESPONSE_TRIGGER 0

#define WMI_MAX_KEY_INDEX   3
#define WMI_MAX_KEY_LEN     32

#define WMI_KEY_PAIRWISE 0x00
#define WMI_KEY_GROUP    0x01

#define WMI_CIPHER_NONE     0x0 /* clear key */
#define WMI_CIPHER_WEP      0x1
#define WMI_CIPHER_TKIP     0x2
#define WMI_CIPHER_AES_OCB  0x3
#define WMI_CIPHER_AES_CCM  0x4
#define WMI_CIPHER_WAPI     0x5
#define WMI_CIPHER_CKIP     0x6
#define WMI_CIPHER_AES_CMAC 0x7
#define WMI_CIPHER_ANY      0x8
#define WMI_CIPHER_AES_GCM  0x9
#define WMI_CIPHER_AES_GMAC 0xa

/* Value to disable fixed rate setting */
#define WMI_FIXED_RATE_NONE	(0xffff)

#define ATH11K_RC_VERSION_OFFSET	28
#define ATH11K_RC_PREAMBLE_OFFSET	8
#define ATH11K_RC_NSS_OFFSET		5

#define ATH11K_HW_RATE_CODE(rate, nss, preamble)	\
	((1 << ATH11K_RC_VERSION_OFFSET) |		\
	 ((nss) << ATH11K_RC_NSS_OFFSET) |		\
	 ((preamble) << ATH11K_RC_PREAMBLE_OFFSET) |	\
	 (rate))

/* Preamble types to be used with VDEV fixed rate configuration */
enum wmi_rate_preamble {
	WMI_RATE_PREAMBLE_OFDM,
	WMI_RATE_PREAMBLE_CCK,
	WMI_RATE_PREAMBLE_HT,
	WMI_RATE_PREAMBLE_VHT,
	WMI_RATE_PREAMBLE_HE,
};

/**
 * enum wmi_rtscts_prot_mode - Enable/Disable RTS/CTS and CTS2Self Protection.
 * @WMI_RTS_CTS_DISABLED: RTS/CTS protection is disabled.
 * @WMI_USE_RTS_CTS: RTS/CTS Enabled.
 * @WMI_USE_CTS2SELF: CTS to self protection Enabled.
 */
enum wmi_rtscts_prot_mode {
	WMI_RTS_CTS_DISABLED = 0,
	WMI_USE_RTS_CTS = 1,
	WMI_USE_CTS2SELF = 2,
};

/**
 * enum wmi_rtscts_profile - Selection of RTS CTS profile along with enabling
 *                           protection mode.
 * @WMI_RTSCTS_FOR_NO_RATESERIES: Neither of rate-series should use RTS-CTS
 * @WMI_RTSCTS_FOR_SECOND_RATESERIES: Only second rate-series will use RTS-CTS
 * @WMI_RTSCTS_ACROSS_SW_RETRIES: Only the second rate-series will use RTS-CTS,
 *                                but if there's a sw retry, both the rate
 *                                series will use RTS-CTS.
 * @WMI_RTSCTS_ERP: RTS/CTS used for ERP protection for every PPDU.
 * @WMI_RTSCTS_FOR_ALL_RATESERIES: Enable RTS-CTS for all rate series.
 */
enum wmi_rtscts_profile {
	WMI_RTSCTS_FOR_NO_RATESERIES = 0,
	WMI_RTSCTS_FOR_SECOND_RATESERIES = 1,
	WMI_RTSCTS_ACROSS_SW_RETRIES = 2,
	WMI_RTSCTS_ERP = 3,
	WMI_RTSCTS_FOR_ALL_RATESERIES = 4,
};

struct ath11k_hal_reg_cap {
	u32 eeprom_rd;
	u32 eeprom_rd_ext;
	u32 regcap1;
	u32 regcap2;
	u32 wireless_modes;
	u32 low_2ghz_chan;
	u32 high_2ghz_chan;
	u32 low_5ghz_chan;
	u32 high_5ghz_chan;
};

struct ath11k_mem_chunk {
	void *vaddr;
	dma_addr_t paddr;
	u32 len;
	u32 req_id;
};

#define WMI_SKB_HEADROOM sizeof(struct wmi_cmd_hdr)

enum wmi_sta_ps_param_rx_wake_policy {
	WMI_STA_PS_RX_WAKE_POLICY_WAKE = 0,
	WMI_STA_PS_RX_WAKE_POLICY_POLL_UAPSD = 1,
};

/* Do not change existing values! Used by ath11k_frame_mode parameter
 * module parameter.
 */
enum ath11k_hw_txrx_mode {
	ATH11K_HW_TXRX_RAW = 0,
	ATH11K_HW_TXRX_NATIVE_WIFI = 1,
	ATH11K_HW_TXRX_ETHERNET = 2,
};

struct wmi_wmm_params {
	u32 tlv_header;
	u32 cwmin;
	u32 cwmax;
	u32 aifs;
	u32 txoplimit;
	u32 acm;
	u32 no_ack;
} __packed;

struct wmi_wmm_params_arg {
	u8 acm;
	u8 aifs;
	u16 cwmin;
	u16 cwmax;
	u16 txop;
	u8 no_ack;
};

struct wmi_vdev_set_wmm_params_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_wmm_params wmm_params[4];
	u32 wmm_param_type;
} __packed;

struct wmi_wmm_params_all_arg {
	struct wmi_wmm_params_arg ac_be;
	struct wmi_wmm_params_arg ac_bk;
	struct wmi_wmm_params_arg ac_vi;
	struct wmi_wmm_params_arg ac_vo;
};

#define ATH11K_TWT_DEF_STA_CONG_TIMER_MS		5000
#define ATH11K_TWT_DEF_DEFAULT_SLOT_SIZE		10
#define ATH11K_TWT_DEF_CONGESTION_THRESH_SETUP		50
#define ATH11K_TWT_DEF_CONGESTION_THRESH_TEARDOWN	20
#define ATH11K_TWT_DEF_CONGESTION_THRESH_CRITICAL	100
#define ATH11K_TWT_DEF_INTERFERENCE_THRESH_TEARDOWN	80
#define ATH11K_TWT_DEF_INTERFERENCE_THRESH_SETUP	50
#define ATH11K_TWT_DEF_MIN_NO_STA_SETUP			10
#define ATH11K_TWT_DEF_MIN_NO_STA_TEARDOWN		2
#define ATH11K_TWT_DEF_NO_OF_BCAST_MCAST_SLOTS		2
#define ATH11K_TWT_DEF_MIN_NO_TWT_SLOTS			2
#define ATH11K_TWT_DEF_MAX_NO_STA_TWT			500
#define ATH11K_TWT_DEF_MODE_CHECK_INTERVAL		10000
#define ATH11K_TWT_DEF_ADD_STA_SLOT_INTERVAL		1000
#define ATH11K_TWT_DEF_REMOVE_STA_SLOT_INTERVAL		5000

struct wmi_twt_enable_params {
	u32 sta_cong_timer_ms;
	u32 mbss_support;
	u32 default_slot_size;
	u32 congestion_thresh_setup;
	u32 congestion_thresh_teardown;
	u32 congestion_thresh_critical;
	u32 interference_thresh_teardown;
	u32 interference_thresh_setup;
	u32 min_no_sta_setup;
	u32 min_no_sta_teardown;
	u32 no_of_bcast_mcast_slots;
	u32 min_no_twt_slots;
	u32 max_no_sta_twt;
	u32 mode_check_interval;
	u32 add_sta_slot_interval;
	u32 remove_sta_slot_interval;
};

struct wmi_twt_enable_params_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 sta_cong_timer_ms;
	u32 mbss_support;
	u32 default_slot_size;
	u32 congestion_thresh_setup;
	u32 congestion_thresh_teardown;
	u32 congestion_thresh_critical;
	u32 interference_thresh_teardown;
	u32 interference_thresh_setup;
	u32 min_no_sta_setup;
	u32 min_no_sta_teardown;
	u32 no_of_bcast_mcast_slots;
	u32 min_no_twt_slots;
	u32 max_no_sta_twt;
	u32 mode_check_interval;
	u32 add_sta_slot_interval;
	u32 remove_sta_slot_interval;
} __packed;

struct wmi_twt_disable_params_cmd {
	u32 tlv_header;
	u32 pdev_id;
} __packed;

enum WMI_HOST_TWT_COMMAND {
	WMI_HOST_TWT_COMMAND_REQUEST_TWT = 0,
	WMI_HOST_TWT_COMMAND_SUGGEST_TWT,
	WMI_HOST_TWT_COMMAND_DEMAND_TWT,
	WMI_HOST_TWT_COMMAND_TWT_GROUPING,
	WMI_HOST_TWT_COMMAND_ACCEPT_TWT,
	WMI_HOST_TWT_COMMAND_ALTERNATE_TWT,
	WMI_HOST_TWT_COMMAND_DICTATE_TWT,
	WMI_HOST_TWT_COMMAND_REJECT_TWT,
};

#define WMI_TWT_ADD_DIALOG_FLAG_BCAST           BIT(8)
#define WMI_TWT_ADD_DIALOG_FLAG_TRIGGER         BIT(9)
#define WMI_TWT_ADD_DIALOG_FLAG_FLOW_TYPE       BIT(10)
#define WMI_TWT_ADD_DIALOG_FLAG_PROTECTION      BIT(11)

struct wmi_twt_add_dialog_params_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 dialog_id;
	u32 wake_intvl_us;
	u32 wake_intvl_mantis;
	u32 wake_dura_us;
	u32 sp_offset_us;
	u32 flags;
} __packed;

struct wmi_twt_add_dialog_params {
	u32 vdev_id;
	u8 peer_macaddr[ETH_ALEN];
	u32 dialog_id;
	u32 wake_intvl_us;
	u32 wake_intvl_mantis;
	u32 wake_dura_us;
	u32 sp_offset_us;
	u8 twt_cmd;
	u8 flag_bcast;
	u8 flag_trigger;
	u8 flag_flow_type;
	u8 flag_protection;
} __packed;

enum  wmi_twt_add_dialog_status {
	WMI_ADD_TWT_STATUS_OK,
	WMI_ADD_TWT_STATUS_TWT_NOT_ENABLED,
	WMI_ADD_TWT_STATUS_USED_DIALOG_ID,
	WMI_ADD_TWT_STATUS_INVALID_PARAM,
	WMI_ADD_TWT_STATUS_NOT_READY,
	WMI_ADD_TWT_STATUS_NO_RESOURCE,
	WMI_ADD_TWT_STATUS_NO_ACK,
	WMI_ADD_TWT_STATUS_NO_RESPONSE,
	WMI_ADD_TWT_STATUS_DENIED,
	WMI_ADD_TWT_STATUS_UNKNOWN_ERROR,
};

struct wmi_twt_add_dialog_event {
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 dialog_id;
	u32 status;
} __packed;

struct wmi_twt_del_dialog_params {
	u32 vdev_id;
	u8 peer_macaddr[ETH_ALEN];
	u32 dialog_id;
} __packed;

struct wmi_twt_del_dialog_params_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 dialog_id;
} __packed;

struct wmi_twt_pause_dialog_params {
	u32 vdev_id;
	u8 peer_macaddr[ETH_ALEN];
	u32 dialog_id;
} __packed;

struct wmi_twt_pause_dialog_params_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 dialog_id;
} __packed;

struct wmi_twt_resume_dialog_params {
	u32 vdev_id;
	u8 peer_macaddr[ETH_ALEN];
	u32 dialog_id;
	u32 sp_offset_us;
	u32 next_twt_size;
} __packed;

struct wmi_twt_resume_dialog_params_cmd {
	u32 tlv_header;
	u32 vdev_id;
	struct wmi_mac_addr peer_macaddr;
	u32 dialog_id;
	u32 sp_offset_us;
	u32 next_twt_size;
} __packed;

struct wmi_obss_spatial_reuse_params_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 enable;
	s32 obss_min;
	s32 obss_max;
	u32 vdev_id;
} __packed;

struct wmi_pdev_obss_pd_bitmap_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 bitmap[2];
} __packed;

#define ATH11K_BSS_COLOR_COLLISION_SCAN_PERIOD_MS		200
#define ATH11K_OBSS_COLOR_COLLISION_DETECTION_DISABLE		0
#define ATH11K_OBSS_COLOR_COLLISION_DETECTION			1

#define ATH11K_BSS_COLOR_COLLISION_DETECTION_STA_PERIOD_MS	10000
#define ATH11K_BSS_COLOR_COLLISION_DETECTION_AP_PERIOD_MS	5000

enum wmi_bss_color_collision {
	WMI_BSS_COLOR_COLLISION_DISABLE = 0,
	WMI_BSS_COLOR_COLLISION_DETECTION,
	WMI_BSS_COLOR_FREE_SLOT_TIMER_EXPIRY,
	WMI_BSS_COLOR_FREE_SLOT_AVAILABLE,
};

struct wmi_obss_color_collision_cfg_params_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 flags;
	u32 evt_type;
	u32 current_bss_color;
	u32 detection_period_ms;
	u32 scan_period_ms;
	u32 free_slot_expiry_time_ms;
} __packed;

struct wmi_bss_color_change_enable_params_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 enable;
} __packed;

struct wmi_obss_color_collision_event {
	u32 vdev_id;
	u32 evt_type;
	u64 obss_color_bitmap;
} __packed;

#define ATH11K_IPV4_TH_SEED_SIZE 5
#define ATH11K_IPV6_TH_SEED_SIZE 11

struct ath11k_wmi_pdev_lro_config_cmd {
	u32 tlv_header;
	u32 lro_enable;
	u32 res;
	u32 th_4[ATH11K_IPV4_TH_SEED_SIZE];
	u32 th_6[ATH11K_IPV6_TH_SEED_SIZE];
	u32 pdev_id;
} __packed;

#define ATH11K_WMI_SPECTRAL_COUNT_DEFAULT                 0
#define ATH11K_WMI_SPECTRAL_PERIOD_DEFAULT              224
#define ATH11K_WMI_SPECTRAL_PRIORITY_DEFAULT              1
#define ATH11K_WMI_SPECTRAL_FFT_SIZE_DEFAULT              7
#define ATH11K_WMI_SPECTRAL_GC_ENA_DEFAULT                1
#define ATH11K_WMI_SPECTRAL_RESTART_ENA_DEFAULT           0
#define ATH11K_WMI_SPECTRAL_NOISE_FLOOR_REF_DEFAULT     -96
#define ATH11K_WMI_SPECTRAL_INIT_DELAY_DEFAULT           80
#define ATH11K_WMI_SPECTRAL_NB_TONE_THR_DEFAULT          12
#define ATH11K_WMI_SPECTRAL_STR_BIN_THR_DEFAULT           8
#define ATH11K_WMI_SPECTRAL_WB_RPT_MODE_DEFAULT           0
#define ATH11K_WMI_SPECTRAL_RSSI_RPT_MODE_DEFAULT         0
#define ATH11K_WMI_SPECTRAL_RSSI_THR_DEFAULT           0xf0
#define ATH11K_WMI_SPECTRAL_PWR_FORMAT_DEFAULT            0
#define ATH11K_WMI_SPECTRAL_RPT_MODE_DEFAULT              2
#define ATH11K_WMI_SPECTRAL_BIN_SCALE_DEFAULT             1
#define ATH11K_WMI_SPECTRAL_DBM_ADJ_DEFAULT               1
#define ATH11K_WMI_SPECTRAL_CHN_MASK_DEFAULT              1

struct ath11k_wmi_vdev_spectral_conf_param {
	u32 vdev_id;
	u32 scan_count;
	u32 scan_period;
	u32 scan_priority;
	u32 scan_fft_size;
	u32 scan_gc_ena;
	u32 scan_restart_ena;
	u32 scan_noise_floor_ref;
	u32 scan_init_delay;
	u32 scan_nb_tone_thr;
	u32 scan_str_bin_thr;
	u32 scan_wb_rpt_mode;
	u32 scan_rssi_rpt_mode;
	u32 scan_rssi_thr;
	u32 scan_pwr_format;
	u32 scan_rpt_mode;
	u32 scan_bin_scale;
	u32 scan_dbm_adj;
	u32 scan_chn_mask;
} __packed;

struct ath11k_wmi_vdev_spectral_conf_cmd {
	u32 tlv_header;
	struct ath11k_wmi_vdev_spectral_conf_param param;
} __packed;

#define ATH11K_WMI_SPECTRAL_TRIGGER_CMD_TRIGGER  1
#define ATH11K_WMI_SPECTRAL_TRIGGER_CMD_CLEAR    2
#define ATH11K_WMI_SPECTRAL_ENABLE_CMD_ENABLE    1
#define ATH11K_WMI_SPECTRAL_ENABLE_CMD_DISABLE   2

struct ath11k_wmi_vdev_spectral_enable_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 trigger_cmd;
	u32 enable_cmd;
} __packed;

struct ath11k_wmi_pdev_dma_ring_cfg_req_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 module_id;		/* see enum wmi_direct_buffer_module */
	u32 base_paddr_lo;
	u32 base_paddr_hi;
	u32 head_idx_paddr_lo;
	u32 head_idx_paddr_hi;
	u32 tail_idx_paddr_lo;
	u32 tail_idx_paddr_hi;
	u32 num_elems;		/* Number of elems in the ring */
	u32 buf_size;		/* size of allocated buffer in bytes */

	/* Number of wmi_dma_buf_release_entry packed together */
	u32 num_resp_per_event;

	/* Target should timeout and send whatever resp
	 * it has if this time expires, units in milliseconds
	 */
	u32 event_timeout_ms;
} __packed;

struct ath11k_wmi_dma_buf_release_fixed_param {
	u32 pdev_id;
	u32 module_id;
	u32 num_buf_release_entry;
	u32 num_meta_data_entry;
} __packed;

struct wmi_dma_buf_release_entry {
	u32 tlv_header;
	u32 paddr_lo;

	/* Bits 11:0:   address of data
	 * Bits 31:12:  host context data
	 */
	u32 paddr_hi;
} __packed;

#define WMI_SPECTRAL_META_INFO1_FREQ1		GENMASK(15, 0)
#define WMI_SPECTRAL_META_INFO1_FREQ2		GENMASK(31, 16)

#define WMI_SPECTRAL_META_INFO2_CHN_WIDTH	GENMASK(7, 0)

struct wmi_dma_buf_release_meta_data {
	u32 tlv_header;
	s32 noise_floor[WMI_MAX_CHAINS];
	u32 reset_delay;
	u32 freq1;
	u32 freq2;
	u32 ch_width;
} __packed;

enum wmi_fils_discovery_cmd_type {
	WMI_FILS_DISCOVERY_CMD,
	WMI_UNSOL_BCAST_PROBE_RESP,
};

struct wmi_fils_discovery_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 interval;
	u32 config; /* enum wmi_fils_discovery_cmd_type */
} __packed;

struct wmi_fils_discovery_tmpl_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 buf_len;
} __packed;

struct wmi_probe_tmpl_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 buf_len;
} __packed;

struct target_resource_config {
	u32 num_vdevs;
	u32 num_peers;
	u32 num_active_peers;
	u32 num_offload_peers;
	u32 num_offload_reorder_buffs;
	u32 num_peer_keys;
	u32 num_tids;
	u32 ast_skid_limit;
	u32 tx_chain_mask;
	u32 rx_chain_mask;
	u32 rx_timeout_pri[4];
	u32 rx_decap_mode;
	u32 scan_max_pending_req;
	u32 bmiss_offload_max_vdev;
	u32 roam_offload_max_vdev;
	u32 roam_offload_max_ap_profiles;
	u32 num_mcast_groups;
	u32 num_mcast_table_elems;
	u32 mcast2ucast_mode;
	u32 tx_dbg_log_size;
	u32 num_wds_entries;
	u32 dma_burst_size;
	u32 mac_aggr_delim;
	u32 rx_skip_defrag_timeout_dup_detection_check;
	u32 vow_config;
	u32 gtk_offload_max_vdev;
	u32 num_msdu_desc;
	u32 max_frag_entries;
	u32 max_peer_ext_stats;
	u32 smart_ant_cap;
	u32 bk_minfree;
	u32 be_minfree;
	u32 vi_minfree;
	u32 vo_minfree;
	u32 rx_batchmode;
	u32 tt_support;
	u32 flag1;
	u32 iphdr_pad_config;
	u32 qwrap_config:16,
	    alloc_frag_desc_for_data_pkt:16;
	u32 num_tdls_vdevs;
	u32 num_tdls_conn_table_entries;
	u32 beacon_tx_offload_max_vdev;
	u32 num_multicast_filter_entries;
	u32 num_wow_filters;
	u32 num_keep_alive_pattern;
	u32 keep_alive_pattern_size;
	u32 max_tdls_concurrent_sleep_sta;
	u32 max_tdls_concurrent_buffer_sta;
	u32 wmi_send_separate;
	u32 num_ocb_vdevs;
	u32 num_ocb_channels;
	u32 num_ocb_schedules;
	u32 num_ns_ext_tuples_cfg;
	u32 bpf_instruction_size;
	u32 max_bssid_rx_filters;
	u32 use_pdev_id;
	u32 peer_map_unmap_v2_support;
	u32 sched_params;
	u32 twt_ap_pdev_count;
	u32 twt_ap_sta_count;
	u8 is_reg_cc_ext_event_supported;
	u32 ema_max_vap_cnt;
	u32 ema_max_profile_period;
};

enum wmi_debug_log_param {
	WMI_DEBUG_LOG_PARAM_LOG_LEVEL = 0x1,
	WMI_DEBUG_LOG_PARAM_VDEV_ENABLE,
	WMI_DEBUG_LOG_PARAM_VDEV_DISABLE,
	WMI_DEBUG_LOG_PARAM_VDEV_ENABLE_BITMAP,
	WMI_DEBUG_LOG_PARAM_MOD_ENABLE_BITMAP,
	WMI_DEBUG_LOG_PARAM_WOW_MOD_ENABLE_BITMAP,
};

struct wmi_debug_log_config_cmd_fixed_param {
	u32 tlv_header;
	u32 dbg_log_param;
	u32 value;
} __packed;

#define MAX_RADIOS 3

#define WMI_SERVICE_READY_TIMEOUT_HZ (5 * HZ)
#define WMI_SEND_TIMEOUT_HZ (3 * HZ)

enum ath11k_wmi_peer_ps_state {
	WMI_PEER_PS_STATE_OFF,
	WMI_PEER_PS_STATE_ON,
	WMI_PEER_PS_STATE_DISABLED,
};

enum wmi_peer_ps_supported_bitmap {
	/* Used to indicate that power save state change is valid */
	WMI_PEER_PS_VALID = 0x1,
	WMI_PEER_PS_STATE_TIMESTAMP = 0x2,
};

struct wmi_peer_sta_ps_state_chg_event {
	struct wmi_mac_addr peer_macaddr;
	u32 peer_ps_state;
	u32 ps_supported_bitmap;
	u32 peer_ps_valid;
	u32 peer_ps_timestamp;
} __packed;

struct ath11k_wmi_base {
	struct ath11k_base *ab;
	struct ath11k_pdev_wmi wmi[MAX_RADIOS];
	enum ath11k_htc_ep_id wmi_endpoint_id[MAX_RADIOS];
	u32 max_msg_len[MAX_RADIOS];

	struct completion service_ready;
	struct completion unified_ready;
	DECLARE_BITMAP(svc_map, WMI_MAX_EXT2_SERVICE);
	wait_queue_head_t tx_credits_wq;
	u32 num_mem_chunks;
	u32 rx_decap_mode;
	struct wmi_host_mem_chunk mem_chunks[WMI_MAX_MEM_REQS];

	enum wmi_host_hw_mode_config_type preferred_hw_mode;
	struct target_resource_config  wlan_resource_config;

	struct ath11k_targ_cap *targ_cap;
};

/* Definition of HW data filtering */
enum hw_data_filter_type {
	WMI_HW_DATA_FILTER_DROP_NON_ARP_BC = BIT(0),
	WMI_HW_DATA_FILTER_DROP_NON_ICMPV6_MC = BIT(1),
};

struct wmi_hw_data_filter_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 enable;
	u32 hw_filter_bitmap;
} __packed;

/* WOW structures */
enum wmi_wow_wakeup_event {
	WOW_BMISS_EVENT = 0,
	WOW_BETTER_AP_EVENT,
	WOW_DEAUTH_RECVD_EVENT,
	WOW_MAGIC_PKT_RECVD_EVENT,
	WOW_GTK_ERR_EVENT,
	WOW_FOURWAY_HSHAKE_EVENT,
	WOW_EAPOL_RECVD_EVENT,
	WOW_NLO_DETECTED_EVENT,
	WOW_DISASSOC_RECVD_EVENT,
	WOW_PATTERN_MATCH_EVENT,
	WOW_CSA_IE_EVENT,
	WOW_PROBE_REQ_WPS_IE_EVENT,
	WOW_AUTH_REQ_EVENT,
	WOW_ASSOC_REQ_EVENT,
	WOW_HTT_EVENT,
	WOW_RA_MATCH_EVENT,
	WOW_HOST_AUTO_SHUTDOWN_EVENT,
	WOW_IOAC_MAGIC_EVENT,
	WOW_IOAC_SHORT_EVENT,
	WOW_IOAC_EXTEND_EVENT,
	WOW_IOAC_TIMER_EVENT,
	WOW_DFS_PHYERR_RADAR_EVENT,
	WOW_BEACON_EVENT,
	WOW_CLIENT_KICKOUT_EVENT,
	WOW_EVENT_MAX,
};

enum wmi_wow_interface_cfg {
	WOW_IFACE_PAUSE_ENABLED,
	WOW_IFACE_PAUSE_DISABLED
};

#define C2S(x) case x: return #x

static inline const char *wow_wakeup_event(enum wmi_wow_wakeup_event ev)
{
	switch (ev) {
	C2S(WOW_BMISS_EVENT);
	C2S(WOW_BETTER_AP_EVENT);
	C2S(WOW_DEAUTH_RECVD_EVENT);
	C2S(WOW_MAGIC_PKT_RECVD_EVENT);
	C2S(WOW_GTK_ERR_EVENT);
	C2S(WOW_FOURWAY_HSHAKE_EVENT);
	C2S(WOW_EAPOL_RECVD_EVENT);
	C2S(WOW_NLO_DETECTED_EVENT);
	C2S(WOW_DISASSOC_RECVD_EVENT);
	C2S(WOW_PATTERN_MATCH_EVENT);
	C2S(WOW_CSA_IE_EVENT);
	C2S(WOW_PROBE_REQ_WPS_IE_EVENT);
	C2S(WOW_AUTH_REQ_EVENT);
	C2S(WOW_ASSOC_REQ_EVENT);
	C2S(WOW_HTT_EVENT);
	C2S(WOW_RA_MATCH_EVENT);
	C2S(WOW_HOST_AUTO_SHUTDOWN_EVENT);
	C2S(WOW_IOAC_MAGIC_EVENT);
	C2S(WOW_IOAC_SHORT_EVENT);
	C2S(WOW_IOAC_EXTEND_EVENT);
	C2S(WOW_IOAC_TIMER_EVENT);
	C2S(WOW_DFS_PHYERR_RADAR_EVENT);
	C2S(WOW_BEACON_EVENT);
	C2S(WOW_CLIENT_KICKOUT_EVENT);
	C2S(WOW_EVENT_MAX);
	default:
		return NULL;
	}
}

enum wmi_wow_wake_reason {
	WOW_REASON_UNSPECIFIED = -1,
	WOW_REASON_NLOD = 0,
	WOW_REASON_AP_ASSOC_LOST,
	WOW_REASON_LOW_RSSI,
	WOW_REASON_DEAUTH_RECVD,
	WOW_REASON_DISASSOC_RECVD,
	WOW_REASON_GTK_HS_ERR,
	WOW_REASON_EAP_REQ,
	WOW_REASON_FOURWAY_HS_RECV,
	WOW_REASON_TIMER_INTR_RECV,
	WOW_REASON_PATTERN_MATCH_FOUND,
	WOW_REASON_RECV_MAGIC_PATTERN,
	WOW_REASON_P2P_DISC,
	WOW_REASON_WLAN_HB,
	WOW_REASON_CSA_EVENT,
	WOW_REASON_PROBE_REQ_WPS_IE_RECV,
	WOW_REASON_AUTH_REQ_RECV,
	WOW_REASON_ASSOC_REQ_RECV,
	WOW_REASON_HTT_EVENT,
	WOW_REASON_RA_MATCH,
	WOW_REASON_HOST_AUTO_SHUTDOWN,
	WOW_REASON_IOAC_MAGIC_EVENT,
	WOW_REASON_IOAC_SHORT_EVENT,
	WOW_REASON_IOAC_EXTEND_EVENT,
	WOW_REASON_IOAC_TIMER_EVENT,
	WOW_REASON_ROAM_HO,
	WOW_REASON_DFS_PHYERR_RADADR_EVENT,
	WOW_REASON_BEACON_RECV,
	WOW_REASON_CLIENT_KICKOUT_EVENT,
	WOW_REASON_PAGE_FAULT = 0x3a,
	WOW_REASON_DEBUG_TEST = 0xFF,
};

static inline const char *wow_reason(enum wmi_wow_wake_reason reason)
{
	switch (reason) {
	C2S(WOW_REASON_UNSPECIFIED);
	C2S(WOW_REASON_NLOD);
	C2S(WOW_REASON_AP_ASSOC_LOST);
	C2S(WOW_REASON_LOW_RSSI);
	C2S(WOW_REASON_DEAUTH_RECVD);
	C2S(WOW_REASON_DISASSOC_RECVD);
	C2S(WOW_REASON_GTK_HS_ERR);
	C2S(WOW_REASON_EAP_REQ);
	C2S(WOW_REASON_FOURWAY_HS_RECV);
	C2S(WOW_REASON_TIMER_INTR_RECV);
	C2S(WOW_REASON_PATTERN_MATCH_FOUND);
	C2S(WOW_REASON_RECV_MAGIC_PATTERN);
	C2S(WOW_REASON_P2P_DISC);
	C2S(WOW_REASON_WLAN_HB);
	C2S(WOW_REASON_CSA_EVENT);
	C2S(WOW_REASON_PROBE_REQ_WPS_IE_RECV);
	C2S(WOW_REASON_AUTH_REQ_RECV);
	C2S(WOW_REASON_ASSOC_REQ_RECV);
	C2S(WOW_REASON_HTT_EVENT);
	C2S(WOW_REASON_RA_MATCH);
	C2S(WOW_REASON_HOST_AUTO_SHUTDOWN);
	C2S(WOW_REASON_IOAC_MAGIC_EVENT);
	C2S(WOW_REASON_IOAC_SHORT_EVENT);
	C2S(WOW_REASON_IOAC_EXTEND_EVENT);
	C2S(WOW_REASON_IOAC_TIMER_EVENT);
	C2S(WOW_REASON_ROAM_HO);
	C2S(WOW_REASON_DFS_PHYERR_RADADR_EVENT);
	C2S(WOW_REASON_BEACON_RECV);
	C2S(WOW_REASON_CLIENT_KICKOUT_EVENT);
	C2S(WOW_REASON_PAGE_FAULT);
	C2S(WOW_REASON_DEBUG_TEST);
	default:
		return NULL;
	}
}

#undef C2S

struct wmi_wow_ev_arg {
	u32 vdev_id;
	u32 flag;
	enum wmi_wow_wake_reason wake_reason;
	u32 data_len;
};

enum wmi_tlv_pattern_type {
	WOW_PATTERN_MIN = 0,
	WOW_BITMAP_PATTERN = WOW_PATTERN_MIN,
	WOW_IPV4_SYNC_PATTERN,
	WOW_IPV6_SYNC_PATTERN,
	WOW_WILD_CARD_PATTERN,
	WOW_TIMER_PATTERN,
	WOW_MAGIC_PATTERN,
	WOW_IPV6_RA_PATTERN,
	WOW_IOAC_PKT_PATTERN,
	WOW_IOAC_TMR_PATTERN,
	WOW_PATTERN_MAX
};

#define WOW_DEFAULT_BITMAP_PATTERN_SIZE		148
#define WOW_DEFAULT_BITMASK_SIZE		148

#define WOW_MIN_PATTERN_SIZE	1
#define WOW_MAX_PATTERN_SIZE	148
#define WOW_MAX_PKT_OFFSET	128
#define WOW_HDR_LEN	(sizeof(struct ieee80211_hdr_3addr) + \
	sizeof(struct rfc1042_hdr))
#define WOW_MAX_REDUCE	(WOW_HDR_LEN - sizeof(struct ethhdr) - \
	offsetof(struct ieee80211_hdr_3addr, addr1))

struct wmi_wow_add_del_event_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 is_add;
	u32 event_bitmap;
} __packed;

struct wmi_wow_enable_cmd {
	u32 tlv_header;
	u32 enable;
	u32 pause_iface_config;
	u32 flags;
}  __packed;

struct wmi_wow_host_wakeup_ind {
	u32 tlv_header;
	u32 reserved;
} __packed;

struct wmi_tlv_wow_event_info {
	u32 vdev_id;
	u32 flag;
	u32 wake_reason;
	u32 data_len;
} __packed;

struct wmi_wow_bitmap_pattern {
	u32 tlv_header;
	u8 patternbuf[WOW_DEFAULT_BITMAP_PATTERN_SIZE];
	u8 bitmaskbuf[WOW_DEFAULT_BITMASK_SIZE];
	u32 pattern_offset;
	u32 pattern_len;
	u32 bitmask_len;
	u32 pattern_id;
} __packed;

struct wmi_wow_add_pattern_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 pattern_id;
	u32 pattern_type;
} __packed;

struct wmi_wow_del_pattern_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 pattern_id;
	u32 pattern_type;
} __packed;

#define WMI_PNO_MAX_SCHED_SCAN_PLANS      2
#define WMI_PNO_MAX_SCHED_SCAN_PLAN_INT   7200
#define WMI_PNO_MAX_SCHED_SCAN_PLAN_ITRNS 100
#define WMI_PNO_MAX_NETW_CHANNELS         26
#define WMI_PNO_MAX_NETW_CHANNELS_EX      60
#define WMI_PNO_MAX_SUPP_NETWORKS         WLAN_SCAN_PARAMS_MAX_SSID
#define WMI_PNO_MAX_IE_LENGTH             WLAN_SCAN_PARAMS_MAX_IE_LEN

/* size based of dot11 declaration without extra IEs as we will not carry those for PNO */
#define WMI_PNO_MAX_PB_REQ_SIZE    450

#define WMI_PNO_24G_DEFAULT_CH     1
#define WMI_PNO_5G_DEFAULT_CH      36

#define WMI_ACTIVE_MAX_CHANNEL_TIME 40
#define WMI_PASSIVE_MAX_CHANNEL_TIME   110

/* SSID broadcast type */
enum wmi_ssid_bcast_type {
	BCAST_UNKNOWN      = 0,
	BCAST_NORMAL       = 1,
	BCAST_HIDDEN       = 2,
};

#define WMI_NLO_MAX_SSIDS    16
#define WMI_NLO_MAX_CHAN     48

#define WMI_NLO_CONFIG_STOP                             BIT(0)
#define WMI_NLO_CONFIG_START                            BIT(1)
#define WMI_NLO_CONFIG_RESET                            BIT(2)
#define WMI_NLO_CONFIG_SLOW_SCAN                        BIT(4)
#define WMI_NLO_CONFIG_FAST_SCAN                        BIT(5)
#define WMI_NLO_CONFIG_SSID_HIDE_EN                     BIT(6)

/* This bit is used to indicate if EPNO or supplicant PNO is enabled.
 * Only one of them can be enabled at a given time
 */
#define WMI_NLO_CONFIG_ENLO                             BIT(7)
#define WMI_NLO_CONFIG_SCAN_PASSIVE                     BIT(8)
#define WMI_NLO_CONFIG_ENLO_RESET                       BIT(9)
#define WMI_NLO_CONFIG_SPOOFED_MAC_IN_PROBE_REQ         BIT(10)
#define WMI_NLO_CONFIG_RANDOM_SEQ_NO_IN_PROBE_REQ       BIT(11)
#define WMI_NLO_CONFIG_ENABLE_IE_WHITELIST_IN_PROBE_REQ BIT(12)
#define WMI_NLO_CONFIG_ENABLE_CNLO_RSSI_CONFIG          BIT(13)

struct wmi_nlo_ssid_param {
	u32 valid;
	struct wmi_ssid ssid;
} __packed;

struct wmi_nlo_enc_param {
	u32 valid;
	u32 enc_type;
} __packed;

struct wmi_nlo_auth_param {
	u32 valid;
	u32 auth_type;
} __packed;

struct wmi_nlo_bcast_nw_param {
	u32 valid;
	u32 bcast_nw_type;
} __packed;

struct wmi_nlo_rssi_param {
	u32 valid;
	s32 rssi;
} __packed;

struct nlo_configured_parameters {
	/* TLV tag and len;*/
	u32 tlv_header;
	struct wmi_nlo_ssid_param ssid;
	struct wmi_nlo_enc_param enc_type;
	struct wmi_nlo_auth_param auth_type;
	struct wmi_nlo_rssi_param rssi_cond;

	/* indicates if the SSID is hidden or not */
	struct wmi_nlo_bcast_nw_param bcast_nw_type;
} __packed;

struct wmi_network_type {
	struct wmi_ssid ssid;
	u32 authentication;
	u32 encryption;
	u32 bcast_nw_type;
	u8 channel_count;
	u16 channels[WMI_PNO_MAX_NETW_CHANNELS_EX];
	s32 rssi_threshold;
};

struct wmi_pno_scan_req {
	u8 enable;
	u8 vdev_id;
	u8 uc_networks_count;
	struct wmi_network_type a_networks[WMI_PNO_MAX_SUPP_NETWORKS];
	u32 fast_scan_period;
	u32 slow_scan_period;
	u8 fast_scan_max_cycles;

	bool do_passive_scan;

	u32 delay_start_time;
	u32 active_min_time;
	u32 active_max_time;
	u32 passive_min_time;
	u32 passive_max_time;

	/* mac address randomization attributes */
	u32 enable_pno_scan_randomization;
	u8 mac_addr[ETH_ALEN];
	u8 mac_addr_mask[ETH_ALEN];
};

struct wmi_wow_nlo_config_cmd {
	u32 tlv_header;
	u32 flags;
	u32 vdev_id;
	u32 fast_scan_max_cycles;
	u32 active_dwell_time;
	u32 passive_dwell_time;
	u32 probe_bundle_size;

	/* ART = IRT */
	u32 rest_time;

	/* Max value that can be reached after SBM */
	u32 max_rest_time;

	/* SBM */
	u32 scan_backoff_multiplier;

	/* SCBM */
	u32 fast_scan_period;

	/* specific to windows */
	u32 slow_scan_period;

	u32 no_of_ssids;

	u32 num_of_channels;

	/* NLO scan start delay time in milliseconds */
	u32 delay_start_time;

	/* MAC Address to use in Probe Req as SA */
	struct wmi_mac_addr mac_addr;

	/* Mask on which MAC has to be randomized */
	struct wmi_mac_addr mac_mask;

	/* IE bitmap to use in Probe Req */
	u32 ie_bitmap[8];

	/* Number of vendor OUIs. In the TLV vendor_oui[] */
	u32 num_vendor_oui;

	/* Number of connected NLO band preferences */
	u32 num_cnlo_band_pref;

	/* The TLVs will follow.
	 * nlo_configured_parameters nlo_list[];
	 * u32 channel_list[num_of_channels];
	 */
} __packed;

#define WMI_MAX_NS_OFFLOADS           2
#define WMI_MAX_ARP_OFFLOADS          2

#define WMI_ARPOL_FLAGS_VALID              BIT(0)
#define WMI_ARPOL_FLAGS_MAC_VALID          BIT(1)
#define WMI_ARPOL_FLAGS_REMOTE_IP_VALID    BIT(2)

struct wmi_arp_offload_tuple {
	u32 tlv_header;
	u32 flags;
	u8 target_ipaddr[4];
	u8 remote_ipaddr[4];
	struct wmi_mac_addr target_mac;
} __packed;

#define WMI_NSOL_FLAGS_VALID               BIT(0)
#define WMI_NSOL_FLAGS_MAC_VALID           BIT(1)
#define WMI_NSOL_FLAGS_REMOTE_IP_VALID     BIT(2)
#define WMI_NSOL_FLAGS_IS_IPV6_ANYCAST     BIT(3)

#define WMI_NSOL_MAX_TARGET_IPS    2

struct wmi_ns_offload_tuple {
	u32 tlv_header;
	u32 flags;
	u8 target_ipaddr[WMI_NSOL_MAX_TARGET_IPS][16];
	u8 solicitation_ipaddr[16];
	u8 remote_ipaddr[16];
	struct wmi_mac_addr target_mac;
} __packed;

struct wmi_set_arp_ns_offload_cmd {
	u32 tlv_header;
	u32 flags;
	u32 vdev_id;
	u32 num_ns_ext_tuples;
	/* The TLVs follow:
	 * wmi_ns_offload_tuple  ns_tuples[WMI_MAX_NS_OFFLOADS];
	 * wmi_arp_offload_tuple arp_tuples[WMI_MAX_ARP_OFFLOADS];
	 * wmi_ns_offload_tuple  ns_ext_tuples[num_ns_ext_tuples];
	 */
} __packed;

#define GTK_OFFLOAD_OPCODE_MASK             0xFF000000
#define GTK_OFFLOAD_ENABLE_OPCODE           0x01000000
#define GTK_OFFLOAD_DISABLE_OPCODE          0x02000000
#define GTK_OFFLOAD_REQUEST_STATUS_OPCODE   0x04000000

#define GTK_OFFLOAD_KEK_BYTES       16
#define GTK_OFFLOAD_KCK_BYTES       16
#define GTK_REPLAY_COUNTER_BYTES    8
#define WMI_MAX_KEY_LEN             32
#define IGTK_PN_SIZE                6

struct wmi_replayc_cnt {
	union {
		u8 counter[GTK_REPLAY_COUNTER_BYTES];
		struct {
			u32 word0;
			u32 word1;
		} __packed;
	} __packed;
} __packed;

struct wmi_gtk_offload_status_event {
	u32 vdev_id;
	u32 flags;
	u32 refresh_cnt;
	struct wmi_replayc_cnt replay_ctr;
	u8 igtk_key_index;
	u8 igtk_key_length;
	u8 igtk_key_rsc[IGTK_PN_SIZE];
	u8 igtk_key[WMI_MAX_KEY_LEN];
	u8 gtk_key_index;
	u8 gtk_key_length;
	u8 gtk_key_rsc[GTK_REPLAY_COUNTER_BYTES];
	u8 gtk_key[WMI_MAX_KEY_LEN];
} __packed;

struct wmi_gtk_rekey_offload_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 flags;
	u8 kek[GTK_OFFLOAD_KEK_BYTES];
	u8 kck[GTK_OFFLOAD_KCK_BYTES];
	u8 replay_ctr[GTK_REPLAY_COUNTER_BYTES];
} __packed;

#define BIOS_SAR_TABLE_LEN	(22)
#define BIOS_SAR_RSVD1_LEN	(6)
#define BIOS_SAR_RSVD2_LEN	(18)

struct wmi_pdev_set_sar_table_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 sar_len;
	u32 rsvd_len;
} __packed;

struct wmi_pdev_set_geo_table_cmd {
	u32 tlv_header;
	u32 pdev_id;
	u32 rsvd_len;
} __packed;

struct wmi_sta_keepalive_cmd {
	u32 tlv_header;
	u32 vdev_id;
	u32 enabled;

	/* WMI_STA_KEEPALIVE_METHOD_ */
	u32 method;

	/* in seconds */
	u32 interval;

	/* following this structure is the TLV for struct
	 * wmi_sta_keepalive_arp_resp
	 */
} __packed;

struct wmi_sta_keepalive_arp_resp {
	u32 tlv_header;
	u32 src_ip4_addr;
	u32 dest_ip4_addr;
	struct wmi_mac_addr dest_mac_addr;
} __packed;

struct wmi_sta_keepalive_arg {
	u32 vdev_id;
	u32 enabled;
	u32 method;
	u32 interval;
	u32 src_ip4_addr;
	u32 dest_ip4_addr;
	const u8 dest_mac_addr[ETH_ALEN];
};

enum wmi_sta_keepalive_method {
	WMI_STA_KEEPALIVE_METHOD_NULL_FRAME = 1,
	WMI_STA_KEEPALIVE_METHOD_UNSOLICITED_ARP_RESPONSE = 2,
	WMI_STA_KEEPALIVE_METHOD_ETHERNET_LOOPBACK = 3,
	WMI_STA_KEEPALIVE_METHOD_GRATUITOUS_ARP_REQUEST = 4,
	WMI_STA_KEEPALIVE_METHOD_MGMT_VENDOR_ACTION = 5,
};

#define WMI_STA_KEEPALIVE_INTERVAL_DEFAULT	30
#define WMI_STA_KEEPALIVE_INTERVAL_DISABLE	0

enum wmi_wmm_params_type {
	WMI_WMM_PARAM_TYPE_LEGACY = 0,
	WMI_WMM_PARAM_TYPE_11AX_MU_EDCA = 1,
};

const void **ath11k_wmi_tlv_parse_alloc(struct ath11k_base *ab,
					struct sk_buff *skb, gfp_t gfp);
int ath11k_wmi_cmd_send(struct ath11k_pdev_wmi *wmi, struct sk_buff *skb,
			u32 cmd_id);
struct sk_buff *ath11k_wmi_alloc_skb(struct ath11k_wmi_base *wmi_sc, u32 len);
int ath11k_wmi_mgmt_send(struct ath11k *ar, u32 vdev_id, u32 buf_id,
			 struct sk_buff *frame);
int ath11k_wmi_p2p_go_bcn_ie(struct ath11k *ar, u32 vdev_id,
			     const u8 *p2p_ie);
int ath11k_wmi_bcn_tmpl(struct ath11k *ar, u32 vdev_id,
			struct ieee80211_mutable_offsets *offs,
			struct sk_buff *bcn, u32 ema_param);
int ath11k_wmi_vdev_down(struct ath11k *ar, u8 vdev_id);
int ath11k_wmi_vdev_up(struct ath11k *ar, u32 vdev_id, u32 aid,
		       const u8 *bssid, u8 *tx_bssid, u32 nontx_profile_idx,
		       u32 nontx_profile_cnt);
int ath11k_wmi_vdev_stop(struct ath11k *ar, u8 vdev_id);
int ath11k_wmi_vdev_start(struct ath11k *ar, struct wmi_vdev_start_req_arg *arg,
			  bool restart);
int ath11k_wmi_set_peer_param(struct ath11k *ar, const u8 *peer_addr,
			      u32 vdev_id, u32 param_id, u32 param_val);
int ath11k_wmi_pdev_set_param(struct ath11k *ar, u32 param_id,
			      u32 param_value, u8 pdev_id);
int ath11k_wmi_pdev_set_ps_mode(struct ath11k *ar, int vdev_id,
				enum wmi_sta_ps_mode psmode);
int ath11k_wmi_wait_for_unified_ready(struct ath11k_base *ab);
int ath11k_wmi_cmd_init(struct ath11k_base *ab);
int ath11k_wmi_wait_for_service_ready(struct ath11k_base *ab);
int ath11k_wmi_connect(struct ath11k_base *ab);
int ath11k_wmi_pdev_attach(struct ath11k_base *ab,
			   u8 pdev_id);
int ath11k_wmi_attach(struct ath11k_base *ab);
void ath11k_wmi_detach(struct ath11k_base *ab);
int ath11k_wmi_vdev_create(struct ath11k *ar, u8 *macaddr,
			   struct vdev_create_params *param);
int ath11k_wmi_peer_rx_reorder_queue_setup(struct ath11k *ar, int vdev_id,
					   const u8 *addr, dma_addr_t paddr,
					   u8 tid, u8 ba_window_size_valid,
					   u32 ba_window_size);
int ath11k_wmi_send_peer_create_cmd(struct ath11k *ar,
				    struct peer_create_params *param);
int ath11k_wmi_vdev_set_param_cmd(struct ath11k *ar, u32 vdev_id,
				  u32 param_id, u32 param_value);

int ath11k_wmi_set_sta_ps_param(struct ath11k *ar, u32 vdev_id,
				u32 param, u32 param_value);
int ath11k_wmi_force_fw_hang_cmd(struct ath11k *ar, u32 type, u32 delay_time_ms);
int ath11k_wmi_send_peer_delete_cmd(struct ath11k *ar,
				    const u8 *peer_addr, u8 vdev_id);
int ath11k_wmi_vdev_delete(struct ath11k *ar, u8 vdev_id);
void ath11k_wmi_start_scan_init(struct ath11k *ar, struct scan_req_params *arg);
int ath11k_wmi_send_scan_start_cmd(struct ath11k *ar,
				   struct scan_req_params *params);
int ath11k_wmi_send_scan_stop_cmd(struct ath11k *ar,
				  struct scan_cancel_param *param);
int ath11k_wmi_send_wmm_update_cmd_tlv(struct ath11k *ar, u32 vdev_id,
				       struct wmi_wmm_params_all_arg *param,
				       enum wmi_wmm_params_type wmm_param_type);
int ath11k_wmi_pdev_suspend(struct ath11k *ar, u32 suspend_opt,
			    u32 pdev_id);
int ath11k_wmi_pdev_resume(struct ath11k *ar, u32 pdev_id);

int ath11k_wmi_send_peer_assoc_cmd(struct ath11k *ar,
				   struct peer_assoc_params *param);
int ath11k_wmi_vdev_install_key(struct ath11k *ar,
				struct wmi_vdev_install_key_arg *arg);
int ath11k_wmi_pdev_bss_chan_info_request(struct ath11k *ar,
					  enum wmi_bss_chan_info_req_type type);
int ath11k_wmi_send_stats_request_cmd(struct ath11k *ar,
				      struct stats_request_params *param);
int ath11k_wmi_send_pdev_temperature_cmd(struct ath11k *ar);
int ath11k_wmi_send_peer_flush_tids_cmd(struct ath11k *ar,
					u8 peer_addr[ETH_ALEN],
					struct peer_flush_params *param);
int ath11k_wmi_send_set_ap_ps_param_cmd(struct ath11k *ar, u8 *peer_addr,
					struct ap_ps_params *param);
int ath11k_wmi_send_scan_chan_list_cmd(struct ath11k *ar,
				       struct scan_chan_list_params *chan_list);
int ath11k_wmi_send_dfs_phyerr_offload_enable_cmd(struct ath11k *ar,
						  u32 pdev_id);
int ath11k_wmi_addba_clear_resp(struct ath11k *ar, u32 vdev_id, const u8 *mac);
int ath11k_wmi_addba_send(struct ath11k *ar, u32 vdev_id, const u8 *mac,
			  u32 tid, u32 buf_size);
int ath11k_wmi_addba_set_resp(struct ath11k *ar, u32 vdev_id, const u8 *mac,
			      u32 tid, u32 status);
int ath11k_wmi_delba_send(struct ath11k *ar, u32 vdev_id, const u8 *mac,
			  u32 tid, u32 initiator, u32 reason);
int ath11k_wmi_send_bcn_offload_control_cmd(struct ath11k *ar,
					    u32 vdev_id, u32 bcn_ctrl_op);
int ath11k_wmi_send_set_current_country_cmd(struct ath11k *ar,
					    struct wmi_set_current_country_params *param);
int
ath11k_wmi_send_init_country_cmd(struct ath11k *ar,
				 struct wmi_init_country_params init_cc_param);

int ath11k_wmi_send_11d_scan_start_cmd(struct ath11k *ar,
				       struct wmi_11d_scan_start_params *param);
int ath11k_wmi_send_11d_scan_stop_cmd(struct ath11k *ar, u32 vdev_id);

int
ath11k_wmi_send_thermal_mitigation_param_cmd(struct ath11k *ar,
					     struct thermal_mitigation_params *param);
int ath11k_wmi_pdev_pktlog_enable(struct ath11k *ar, u32 pktlog_filter);
int ath11k_wmi_pdev_pktlog_disable(struct ath11k *ar);
int ath11k_wmi_pdev_peer_pktlog_filter(struct ath11k *ar, u8 *addr, u8 enable);
int
ath11k_wmi_rx_reord_queue_remove(struct ath11k *ar,
				 struct rx_reorder_queue_remove_params *param);
int ath11k_wmi_send_pdev_set_regdomain(struct ath11k *ar,
				       struct pdev_set_regdomain_params *param);
int ath11k_wmi_pull_fw_stats(struct ath11k_base *ab, struct sk_buff *skb,
			     struct ath11k_fw_stats *stats);
void ath11k_wmi_fw_stats_fill(struct ath11k *ar,
			      struct ath11k_fw_stats *fw_stats, u32 stats_id,
			      char *buf);
int ath11k_wmi_simulate_radar(struct ath11k *ar);
void ath11k_wmi_fill_default_twt_params(struct wmi_twt_enable_params *twt_params);
int ath11k_wmi_send_twt_enable_cmd(struct ath11k *ar, u32 pdev_id,
				   struct wmi_twt_enable_params *params);
int ath11k_wmi_send_twt_disable_cmd(struct ath11k *ar, u32 pdev_id);
int ath11k_wmi_send_twt_add_dialog_cmd(struct ath11k *ar,
				       struct wmi_twt_add_dialog_params *params);
int ath11k_wmi_send_twt_del_dialog_cmd(struct ath11k *ar,
				       struct wmi_twt_del_dialog_params *params);
int ath11k_wmi_send_twt_pause_dialog_cmd(struct ath11k *ar,
					 struct wmi_twt_pause_dialog_params *params);
int ath11k_wmi_send_twt_resume_dialog_cmd(struct ath11k *ar,
					  struct wmi_twt_resume_dialog_params *params);
int ath11k_wmi_send_obss_spr_cmd(struct ath11k *ar, u32 vdev_id,
				 struct ieee80211_he_obss_pd *he_obss_pd);
int ath11k_wmi_pdev_set_srg_bss_color_bitmap(struct ath11k *ar, u32 *bitmap);
int ath11k_wmi_pdev_set_srg_patial_bssid_bitmap(struct ath11k *ar, u32 *bitmap);
int ath11k_wmi_pdev_srg_obss_color_enable_bitmap(struct ath11k *ar,
						 u32 *bitmap);
int ath11k_wmi_pdev_srg_obss_bssid_enable_bitmap(struct ath11k *ar,
						 u32 *bitmap);
int ath11k_wmi_pdev_non_srg_obss_color_enable_bitmap(struct ath11k *ar,
						     u32 *bitmap);
int ath11k_wmi_pdev_non_srg_obss_bssid_enable_bitmap(struct ath11k *ar,
						     u32 *bitmap);
int ath11k_wmi_send_obss_color_collision_cfg_cmd(struct ath11k *ar, u32 vdev_id,
						 u8 bss_color, u32 period,
						 bool enable);
int ath11k_wmi_send_bss_color_change_enable_cmd(struct ath11k *ar, u32 vdev_id,
						bool enable);
int ath11k_wmi_pdev_lro_cfg(struct ath11k *ar, int pdev_id);
int ath11k_wmi_pdev_dma_ring_cfg(struct ath11k *ar,
				 struct ath11k_wmi_pdev_dma_ring_cfg_req_cmd *param);
int ath11k_wmi_vdev_spectral_enable(struct ath11k *ar, u32 vdev_id,
				    u32 trigger, u32 enable);
int ath11k_wmi_vdev_spectral_conf(struct ath11k *ar,
				  struct ath11k_wmi_vdev_spectral_conf_param *param);
int ath11k_wmi_fils_discovery_tmpl(struct ath11k *ar, u32 vdev_id,
				   struct sk_buff *tmpl);
int ath11k_wmi_fils_discovery(struct ath11k *ar, u32 vdev_id, u32 interval,
			      bool unsol_bcast_probe_resp_enabled);
int ath11k_wmi_probe_resp_tmpl(struct ath11k *ar, u32 vdev_id,
			       struct sk_buff *tmpl);
int ath11k_wmi_set_hw_mode(struct ath11k_base *ab,
			   enum wmi_host_hw_mode_config_type mode);
int ath11k_wmi_wow_host_wakeup_ind(struct ath11k *ar);
int ath11k_wmi_wow_enable(struct ath11k *ar);
int ath11k_wmi_scan_prob_req_oui(struct ath11k *ar,
				 const u8 mac_addr[ETH_ALEN]);
int ath11k_wmi_fw_dbglog_cfg(struct ath11k *ar, u32 *module_id_bitmap,
			     struct ath11k_fw_dbglog *dbglog);
int ath11k_wmi_wow_config_pno(struct ath11k *ar, u32 vdev_id,
			      struct wmi_pno_scan_req  *pno_scan);
int ath11k_wmi_wow_del_pattern(struct ath11k *ar, u32 vdev_id, u32 pattern_id);
int ath11k_wmi_wow_add_pattern(struct ath11k *ar, u32 vdev_id, u32 pattern_id,
			       const u8 *pattern, const u8 *mask,
			       int pattern_len, int pattern_offset);
int ath11k_wmi_wow_add_wakeup_event(struct ath11k *ar, u32 vdev_id,
				    enum wmi_wow_wakeup_event event,
				    u32 enable);
int ath11k_wmi_hw_data_filter_cmd(struct ath11k *ar, u32 vdev_id,
				  u32 filter_bitmap, bool enable);
int ath11k_wmi_arp_ns_offload(struct ath11k *ar,
			      struct ath11k_vif *arvif, bool enable);
int ath11k_wmi_gtk_rekey_offload(struct ath11k *ar,
				 struct ath11k_vif *arvif, bool enable);
int ath11k_wmi_gtk_rekey_getinfo(struct ath11k *ar,
				 struct ath11k_vif *arvif);
int ath11k_wmi_pdev_set_bios_sar_table_param(struct ath11k *ar, const u8 *sar_val);
int ath11k_wmi_pdev_set_bios_geo_table_param(struct ath11k *ar);
int ath11k_wmi_sta_keepalive(struct ath11k *ar,
			     const struct wmi_sta_keepalive_arg *arg);
bool ath11k_wmi_supports_6ghz_cc_ext(struct ath11k *ar);
int ath11k_wmi_send_vdev_set_tpc_power(struct ath11k *ar,
				       u32 vdev_id,
				       struct ath11k_reg_tpc_power_info *param);

#endif
