/*
 * TIFF data tables
 * Copyright (c) 2011 Thomas Kuehnel
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file
 * TIFF data tables
 * @author Thomas Kuehnel
 * @see GeoTIFF specification at
 * http://www.remotesensing.org/geotiff/spec/geotiffhome.html
 */

#ifndef AVCODEC_TIFF_DATA_H
#define AVCODEC_TIFF_DATA_H

#include "tiff.h"

typedef struct TiffGeoTagNameType {
    enum TiffGeoTagType type;
    unsigned offset;
} TiffGeoTagNameType;

#define TIFF_CONF_KEY_ID_OFFSET 1024
#define CONF_NAME_TYPE_MAP(KEY)                 \
    KEY(GTModelTypeGeoKey,              SHORT ) \
    KEY(GTRasterTypeGeoKey,             SHORT ) \
    KEY(GTCitationGeoKey,               STRING) \

#define TIFF_GEOG_KEY_ID_OFFSET 2048
#define GEOG_NAME_TYPE_MAP(KEY)                 \
    KEY(GeographicTypeGeoKey,           SHORT ) \
    KEY(GeogCitationGeoKey,             STRING) \
    KEY(GeogGeodeticDatumGeoKey,        SHORT ) \
    KEY(GeogPrimeMeridianGeoKey,        SHORT ) \
    KEY(GeogLinearUnitsGeoKey,          SHORT ) \
    KEY(GeogLinearUnitSizeGeoKey,       DOUBLE) \
    KEY(GeogAngularUnitsGeoKey,         SHORT ) \
    KEY(GeogAngularUnitSizeGeoKey,      DOUBLE) \
    KEY(GeogEllipsoidGeoKey,            SHORT ) \
    KEY(GeogSemiMajorAxisGeoKey,        DOUBLE) \
    KEY(GeogSemiMinorAxisGeoKey,        DOUBLE) \
    KEY(GeogInvFlatteningGeoKey,        DOUBLE) \
    KEY(GeogAzimuthUnitsGeoKey,         SHORT ) \
    KEY(GeogPrimeMeridianLongGeoKey,    DOUBLE) \

#define TIFF_PROJ_KEY_ID_OFFSET 3072
#define PROJ_NAME_TYPE_MAP(KEY)                 \
    KEY(ProjectedCSTypeGeoKey,          SHORT ) \
    KEY(PCSCitationGeoKey,              STRING) \
    KEY(ProjectionGeoKey,               SHORT ) \
    KEY(ProjCoordTransGeoKey,           SHORT ) \
    KEY(ProjLinearUnitsGeoKey,          SHORT ) \
    KEY(ProjLinearUnitSizeGeoKey,       DOUBLE) \
    KEY(ProjStdParallel1GeoKey,         DOUBLE) \
    KEY(ProjStdParallel2GeoKey,         DOUBLE) \
    KEY(ProjNatOriginLongGeoKey,        DOUBLE) \
    KEY(ProjNatOriginLatGeoKey,         DOUBLE) \
    KEY(ProjFalseEastingGeoKey,         DOUBLE) \
    KEY(ProjFalseNorthingGeoKey,        DOUBLE) \
    KEY(ProjFalseOriginLongGeoKey,      DOUBLE) \
    KEY(ProjFalseOriginLatGeoKey,       DOUBLE) \
    KEY(ProjFalseOriginEastingGeoKey,   DOUBLE) \
    KEY(ProjFalseOriginNorthingGeoKey,  DOUBLE) \
    KEY(ProjCenterLongGeoKey,           DOUBLE) \
    KEY(ProjCenterLatGeoKey,            DOUBLE) \
    KEY(ProjCenterEastingGeoKey,        DOUBLE) \
    KEY(ProjCenterNorthingGeoKey,       DOUBLE) \
    KEY(ProjScaleAtNatOriginGeoKey,     DOUBLE) \
    KEY(ProjScaleAtCenterGeoKey,        DOUBLE) \
    KEY(ProjAzimuthAngleGeoKey,         DOUBLE) \
    KEY(ProjStraightVertPoleLongGeoKey, DOUBLE) \

#define TIFF_VERT_KEY_ID_OFFSET 4096
#define VERT_NAME_TYPE_MAP(KEY)                 \
    KEY(VerticalCSTypeGeoKey,           SHORT ) \
    KEY(VerticalCitationGeoKey,         STRING) \
    KEY(VerticalDatumGeoKey,            SHORT ) \
    KEY(VerticalUnitsGeoKey,            SHORT ) \

#define ADD_OFFSET(NAME, TYPE)                          \
    NAME ## _OFFSET,                                    \
    NAME ## _END = NAME ## _OFFSET + sizeof(#NAME) - 1, \

#define STRING(NAME, TYPE) #NAME "\0"

#define ENTRY(NAME, TYPE) { .type = GEOTIFF_ ## TYPE, .offset = NAME ## _OFFSET },
#define NAME_TYPE_MAP(NAME, name) \
    enum { \
        NAME ## _NAME_TYPE_MAP(ADD_OFFSET) \
    }; \
    static const TiffGeoTagNameType tiff_ ## name ## _name_type_map[] = { \
        NAME ## _NAME_TYPE_MAP(ENTRY) \
    }; \
    static const char *const tiff_ ## name ## _name_type_string = \
        NAME ## _NAME_TYPE_MAP(STRING)

NAME_TYPE_MAP(CONF, conf);
NAME_TYPE_MAP(GEOG, geog);
NAME_TYPE_MAP(PROJ, proj);
NAME_TYPE_MAP(VERT, vert);

#define TIFF_GEO_KEY_UNDEFINED    0
#define TIFF_GEO_KEY_USER_DEFINED 32767

#define TIFF_GT_MODEL_TYPE_OFFSET 1
static const char *const tiff_gt_model_type_codes[] = {
    "ModelTypeProjected",
    "ModelTypeGeographic",
    "ModelTypeGeocentric"
};

#define TIFF_GT_RASTER_TYPE_OFFSET 1
static const char *const tiff_gt_raster_type_codes[] = {
    "RasterPixelIsArea",
    "RasterPixelIsPoint"
};

#define TIFF_LINEAR_UNIT_OFFSET 9001
static const char *const tiff_linear_unit_codes[] = {
    "Linear_Meter",
    "Linear_Foot",
    "Linear_Foot_US_Survey",
    "Linear_Foot_Modified_American",
    "Linear_Foot_Clarke",
    "Linear_Foot_Indian",
    "Linear_Link",
    "Linear_Link_Benoit",
    "Linear_Link_Sears",
    "Linear_Chain_Benoit",
    "Linear_Chain_Sears",
    "Linear_Yard_Sears",
    "Linear_Yard_Indian",
    "Linear_Fathom",
    "Linear_Mile_International_Nautical"
};

#define TIFF_ANGULAR_UNIT_OFFSET 9101
static const char *const tiff_angular_unit_codes[] = {
    "Angular_Radian",
    "Angular_Degree",
    "Angular_Arc_Minute",
    "Angular_Arc_Second",
    "Angular_Grad",
    "Angular_Gon",
    "Angular_DMS",
    "Angular_DMS_Hemisphere"
};

#define TIFF_GCS_TYPE_OFFSET 4201
static const char *const tiff_gcs_type_codes[] = {
    "GCS_Adindan",
    "GCS_AGD66",
    "GCS_AGD84",
    "GCS_Ain_el_Abd",
    "GCS_Afgooye",
    "GCS_Agadez",
    "GCS_Lisbon",
    "GCS_Aratu",
    "GCS_Arc_1950",
    "GCS_Arc_1960",
    "GCS_Batavia",
    "GCS_Barbados",
    "GCS_Beduaram",
    "GCS_Beijing_1954",
    "GCS_Belge_1950",
    "GCS_Bermuda_1957",
    "GCS_Bern_1898",
    "GCS_Bogota",
    "GCS_Bukit_Rimpah",
    "GCS_Camacupa",
    "GCS_Campo_Inchauspe",
    "GCS_Cape",
    "GCS_Carthage",
    "GCS_Chua",
    "GCS_Corrego_Alegre",
    "GCS_Cote_d_Ivoire",
    "GCS_Deir_ez_Zor",
    "GCS_Douala",
    "GCS_Egypt_1907",
    "GCS_ED50",
    "GCS_ED87",
    "GCS_Fahud",
    "GCS_Gandajika_1970",
    "GCS_Garoua",
    "GCS_Guyane_Francaise",
    "GCS_Hu_Tzu_Shan",
    "GCS_HD72",
    "GCS_ID74",
    "GCS_Indian_1954",
    "GCS_Indian_1975",
    "GCS_Jamaica_1875",
    "GCS_JAD69",
    "GCS_Kalianpur",
    "GCS_Kandawala",
    "GCS_Kertau",
    "GCS_KOC",
    "GCS_La_Canoa",
    "GCS_PSAD56",
    "GCS_Lake",
    "GCS_Leigon",
    "GCS_Liberia_1964",
    "GCS_Lome",
    "GCS_Luzon_1911",
    "GCS_Hito_XVIII_1963",
    "GCS_Herat_North",
    "GCS_Mahe_1971",
    "GCS_Makassar",
    "GCS_EUREF89",
    "GCS_Malongo_1987",
    "GCS_Manoca",
    "GCS_Merchich",
    "GCS_Massawa",
    "GCS_Minna",
    "GCS_Mhast",
    "GCS_Monte_Mario",
    "GCS_M_poraloko",
    "GCS_NAD27",
    "GCS_NAD_Michigan",
    "GCS_NAD83",
    "GCS_Nahrwan_1967",
    "GCS_Naparima_1972",
    "GCS_GD49",
    "GCS_NGO_1948",
    "GCS_Datum_73",
    "GCS_NTF",
    "GCS_NSWC_9Z_2",
    "GCS_OSGB_1936",
    "GCS_OSGB70",
    "GCS_OS_SN80",
    "GCS_Padang",
    "GCS_Palestine_1923",
    "GCS_Pointe_Noire",
    "GCS_GDA94",
    "GCS_Pulkovo_1942",
    "GCS_Qatar",
    "GCS_Qatar_1948",
    "GCS_Qornoq",
    "GCS_Loma_Quintana",
    "GCS_Amersfoort",
    "GCS_RT38",
    "GCS_SAD69",
    "GCS_Sapper_Hill_1943",
    "GCS_Schwarzeck",
    "GCS_Segora",
    "GCS_Serindung",
    "GCS_Sudan",
    "GCS_Tananarive",
    "GCS_Timbalai_1948",
    "GCS_TM65",
    "GCS_TM75",
    "GCS_Tokyo",
    "GCS_Trinidad_1903",
    "GCS_TC_1948",
    "GCS_Voirol_1875",
    "GCS_Voirol_Unifie",
    "GCS_Bern_1938",
    "GCS_Nord_Sahara_1959",
    "GCS_Stockholm_1938",
    "GCS_Yacare",
    "GCS_Yoff",
    "GCS_Zanderij",
    "GCS_MGI",
    "GCS_Belge_1972",
    "GCS_DHDN",
    "GCS_Conakry_1905",
    "GCS_WGS_72",
    "GCS_WGS_72BE",
    "GCS_WGS_84",
    "GCS_Bern_1898_Bern",
    "GCS_Bogota_Bogota",
    "GCS_Lisbon_Lisbon",
    "GCS_Makassar_Jakarta",
    "GCS_MGI_Ferro",
    "GCS_Monte_Mario_Rome",
    "GCS_NTF_Paris",
    "GCS_Padang_Jakarta",
    "GCS_Belge_1950_Brussels",
    "GCS_Tananarive_Paris",
    "GCS_Voirol_1875_Paris",
    "GCS_Voirol_Unifie_Paris",
    "GCS_Batavia_Jakarta",
    "GCS_ATF_Paris",
    "GCS_NDG_Paris"
};

#define TIFF_GCSE_TYPE_OFFSET 4001
static const char *const tiff_gcse_type_codes[] = {
    "GCSE_Airy1830",
    "GCSE_AiryModified1849",
    "GCSE_AustralianNationalSpheroid",
    "GCSE_Bessel1841",
    "GCSE_BesselModified",
    "GCSE_BesselNamibia",
    "GCSE_Clarke1858",
    "GCSE_Clarke1866",
    "GCSE_Clarke1866Michigan",
    "GCSE_Clarke1880_Benoit",
    "GCSE_Clarke1880_IGN",
    "GCSE_Clarke1880_RGS",
    "GCSE_Clarke1880_Arc",
    "GCSE_Clarke1880_SGA1922",
    "GCSE_Everest1830_1937Adjustment",
    "GCSE_Everest1830_1967Definition",
    "GCSE_Everest1830_1975Definition",
    "GCSE_Everest1830Modified",
    "GCSE_GRS1980",
    "GCSE_Helmert1906",
    "GCSE_IndonesianNationalSpheroid",
    "GCSE_International1924",
    "GCSE_International1967",
    "GCSE_Krassowsky1940",
    "GCSE_NWL9D",
    "GCSE_NWL10D",
    "GCSE_Plessis1817",
    "GCSE_Struve1860",
    "GCSE_WarOffice",
    "GCSE_WGS84",
    "GCSE_GEM10C",
    "GCSE_OSU86F",
    "GCSE_OSU91A",
    "GCSE_Clarke1880",
    "GCSE_Sphere"
};

#define TIFF_GEODETIC_DATUM_OFFSET 6201
static const char *const tiff_geodetic_datum_codes[] = {
    "Datum_Adindan",
    "Datum_Australian_Geodetic_Datum_1966",
    "Datum_Australian_Geodetic_Datum_1984",
    "Datum_Ain_el_Abd_1970",
    "Datum_Afgooye",
    "Datum_Agadez",
    "Datum_Lisbon",
    "Datum_Aratu",
    "Datum_Arc_1950",
    "Datum_Arc_1960",
    "Datum_Batavia",
    "Datum_Barbados",
    "Datum_Beduaram",
    "Datum_Beijing_1954",
    "Datum_Reseau_National_Belge_1950",
    "Datum_Bermuda_1957",
    "Datum_Bern_1898",
    "Datum_Bogota",
    "Datum_Bukit_Rimpah",
    "Datum_Camacupa",
    "Datum_Campo_Inchauspe",
    "Datum_Cape",
    "Datum_Carthage",
    "Datum_Chua",
    "Datum_Corrego_Alegre",
    "Datum_Cote_d_Ivoire",
    "Datum_Deir_ez_Zor",
    "Datum_Douala",
    "Datum_Egypt_1907",
    "Datum_European_Datum_1950",
    "Datum_European_Datum_1987",
    "Datum_Fahud",
    "Datum_Gandajika_1970",
    "Datum_Garoua",
    "Datum_Guyane_Francaise",
    "Datum_Hu_Tzu_Shan",
    "Datum_Hungarian_Datum_1972",
    "Datum_Indonesian_Datum_1974",
    "Datum_Indian_1954",
    "Datum_Indian_1975",
    "Datum_Jamaica_1875",
    "Datum_Jamaica_1969",
    "Datum_Kalianpur",
    "Datum_Kandawala",
    "Datum_Kertau",
    "Datum_Kuwait_Oil_Company",
    "Datum_La_Canoa",
    "Datum_Provisional_S_American_Datum_1956",
    "Datum_Lake",
    "Datum_Leigon",
    "Datum_Liberia_1964",
    "Datum_Lome",
    "Datum_Luzon_1911",
    "Datum_Hito_XVIII_1963",
    "Datum_Herat_North",
    "Datum_Mahe_1971",
    "Datum_Makassar",
    "Datum_European_Reference_System_1989",
    "Datum_Malongo_1987",
    "Datum_Manoca",
    "Datum_Merchich",
    "Datum_Massawa",
    "Datum_Minna",
    "Datum_Mhast",
    "Datum_Monte_Mario",
    "Datum_M_poraloko",
    "Datum_North_American_Datum_1927",
    "Datum_NAD_Michigan",
    "Datum_North_American_Datum_1983",
    "Datum_Nahrwan_1967",
    "Datum_Naparima_1972",
    "Datum_New_Zealand_Geodetic_Datum_1949",
    "Datum_NGO_1948",
    "Datum_Datum_73",
    "Datum_Nouvelle_Triangulation_Francaise",
    "Datum_NSWC_9Z_2",
    "Datum_OSGB_1936",
    "Datum_OSGB_1970_SN",
    "Datum_OS_SN_1980",
    "Datum_Padang_1884",
    "Datum_Palestine_1923",
    "Datum_Pointe_Noire",
    "Datum_Geocentric_Datum_of_Australia_1994",
    "Datum_Pulkovo_1942",
    "Datum_Qatar",
    "Datum_Qatar_1948",
    "Datum_Qornoq",
    "Datum_Loma_Quintana",
    "Datum_Amersfoort",
    "Datum_RT38",
    "Datum_South_American_Datum_1969",
    "Datum_Sapper_Hill_1943",
    "Datum_Schwarzeck",
    "Datum_Segora",
    "Datum_Serindung",
    "Datum_Sudan",
    "Datum_Tananarive_1925",
    "Datum_Timbalai_1948",
    "Datum_TM65",
    "Datum_TM75",
    "Datum_Tokyo",
    "Datum_Trinidad_1903",
    "Datum_Trucial_Coast_1948",
    "Datum_Voirol_1875",
    "Datum_Voirol_Unifie_1960",
    "Datum_Bern_1938",
    "Datum_Nord_Sahara_1959",
    "Datum_Stockholm_1938",
    "Datum_Yacare",
    "Datum_Yoff",
    "Datum_Zanderij",
    "Datum_Militar_Geographische_Institut",
    "Datum_Reseau_National_Belge_1972",
    "Datum_Deutsche_Hauptdreiecksnetz",
    "Datum_Conakry_1905",
    "Datum_WGS72",
    "Datum_WGS72_Transit_Broadcast_Ephemeris",
    "Datum_WGS84",
    "Datum_Ancienne_Triangulation_Francaise",
    "Datum_Nord_de_Guerre"
};

#define TIFF_GEODETIC_DATUM_E_OFFSET 6001
static const char *const tiff_geodetic_datum_e_codes[] = {
    "DatumE_Airy1830",
    "DatumE_AiryModified1849",
    "DatumE_AustralianNationalSpheroid",
    "DatumE_Bessel1841",
    "DatumE_BesselModified",
    "DatumE_BesselNamibia",
    "DatumE_Clarke1858",
    "DatumE_Clarke1866",
    "DatumE_Clarke1866Michigan",
    "DatumE_Clarke1880_Benoit",
    "DatumE_Clarke1880_IGN",
    "DatumE_Clarke1880_RGS",
    "DatumE_Clarke1880_Arc",
    "DatumE_Clarke1880_SGA1922",
    "DatumE_Everest1830_1937Adjustment",
    "DatumE_Everest1830_1967Definition",
    "DatumE_Everest1830_1975Definition",
    "DatumE_Everest1830Modified",
    "DatumE_GRS1980",
    "DatumE_Helmert1906",
    "DatumE_IndonesianNationalSpheroid",
    "DatumE_International1924",
    "DatumE_International1967",
    "DatumE_Krassowsky1960",
    "DatumE_NWL9D",
    "DatumE_NWL10D",
    "DatumE_Plessis1817",
    "DatumE_Struve1860",
    "DatumE_WarOffice",
    "DatumE_WGS84",
    "DatumE_GEM10C",
    "DatumE_OSU86F",
    "DatumE_OSU91A",
    "DatumE_Clarke1880",
    "DatumE_Sphere"
};

#define TIFF_ELLIPSOID_OFFSET 7001
static const char *const tiff_ellipsoid_codes[] = {
    "Ellipse_Airy_1830",
    "Ellipse_Airy_Modified_1849",
    "Ellipse_Australian_National_Spheroid",
    "Ellipse_Bessel_1841",
    "Ellipse_Bessel_Modified",
    "Ellipse_Bessel_Namibia",
    "Ellipse_Clarke_1858",
    "Ellipse_Clarke_1866",
    "Ellipse_Clarke_1866_Michigan",
    "Ellipse_Clarke_1880_Benoit",
    "Ellipse_Clarke_1880_IGN",
    "Ellipse_Clarke_1880_RGS",
    "Ellipse_Clarke_1880_Arc",
    "Ellipse_Clarke_1880_SGA_1922",
    "Ellipse_Everest_1830_1937_Adjustment",
    "Ellipse_Everest_1830_1967_Definition",
    "Ellipse_Everest_1830_1975_Definition",
    "Ellipse_Everest_1830_Modified",
    "Ellipse_GRS_1980",
    "Ellipse_Helmert_1906",
    "Ellipse_Indonesian_National_Spheroid",
    "Ellipse_International_1924",
    "Ellipse_International_1967",
    "Ellipse_Krassowsky_1940",
    "Ellipse_NWL_9D",
    "Ellipse_NWL_10D",
    "Ellipse_Plessis_1817",
    "Ellipse_Struve_1860",
    "Ellipse_War_Office",
    "Ellipse_WGS_84",
    "Ellipse_GEM_10C",
    "Ellipse_OSU86F",
    "Ellipse_OSU91A",
    "Ellipse_Clarke_1880",
    "Ellipse_Sphere"
};

#define TIFF_PRIME_MERIDIAN_OFFSET 8901
static const char *const tiff_prime_meridian_codes[] = {
    "PM_Greenwich",
    "PM_Lisbon",
    "PM_Paris",
    "PM_Bogota",
    "PM_Madrid",
    "PM_Rome",
    "PM_Bern",
    "PM_Jakarta",
    "PM_Ferro",
    "PM_Brussels",
    "PM_Stockholm"
};

static const TiffGeoTagKeyName tiff_proj_cs_type_codes[] = {
    {20137, "PCS_Adindan_UTM_zone_37N"},
    {20138, "PCS_Adindan_UTM_zone_38N"},
    {20248, "PCS_AGD66_AMG_zone_48"},
    {20249, "PCS_AGD66_AMG_zone_49"},
    {20250, "PCS_AGD66_AMG_zone_50"},
    {20251, "PCS_AGD66_AMG_zone_51"},
    {20252, "PCS_AGD66_AMG_zone_52"},
    {20253, "PCS_AGD66_AMG_zone_53"},
    {20254, "PCS_AGD66_AMG_zone_54"},
    {20255, "PCS_AGD66_AMG_zone_55"},
    {20256, "PCS_AGD66_AMG_zone_56"},
    {20257, "PCS_AGD66_AMG_zone_57"},
    {20258, "PCS_AGD66_AMG_zone_58"},
    {20348, "PCS_AGD84_AMG_zone_48"},
    {20349, "PCS_AGD84_AMG_zone_49"},
    {20350, "PCS_AGD84_AMG_zone_50"},
    {20351, "PCS_AGD84_AMG_zone_51"},
    {20352, "PCS_AGD84_AMG_zone_52"},
    {20353, "PCS_AGD84_AMG_zone_53"},
    {20354, "PCS_AGD84_AMG_zone_54"},
    {20355, "PCS_AGD84_AMG_zone_55"},
    {20356, "PCS_AGD84_AMG_zone_56"},
    {20357, "PCS_AGD84_AMG_zone_57"},
    {20358, "PCS_AGD84_AMG_zone_58"},
    {20437, "PCS_Ain_el_Abd_UTM_zone_37N"},
    {20438, "PCS_Ain_el_Abd_UTM_zone_38N"},
    {20439, "PCS_Ain_el_Abd_UTM_zone_39N"},
    {20499, "PCS_Ain_el_Abd_Bahrain_Grid"},
    {20538, "PCS_Afgooye_UTM_zone_38N"},
    {20539, "PCS_Afgooye_UTM_zone_39N"},
    {20700, "PCS_Lisbon_Portugese_Grid"},
    {20822, "PCS_Aratu_UTM_zone_22S"},
    {20823, "PCS_Aratu_UTM_zone_23S"},
    {20824, "PCS_Aratu_UTM_zone_24S"},
    {20973, "PCS_Arc_1950_Lo13"},
    {20975, "PCS_Arc_1950_Lo15"},
    {20977, "PCS_Arc_1950_Lo17"},
    {20979, "PCS_Arc_1950_Lo19"},
    {20981, "PCS_Arc_1950_Lo21"},
    {20983, "PCS_Arc_1950_Lo23"},
    {20985, "PCS_Arc_1950_Lo25"},
    {20987, "PCS_Arc_1950_Lo27"},
    {20989, "PCS_Arc_1950_Lo29"},
    {20991, "PCS_Arc_1950_Lo31"},
    {20993, "PCS_Arc_1950_Lo33"},
    {20995, "PCS_Arc_1950_Lo35"},
    {21100, "PCS_Batavia_NEIEZ"},
    {21148, "PCS_Batavia_UTM_zone_48S"},
    {21149, "PCS_Batavia_UTM_zone_49S"},
    {21150, "PCS_Batavia_UTM_zone_50S"},
    {21413, "PCS_Beijing_Gauss_zone_13"},
    {21414, "PCS_Beijing_Gauss_zone_14"},
    {21415, "PCS_Beijing_Gauss_zone_15"},
    {21416, "PCS_Beijing_Gauss_zone_16"},
    {21417, "PCS_Beijing_Gauss_zone_17"},
    {21418, "PCS_Beijing_Gauss_zone_18"},
    {21419, "PCS_Beijing_Gauss_zone_19"},
    {21420, "PCS_Beijing_Gauss_zone_20"},
    {21421, "PCS_Beijing_Gauss_zone_21"},
    {21422, "PCS_Beijing_Gauss_zone_22"},
    {21423, "PCS_Beijing_Gauss_zone_23"},
    {21473, "PCS_Beijing_Gauss_13N"},
    {21474, "PCS_Beijing_Gauss_14N"},
    {21475, "PCS_Beijing_Gauss_15N"},
    {21476, "PCS_Beijing_Gauss_16N"},
    {21477, "PCS_Beijing_Gauss_17N"},
    {21478, "PCS_Beijing_Gauss_18N"},
    {21479, "PCS_Beijing_Gauss_19N"},
    {21480, "PCS_Beijing_Gauss_20N"},
    {21481, "PCS_Beijing_Gauss_21N"},
    {21482, "PCS_Beijing_Gauss_22N"},
    {21483, "PCS_Beijing_Gauss_23N"},
    {21500, "PCS_Belge_Lambert_50"},
    {21790, "PCS_Bern_1898_Swiss_Old"},
    {21817, "PCS_Bogota_UTM_zone_17N"},
    {21818, "PCS_Bogota_UTM_zone_18N"},
    {21891, "PCS_Bogota_Colombia_3W"},
    {21892, "PCS_Bogota_Colombia_Bogota"},
    {21893, "PCS_Bogota_Colombia_3E"},
    {21894, "PCS_Bogota_Colombia_6E"},
    {22032, "PCS_Camacupa_UTM_32S"},
    {22033, "PCS_Camacupa_UTM_33S"},
    {22191, "PCS_C_Inchauspe_Argentina_1"},
    {22192, "PCS_C_Inchauspe_Argentina_2"},
    {22193, "PCS_C_Inchauspe_Argentina_3"},
    {22194, "PCS_C_Inchauspe_Argentina_4"},
    {22195, "PCS_C_Inchauspe_Argentina_5"},
    {22196, "PCS_C_Inchauspe_Argentina_6"},
    {22197, "PCS_C_Inchauspe_Argentina_7"},
    {22332, "PCS_Carthage_UTM_zone_32N"},
    {22391, "PCS_Carthage_Nord_Tunisie"},
    {22392, "PCS_Carthage_Sud_Tunisie"},
    {22523, "PCS_Corrego_Alegre_UTM_23S"},
    {22524, "PCS_Corrego_Alegre_UTM_24S"},
    {22832, "PCS_Douala_UTM_zone_32N"},
    {22992, "PCS_Egypt_1907_Red_Belt"},
    {22993, "PCS_Egypt_1907_Purple_Belt"},
    {22994, "PCS_Egypt_1907_Ext_Purple"},
    {23028, "PCS_ED50_UTM_zone_28N"},
    {23029, "PCS_ED50_UTM_zone_29N"},
    {23030, "PCS_ED50_UTM_zone_30N"},
    {23031, "PCS_ED50_UTM_zone_31N"},
    {23032, "PCS_ED50_UTM_zone_32N"},
    {23033, "PCS_ED50_UTM_zone_33N"},
    {23034, "PCS_ED50_UTM_zone_34N"},
    {23035, "PCS_ED50_UTM_zone_35N"},
    {23036, "PCS_ED50_UTM_zone_36N"},
    {23037, "PCS_ED50_UTM_zone_37N"},
    {23038, "PCS_ED50_UTM_zone_38N"},
    {23239, "PCS_Fahud_UTM_zone_39N"},
    {23240, "PCS_Fahud_UTM_zone_40N"},
    {23433, "PCS_Garoua_UTM_zone_33N"},
    {23846, "PCS_ID74_UTM_zone_46N"},
    {23847, "PCS_ID74_UTM_zone_47N"},
    {23848, "PCS_ID74_UTM_zone_48N"},
    {23849, "PCS_ID74_UTM_zone_49N"},
    {23850, "PCS_ID74_UTM_zone_50N"},
    {23851, "PCS_ID74_UTM_zone_51N"},
    {23852, "PCS_ID74_UTM_zone_52N"},
    {23853, "PCS_ID74_UTM_zone_53N"},
    {23886, "PCS_ID74_UTM_zone_46S"},
    {23887, "PCS_ID74_UTM_zone_47S"},
    {23888, "PCS_ID74_UTM_zone_48S"},
    {23889, "PCS_ID74_UTM_zone_49S"},
    {23890, "PCS_ID74_UTM_zone_50S"},
    {23891, "PCS_ID74_UTM_zone_51S"},
    {23892, "PCS_ID74_UTM_zone_52S"},
    {23893, "PCS_ID74_UTM_zone_53S"},
    {23894, "PCS_ID74_UTM_zone_54S"},
    {23947, "PCS_Indian_1954_UTM_47N"},
    {23948, "PCS_Indian_1954_UTM_48N"},
    {24047, "PCS_Indian_1975_UTM_47N"},
    {24048, "PCS_Indian_1975_UTM_48N"},
    {24100, "PCS_Jamaica_1875_Old_Grid"},
    {24200, "PCS_JAD69_Jamaica_Grid"},
    {24370, "PCS_Kalianpur_India_0"},
    {24371, "PCS_Kalianpur_India_I"},
    {24372, "PCS_Kalianpur_India_IIa"},
    {24373, "PCS_Kalianpur_India_IIIa"},
    {24374, "PCS_Kalianpur_India_IVa"},
    {24382, "PCS_Kalianpur_India_IIb"},
    {24383, "PCS_Kalianpur_India_IIIb"},
    {24384, "PCS_Kalianpur_India_IVb"},
    {24500, "PCS_Kertau_Singapore_Grid"},
    {24547, "PCS_Kertau_UTM_zone_47N"},
    {24548, "PCS_Kertau_UTM_zone_48N"},
    {24720, "PCS_La_Canoa_UTM_zone_20N"},
    {24721, "PCS_La_Canoa_UTM_zone_21N"},
    {24818, "PCS_PSAD56_UTM_zone_18N"},
    {24819, "PCS_PSAD56_UTM_zone_19N"},
    {24820, "PCS_PSAD56_UTM_zone_20N"},
    {24821, "PCS_PSAD56_UTM_zone_21N"},
    {24877, "PCS_PSAD56_UTM_zone_17S"},
    {24878, "PCS_PSAD56_UTM_zone_18S"},
    {24879, "PCS_PSAD56_UTM_zone_19S"},
    {24880, "PCS_PSAD56_UTM_zone_20S"},
    {24891, "PCS_PSAD56_Peru_west_zone"},
    {24892, "PCS_PSAD56_Peru_central"},
    {24893, "PCS_PSAD56_Peru_east_zone"},
    {25000, "PCS_Leigon_Ghana_Grid"},
    {25231, "PCS_Lome_UTM_zone_31N"},
    {25391, "PCS_Luzon_Philippines_I"},
    {25392, "PCS_Luzon_Philippines_II"},
    {25393, "PCS_Luzon_Philippines_III"},
    {25394, "PCS_Luzon_Philippines_IV"},
    {25395, "PCS_Luzon_Philippines_V"},
    {25700, "PCS_Makassar_NEIEZ"},
    {25932, "PCS_Malongo_1987_UTM_32S"},
    {26191, "PCS_Merchich_Nord_Maroc"},
    {26192, "PCS_Merchich_Sud_Maroc"},
    {26193, "PCS_Merchich_Sahara"},
    {26237, "PCS_Massawa_UTM_zone_37N"},
    {26331, "PCS_Minna_UTM_zone_31N"},
    {26332, "PCS_Minna_UTM_zone_32N"},
    {26391, "PCS_Minna_Nigeria_West"},
    {26392, "PCS_Minna_Nigeria_Mid_Belt"},
    {26393, "PCS_Minna_Nigeria_East"},
    {26432, "PCS_Mhast_UTM_zone_32S"},
    {26591, "PCS_Monte_Mario_Italy_1"},
    {26592, "PCS_Monte_Mario_Italy_2"},
    {26632, "PCS_M_poraloko_UTM_32N"},
    {26692, "PCS_M_poraloko_UTM_32S"},
    {26703, "PCS_NAD27_UTM_zone_3N"},
    {26704, "PCS_NAD27_UTM_zone_4N"},
    {26705, "PCS_NAD27_UTM_zone_5N"},
    {26706, "PCS_NAD27_UTM_zone_6N"},
    {26707, "PCS_NAD27_UTM_zone_7N"},
    {26708, "PCS_NAD27_UTM_zone_8N"},
    {26709, "PCS_NAD27_UTM_zone_9N"},
    {26710, "PCS_NAD27_UTM_zone_10N"},
    {26711, "PCS_NAD27_UTM_zone_11N"},
    {26712, "PCS_NAD27_UTM_zone_12N"},
    {26713, "PCS_NAD27_UTM_zone_13N"},
    {26714, "PCS_NAD27_UTM_zone_14N"},
    {26715, "PCS_NAD27_UTM_zone_15N"},
    {26716, "PCS_NAD27_UTM_zone_16N"},
    {26717, "PCS_NAD27_UTM_zone_17N"},
    {26718, "PCS_NAD27_UTM_zone_18N"},
    {26719, "PCS_NAD27_UTM_zone_19N"},
    {26720, "PCS_NAD27_UTM_zone_20N"},
    {26721, "PCS_NAD27_UTM_zone_21N"},
    {26722, "PCS_NAD27_UTM_zone_22N"},
    {26729, "PCS_NAD27_Alabama_East"},
    {26730, "PCS_NAD27_Alabama_West"},
    {26731, "PCS_NAD27_Alaska_zone_1"},
    {26732, "PCS_NAD27_Alaska_zone_2"},
    {26733, "PCS_NAD27_Alaska_zone_3"},
    {26734, "PCS_NAD27_Alaska_zone_4"},
    {26735, "PCS_NAD27_Alaska_zone_5"},
    {26736, "PCS_NAD27_Alaska_zone_6"},
    {26737, "PCS_NAD27_Alaska_zone_7"},
    {26738, "PCS_NAD27_Alaska_zone_8"},
    {26739, "PCS_NAD27_Alaska_zone_9"},
    {26740, "PCS_NAD27_Alaska_zone_10"},
    {26741, "PCS_NAD27_California_I"},
    {26742, "PCS_NAD27_California_II"},
    {26743, "PCS_NAD27_California_III"},
    {26744, "PCS_NAD27_California_IV"},
    {26745, "PCS_NAD27_California_V"},
    {26746, "PCS_NAD27_California_VI"},
    {26747, "PCS_NAD27_California_VII"},
    {26748, "PCS_NAD27_Arizona_East"},
    {26749, "PCS_NAD27_Arizona_Central"},
    {26750, "PCS_NAD27_Arizona_West"},
    {26751, "PCS_NAD27_Arkansas_North"},
    {26752, "PCS_NAD27_Arkansas_South"},
    {26753, "PCS_NAD27_Colorado_North"},
    {26754, "PCS_NAD27_Colorado_Central"},
    {26755, "PCS_NAD27_Colorado_South"},
    {26756, "PCS_NAD27_Connecticut"},
    {26757, "PCS_NAD27_Delaware"},
    {26758, "PCS_NAD27_Florida_East"},
    {26759, "PCS_NAD27_Florida_West"},
    {26760, "PCS_NAD27_Florida_North"},
    {26761, "PCS_NAD27_Hawaii_zone_1"},
    {26762, "PCS_NAD27_Hawaii_zone_2"},
    {26763, "PCS_NAD27_Hawaii_zone_3"},
    {26764, "PCS_NAD27_Hawaii_zone_4"},
    {26765, "PCS_NAD27_Hawaii_zone_5"},
    {26766, "PCS_NAD27_Georgia_East"},
    {26767, "PCS_NAD27_Georgia_West"},
    {26768, "PCS_NAD27_Idaho_East"},
    {26769, "PCS_NAD27_Idaho_Central"},
    {26770, "PCS_NAD27_Idaho_West"},
    {26771, "PCS_NAD27_Illinois_East"},
    {26772, "PCS_NAD27_Illinois_West"},
    {26773, "PCS_NAD27_Indiana_East"},
    {26774, "PCS_NAD27_Indiana_West"},
    {26775, "PCS_NAD27_Iowa_North"},
    {26776, "PCS_NAD27_Iowa_South"},
    {26777, "PCS_NAD27_Kansas_North"},
    {26778, "PCS_NAD27_Kansas_South"},
    {26779, "PCS_NAD27_Kentucky_North"},
    {26780, "PCS_NAD27_Kentucky_South"},
    {26781, "PCS_NAD27_Louisiana_North"},
    {26782, "PCS_NAD27_Louisiana_South"},
    {26783, "PCS_NAD27_Maine_East"},
    {26784, "PCS_NAD27_Maine_West"},
    {26785, "PCS_NAD27_Maryland"},
    {26786, "PCS_NAD27_Massachusetts"},
    {26787, "PCS_NAD27_Massachusetts_Is"},
    {26788, "PCS_NAD27_Michigan_North"},
    {26789, "PCS_NAD27_Michigan_Central"},
    {26790, "PCS_NAD27_Michigan_South"},
    {26791, "PCS_NAD27_Minnesota_North"},
    {26792, "PCS_NAD27_Minnesota_Cent"},
    {26793, "PCS_NAD27_Minnesota_South"},
    {26794, "PCS_NAD27_Mississippi_East"},
    {26795, "PCS_NAD27_Mississippi_West"},
    {26796, "PCS_NAD27_Missouri_East"},
    {26797, "PCS_NAD27_Missouri_Central"},
    {26798, "PCS_NAD27_Missouri_West"},
    {26801, "PCS_NAD_Michigan_Michigan_East"},
    {26802, "PCS_NAD_Michigan_Michigan_Old_Central"},
    {26803, "PCS_NAD_Michigan_Michigan_West"},
    {26903, "PCS_NAD83_UTM_zone_3N"},
    {26904, "PCS_NAD83_UTM_zone_4N"},
    {26905, "PCS_NAD83_UTM_zone_5N"},
    {26906, "PCS_NAD83_UTM_zone_6N"},
    {26907, "PCS_NAD83_UTM_zone_7N"},
    {26908, "PCS_NAD83_UTM_zone_8N"},
    {26909, "PCS_NAD83_UTM_zone_9N"},
    {26910, "PCS_NAD83_UTM_zone_10N"},
    {26911, "PCS_NAD83_UTM_zone_11N"},
    {26912, "PCS_NAD83_UTM_zone_12N"},
    {26913, "PCS_NAD83_UTM_zone_13N"},
    {26914, "PCS_NAD83_UTM_zone_14N"},
    {26915, "PCS_NAD83_UTM_zone_15N"},
    {26916, "PCS_NAD83_UTM_zone_16N"},
    {26917, "PCS_NAD83_UTM_zone_17N"},
    {26918, "PCS_NAD83_UTM_zone_18N"},
    {26919, "PCS_NAD83_UTM_zone_19N"},
    {26920, "PCS_NAD83_UTM_zone_20N"},
    {26921, "PCS_NAD83_UTM_zone_21N"},
    {26922, "PCS_NAD83_UTM_zone_22N"},
    {26923, "PCS_NAD83_UTM_zone_23N"},
    {26929, "PCS_NAD83_Alabama_East"},
    {26930, "PCS_NAD83_Alabama_West"},
    {26931, "PCS_NAD83_Alaska_zone_1"},
    {26932, "PCS_NAD83_Alaska_zone_2"},
    {26933, "PCS_NAD83_Alaska_zone_3"},
    {26934, "PCS_NAD83_Alaska_zone_4"},
    {26935, "PCS_NAD83_Alaska_zone_5"},
    {26936, "PCS_NAD83_Alaska_zone_6"},
    {26937, "PCS_NAD83_Alaska_zone_7"},
    {26938, "PCS_NAD83_Alaska_zone_8"},
    {26939, "PCS_NAD83_Alaska_zone_9"},
    {26940, "PCS_NAD83_Alaska_zone_10"},
    {26941, "PCS_NAD83_California_1"},
    {26942, "PCS_NAD83_California_2"},
    {26943, "PCS_NAD83_California_3"},
    {26944, "PCS_NAD83_California_4"},
    {26945, "PCS_NAD83_California_5"},
    {26946, "PCS_NAD83_California_6"},
    {26948, "PCS_NAD83_Arizona_East"},
    {26949, "PCS_NAD83_Arizona_Central"},
    {26950, "PCS_NAD83_Arizona_West"},
    {26951, "PCS_NAD83_Arkansas_North"},
    {26952, "PCS_NAD83_Arkansas_South"},
    {26953, "PCS_NAD83_Colorado_North"},
    {26954, "PCS_NAD83_Colorado_Central"},
    {26955, "PCS_NAD83_Colorado_South"},
    {26956, "PCS_NAD83_Connecticut"},
    {26957, "PCS_NAD83_Delaware"},
    {26958, "PCS_NAD83_Florida_East"},
    {26959, "PCS_NAD83_Florida_West"},
    {26960, "PCS_NAD83_Florida_North"},
    {26961, "PCS_NAD83_Hawaii_zone_1"},
    {26962, "PCS_NAD83_Hawaii_zone_2"},
    {26963, "PCS_NAD83_Hawaii_zone_3"},
    {26964, "PCS_NAD83_Hawaii_zone_4"},
    {26965, "PCS_NAD83_Hawaii_zone_5"},
    {26966, "PCS_NAD83_Georgia_East"},
    {26967, "PCS_NAD83_Georgia_West"},
    {26968, "PCS_NAD83_Idaho_East"},
    {26969, "PCS_NAD83_Idaho_Central"},
    {26970, "PCS_NAD83_Idaho_West"},
    {26971, "PCS_NAD83_Illinois_East"},
    {26972, "PCS_NAD83_Illinois_West"},
    {26973, "PCS_NAD83_Indiana_East"},
    {26974, "PCS_NAD83_Indiana_West"},
    {26975, "PCS_NAD83_Iowa_North"},
    {26976, "PCS_NAD83_Iowa_South"},
    {26977, "PCS_NAD83_Kansas_North"},
    {26978, "PCS_NAD83_Kansas_South"},
    {26979, "PCS_NAD83_Kentucky_North"},
    {26980, "PCS_NAD83_Kentucky_South"},
    {26981, "PCS_NAD83_Louisiana_North"},
    {26982, "PCS_NAD83_Louisiana_South"},
    {26983, "PCS_NAD83_Maine_East"},
    {26984, "PCS_NAD83_Maine_West"},
    {26985, "PCS_NAD83_Maryland"},
    {26986, "PCS_NAD83_Massachusetts"},
    {26987, "PCS_NAD83_Massachusetts_Is"},
    {26988, "PCS_NAD83_Michigan_North"},
    {26989, "PCS_NAD83_Michigan_Central"},
    {26990, "PCS_NAD83_Michigan_South"},
    {26991, "PCS_NAD83_Minnesota_North"},
    {26992, "PCS_NAD83_Minnesota_Cent"},
    {26993, "PCS_NAD83_Minnesota_South"},
    {26994, "PCS_NAD83_Mississippi_East"},
    {26995, "PCS_NAD83_Mississippi_West"},
    {26996, "PCS_NAD83_Missouri_East"},
    {26997, "PCS_NAD83_Missouri_Central"},
    {26998, "PCS_NAD83_Missouri_West"},
    {27038, "PCS_Nahrwan_1967_UTM_38N"},
    {27039, "PCS_Nahrwan_1967_UTM_39N"},
    {27040, "PCS_Nahrwan_1967_UTM_40N"},
    {27120, "PCS_Naparima_UTM_20N"},
    {27200, "PCS_GD49_NZ_Map_Grid"},
    {27291, "PCS_GD49_North_Island_Grid"},
    {27292, "PCS_GD49_South_Island_Grid"},
    {27429, "PCS_Datum_73_UTM_zone_29N"},
    {27500, "PCS_ATF_Nord_de_Guerre"},
    {27581, "PCS_NTF_France_I"},
    {27582, "PCS_NTF_France_II"},
    {27583, "PCS_NTF_France_III"},
    {27591, "PCS_NTF_Nord_France"},
    {27592, "PCS_NTF_Centre_France"},
    {27593, "PCS_NTF_Sud_France"},
    {27700, "PCS_British_National_Grid"},
    {28232, "PCS_Point_Noire_UTM_32S"},
    {28348, "PCS_GDA94_MGA_zone_48"},
    {28349, "PCS_GDA94_MGA_zone_49"},
    {28350, "PCS_GDA94_MGA_zone_50"},
    {28351, "PCS_GDA94_MGA_zone_51"},
    {28352, "PCS_GDA94_MGA_zone_52"},
    {28353, "PCS_GDA94_MGA_zone_53"},
    {28354, "PCS_GDA94_MGA_zone_54"},
    {28355, "PCS_GDA94_MGA_zone_55"},
    {28356, "PCS_GDA94_MGA_zone_56"},
    {28357, "PCS_GDA94_MGA_zone_57"},
    {28358, "PCS_GDA94_MGA_zone_58"},
    {28404, "PCS_Pulkovo_Gauss_zone_4"},
    {28405, "PCS_Pulkovo_Gauss_zone_5"},
    {28406, "PCS_Pulkovo_Gauss_zone_6"},
    {28407, "PCS_Pulkovo_Gauss_zone_7"},
    {28408, "PCS_Pulkovo_Gauss_zone_8"},
    {28409, "PCS_Pulkovo_Gauss_zone_9"},
    {28410, "PCS_Pulkovo_Gauss_zone_10"},
    {28411, "PCS_Pulkovo_Gauss_zone_11"},
    {28412, "PCS_Pulkovo_Gauss_zone_12"},
    {28413, "PCS_Pulkovo_Gauss_zone_13"},
    {28414, "PCS_Pulkovo_Gauss_zone_14"},
    {28415, "PCS_Pulkovo_Gauss_zone_15"},
    {28416, "PCS_Pulkovo_Gauss_zone_16"},
    {28417, "PCS_Pulkovo_Gauss_zone_17"},
    {28418, "PCS_Pulkovo_Gauss_zone_18"},
    {28419, "PCS_Pulkovo_Gauss_zone_19"},
    {28420, "PCS_Pulkovo_Gauss_zone_20"},
    {28421, "PCS_Pulkovo_Gauss_zone_21"},
    {28422, "PCS_Pulkovo_Gauss_zone_22"},
    {28423, "PCS_Pulkovo_Gauss_zone_23"},
    {28424, "PCS_Pulkovo_Gauss_zone_24"},
    {28425, "PCS_Pulkovo_Gauss_zone_25"},
    {28426, "PCS_Pulkovo_Gauss_zone_26"},
    {28427, "PCS_Pulkovo_Gauss_zone_27"},
    {28428, "PCS_Pulkovo_Gauss_zone_28"},
    {28429, "PCS_Pulkovo_Gauss_zone_29"},
    {28430, "PCS_Pulkovo_Gauss_zone_30"},
    {28431, "PCS_Pulkovo_Gauss_zone_31"},
    {28432, "PCS_Pulkovo_Gauss_zone_32"},
    {28464, "PCS_Pulkovo_Gauss_4N"},
    {28465, "PCS_Pulkovo_Gauss_5N"},
    {28466, "PCS_Pulkovo_Gauss_6N"},
    {28467, "PCS_Pulkovo_Gauss_7N"},
    {28468, "PCS_Pulkovo_Gauss_8N"},
    {28469, "PCS_Pulkovo_Gauss_9N"},
    {28470, "PCS_Pulkovo_Gauss_10N"},
    {28471, "PCS_Pulkovo_Gauss_11N"},
    {28472, "PCS_Pulkovo_Gauss_12N"},
    {28473, "PCS_Pulkovo_Gauss_13N"},
    {28474, "PCS_Pulkovo_Gauss_14N"},
    {28475, "PCS_Pulkovo_Gauss_15N"},
    {28476, "PCS_Pulkovo_Gauss_16N"},
    {28477, "PCS_Pulkovo_Gauss_17N"},
    {28478, "PCS_Pulkovo_Gauss_18N"},
    {28479, "PCS_Pulkovo_Gauss_19N"},
    {28480, "PCS_Pulkovo_Gauss_20N"},
    {28481, "PCS_Pulkovo_Gauss_21N"},
    {28482, "PCS_Pulkovo_Gauss_22N"},
    {28483, "PCS_Pulkovo_Gauss_23N"},
    {28484, "PCS_Pulkovo_Gauss_24N"},
    {28485, "PCS_Pulkovo_Gauss_25N"},
    {28486, "PCS_Pulkovo_Gauss_26N"},
    {28487, "PCS_Pulkovo_Gauss_27N"},
    {28488, "PCS_Pulkovo_Gauss_28N"},
    {28489, "PCS_Pulkovo_Gauss_29N"},
    {28490, "PCS_Pulkovo_Gauss_30N"},
    {28491, "PCS_Pulkovo_Gauss_31N"},
    {28492, "PCS_Pulkovo_Gauss_32N"},
    {28600, "PCS_Qatar_National_Grid"},
    {28991, "PCS_RD_Netherlands_Old"},
    {28992, "PCS_RD_Netherlands_New"},
    {29118, "PCS_SAD69_UTM_zone_18N"},
    {29119, "PCS_SAD69_UTM_zone_19N"},
    {29120, "PCS_SAD69_UTM_zone_20N"},
    {29121, "PCS_SAD69_UTM_zone_21N"},
    {29122, "PCS_SAD69_UTM_zone_22N"},
    {29177, "PCS_SAD69_UTM_zone_17S"},
    {29178, "PCS_SAD69_UTM_zone_18S"},
    {29179, "PCS_SAD69_UTM_zone_19S"},
    {29180, "PCS_SAD69_UTM_zone_20S"},
    {29181, "PCS_SAD69_UTM_zone_21S"},
    {29182, "PCS_SAD69_UTM_zone_22S"},
    {29183, "PCS_SAD69_UTM_zone_23S"},
    {29184, "PCS_SAD69_UTM_zone_24S"},
    {29185, "PCS_SAD69_UTM_zone_25S"},
    {29220, "PCS_Sapper_Hill_UTM_20S"},
    {29221, "PCS_Sapper_Hill_UTM_21S"},
    {29333, "PCS_Schwarzeck_UTM_33S"},
    {29635, "PCS_Sudan_UTM_zone_35N"},
    {29636, "PCS_Sudan_UTM_zone_36N"},
    {29700, "PCS_Tananarive_Laborde"},
    {29738, "PCS_Tananarive_UTM_38S"},
    {29739, "PCS_Tananarive_UTM_39S"},
    {29800, "PCS_Timbalai_1948_Borneo"},
    {29849, "PCS_Timbalai_1948_UTM_49N"},
    {29850, "PCS_Timbalai_1948_UTM_50N"},
    {29900, "PCS_TM65_Irish_Nat_Grid"},
    {30200, "PCS_Trinidad_1903_Trinidad"},
    {30339, "PCS_TC_1948_UTM_zone_39N"},
    {30340, "PCS_TC_1948_UTM_zone_40N"},
    {30491, "PCS_Voirol_N_Algerie_ancien"},
    {30492, "PCS_Voirol_S_Algerie_ancien"},
    {30591, "PCS_Voirol_Unifie_N_Algerie"},
    {30592, "PCS_Voirol_Unifie_S_Algerie"},
    {30600, "PCS_Bern_1938_Swiss_New"},
    {30729, "PCS_Nord_Sahara_UTM_29N"},
    {30730, "PCS_Nord_Sahara_UTM_30N"},
    {30731, "PCS_Nord_Sahara_UTM_31N"},
    {30732, "PCS_Nord_Sahara_UTM_32N"},
    {31028, "PCS_Yoff_UTM_zone_28N"},
    {31121, "PCS_Zanderij_UTM_zone_21N"},
    {31291, "PCS_MGI_Austria_West"},
    {31292, "PCS_MGI_Austria_Central"},
    {31293, "PCS_MGI_Austria_East"},
    {31300, "PCS_Belge_Lambert_72"},
    {31491, "PCS_DHDN_Germany_zone_1"},
    {31492, "PCS_DHDN_Germany_zone_2"},
    {31493, "PCS_DHDN_Germany_zone_3"},
    {31494, "PCS_DHDN_Germany_zone_4"},
    {31495, "PCS_DHDN_Germany_zone_5"},
    {32001, "PCS_NAD27_Montana_North"},
    {32002, "PCS_NAD27_Montana_Central"},
    {32003, "PCS_NAD27_Montana_South"},
    {32005, "PCS_NAD27_Nebraska_North"},
    {32006, "PCS_NAD27_Nebraska_South"},
    {32007, "PCS_NAD27_Nevada_East"},
    {32008, "PCS_NAD27_Nevada_Central"},
    {32009, "PCS_NAD27_Nevada_West"},
    {32010, "PCS_NAD27_New_Hampshire"},
    {32011, "PCS_NAD27_New_Jersey"},
    {32012, "PCS_NAD27_New_Mexico_East"},
    {32013, "PCS_NAD27_New_Mexico_Cent"},
    {32014, "PCS_NAD27_New_Mexico_West"},
    {32015, "PCS_NAD27_New_York_East"},
    {32016, "PCS_NAD27_New_York_Central"},
    {32017, "PCS_NAD27_New_York_West"},
    {32018, "PCS_NAD27_New_York_Long_Is"},
    {32019, "PCS_NAD27_North_Carolina"},
    {32020, "PCS_NAD27_North_Dakota_N"},
    {32021, "PCS_NAD27_North_Dakota_S"},
    {32022, "PCS_NAD27_Ohio_North"},
    {32023, "PCS_NAD27_Ohio_South"},
    {32024, "PCS_NAD27_Oklahoma_North"},
    {32025, "PCS_NAD27_Oklahoma_South"},
    {32026, "PCS_NAD27_Oregon_North"},
    {32027, "PCS_NAD27_Oregon_South"},
    {32028, "PCS_NAD27_Pennsylvania_N"},
    {32029, "PCS_NAD27_Pennsylvania_S"},
    {32030, "PCS_NAD27_Rhode_Island"},
    {32031, "PCS_NAD27_South_Carolina_N"},
    {32033, "PCS_NAD27_South_Carolina_S"},
    {32034, "PCS_NAD27_South_Dakota_N"},
    {32035, "PCS_NAD27_South_Dakota_S"},
    {32036, "PCS_NAD27_Tennessee"},
    {32037, "PCS_NAD27_Texas_North"},
    {32038, "PCS_NAD27_Texas_North_Cen"},
    {32039, "PCS_NAD27_Texas_Central"},
    {32040, "PCS_NAD27_Texas_South_Cen"},
    {32041, "PCS_NAD27_Texas_South"},
    {32042, "PCS_NAD27_Utah_North"},
    {32043, "PCS_NAD27_Utah_Central"},
    {32044, "PCS_NAD27_Utah_South"},
    {32045, "PCS_NAD27_Vermont"},
    {32046, "PCS_NAD27_Virginia_North"},
    {32047, "PCS_NAD27_Virginia_South"},
    {32048, "PCS_NAD27_Washington_North"},
    {32049, "PCS_NAD27_Washington_South"},
    {32050, "PCS_NAD27_West_Virginia_N"},
    {32051, "PCS_NAD27_West_Virginia_S"},
    {32052, "PCS_NAD27_Wisconsin_North"},
    {32053, "PCS_NAD27_Wisconsin_Cen"},
    {32054, "PCS_NAD27_Wisconsin_South"},
    {32055, "PCS_NAD27_Wyoming_East"},
    {32056, "PCS_NAD27_Wyoming_E_Cen"},
    {32057, "PCS_NAD27_Wyoming_W_Cen"},
    {32058, "PCS_NAD27_Wyoming_West"},
    {32059, "PCS_NAD27_Puerto_Rico"},
    {32060, "PCS_NAD27_St_Croix"},
    {32100, "PCS_NAD83_Montana"},
    {32104, "PCS_NAD83_Nebraska"},
    {32107, "PCS_NAD83_Nevada_East"},
    {32108, "PCS_NAD83_Nevada_Central"},
    {32109, "PCS_NAD83_Nevada_West"},
    {32110, "PCS_NAD83_New_Hampshire"},
    {32111, "PCS_NAD83_New_Jersey"},
    {32112, "PCS_NAD83_New_Mexico_East"},
    {32113, "PCS_NAD83_New_Mexico_Cent"},
    {32114, "PCS_NAD83_New_Mexico_West"},
    {32115, "PCS_NAD83_New_York_East"},
    {32116, "PCS_NAD83_New_York_Central"},
    {32117, "PCS_NAD83_New_York_West"},
    {32118, "PCS_NAD83_New_York_Long_Is"},
    {32119, "PCS_NAD83_North_Carolina"},
    {32120, "PCS_NAD83_North_Dakota_N"},
    {32121, "PCS_NAD83_North_Dakota_S"},
    {32122, "PCS_NAD83_Ohio_North"},
    {32123, "PCS_NAD83_Ohio_South"},
    {32124, "PCS_NAD83_Oklahoma_North"},
    {32125, "PCS_NAD83_Oklahoma_South"},
    {32126, "PCS_NAD83_Oregon_North"},
    {32127, "PCS_NAD83_Oregon_South"},
    {32128, "PCS_NAD83_Pennsylvania_N"},
    {32129, "PCS_NAD83_Pennsylvania_S"},
    {32130, "PCS_NAD83_Rhode_Island"},
    {32133, "PCS_NAD83_South_Carolina"},
    {32134, "PCS_NAD83_South_Dakota_N"},
    {32135, "PCS_NAD83_South_Dakota_S"},
    {32136, "PCS_NAD83_Tennessee"},
    {32137, "PCS_NAD83_Texas_North"},
    {32138, "PCS_NAD83_Texas_North_Cen"},
    {32139, "PCS_NAD83_Texas_Central"},
    {32140, "PCS_NAD83_Texas_South_Cen"},
    {32141, "PCS_NAD83_Texas_South"},
    {32142, "PCS_NAD83_Utah_North"},
    {32143, "PCS_NAD83_Utah_Central"},
    {32144, "PCS_NAD83_Utah_South"},
    {32145, "PCS_NAD83_Vermont"},
    {32146, "PCS_NAD83_Virginia_North"},
    {32147, "PCS_NAD83_Virginia_South"},
    {32148, "PCS_NAD83_Washington_North"},
    {32149, "PCS_NAD83_Washington_South"},
    {32150, "PCS_NAD83_West_Virginia_N"},
    {32151, "PCS_NAD83_West_Virginia_S"},
    {32152, "PCS_NAD83_Wisconsin_North"},
    {32153, "PCS_NAD83_Wisconsin_Cen"},
    {32154, "PCS_NAD83_Wisconsin_South"},
    {32155, "PCS_NAD83_Wyoming_East"},
    {32156, "PCS_NAD83_Wyoming_E_Cen"},
    {32157, "PCS_NAD83_Wyoming_W_Cen"},
    {32158, "PCS_NAD83_Wyoming_West"},
    {32161, "PCS_NAD83_Puerto_Rico_Virgin_Is"},
    {32201, "PCS_WGS72_UTM_zone_1N"},
    {32202, "PCS_WGS72_UTM_zone_2N"},
    {32203, "PCS_WGS72_UTM_zone_3N"},
    {32204, "PCS_WGS72_UTM_zone_4N"},
    {32205, "PCS_WGS72_UTM_zone_5N"},
    {32206, "PCS_WGS72_UTM_zone_6N"},
    {32207, "PCS_WGS72_UTM_zone_7N"},
    {32208, "PCS_WGS72_UTM_zone_8N"},
    {32209, "PCS_WGS72_UTM_zone_9N"},
    {32210, "PCS_WGS72_UTM_zone_10N"},
    {32211, "PCS_WGS72_UTM_zone_11N"},
    {32212, "PCS_WGS72_UTM_zone_12N"},
    {32213, "PCS_WGS72_UTM_zone_13N"},
    {32214, "PCS_WGS72_UTM_zone_14N"},
    {32215, "PCS_WGS72_UTM_zone_15N"},
    {32216, "PCS_WGS72_UTM_zone_16N"},
    {32217, "PCS_WGS72_UTM_zone_17N"},
    {32218, "PCS_WGS72_UTM_zone_18N"},
    {32219, "PCS_WGS72_UTM_zone_19N"},
    {32220, "PCS_WGS72_UTM_zone_20N"},
    {32221, "PCS_WGS72_UTM_zone_21N"},
    {32222, "PCS_WGS72_UTM_zone_22N"},
    {32223, "PCS_WGS72_UTM_zone_23N"},
    {32224, "PCS_WGS72_UTM_zone_24N"},
    {32225, "PCS_WGS72_UTM_zone_25N"},
    {32226, "PCS_WGS72_UTM_zone_26N"},
    {32227, "PCS_WGS72_UTM_zone_27N"},
    {32228, "PCS_WGS72_UTM_zone_28N"},
    {32229, "PCS_WGS72_UTM_zone_29N"},
    {32230, "PCS_WGS72_UTM_zone_30N"},
    {32231, "PCS_WGS72_UTM_zone_31N"},
    {32232, "PCS_WGS72_UTM_zone_32N"},
    {32233, "PCS_WGS72_UTM_zone_33N"},
    {32234, "PCS_WGS72_UTM_zone_34N"},
    {32235, "PCS_WGS72_UTM_zone_35N"},
    {32236, "PCS_WGS72_UTM_zone_36N"},
    {32237, "PCS_WGS72_UTM_zone_37N"},
    {32238, "PCS_WGS72_UTM_zone_38N"},
    {32239, "PCS_WGS72_UTM_zone_39N"},
    {32240, "PCS_WGS72_UTM_zone_40N"},
    {32241, "PCS_WGS72_UTM_zone_41N"},
    {32242, "PCS_WGS72_UTM_zone_42N"},
    {32243, "PCS_WGS72_UTM_zone_43N"},
    {32244, "PCS_WGS72_UTM_zone_44N"},
    {32245, "PCS_WGS72_UTM_zone_45N"},
    {32246, "PCS_WGS72_UTM_zone_46N"},
    {32247, "PCS_WGS72_UTM_zone_47N"},
    {32248, "PCS_WGS72_UTM_zone_48N"},
    {32249, "PCS_WGS72_UTM_zone_49N"},
    {32250, "PCS_WGS72_UTM_zone_50N"},
    {32251, "PCS_WGS72_UTM_zone_51N"},
    {32252, "PCS_WGS72_UTM_zone_52N"},
    {32253, "PCS_WGS72_UTM_zone_53N"},
    {32254, "PCS_WGS72_UTM_zone_54N"},
    {32255, "PCS_WGS72_UTM_zone_55N"},
    {32256, "PCS_WGS72_UTM_zone_56N"},
    {32257, "PCS_WGS72_UTM_zone_57N"},
    {32258, "PCS_WGS72_UTM_zone_58N"},
    {32259, "PCS_WGS72_UTM_zone_59N"},
    {32260, "PCS_WGS72_UTM_zone_60N"},
    {32301, "PCS_WGS72_UTM_zone_1S"},
    {32302, "PCS_WGS72_UTM_zone_2S"},
    {32303, "PCS_WGS72_UTM_zone_3S"},
    {32304, "PCS_WGS72_UTM_zone_4S"},
    {32305, "PCS_WGS72_UTM_zone_5S"},
    {32306, "PCS_WGS72_UTM_zone_6S"},
    {32307, "PCS_WGS72_UTM_zone_7S"},
    {32308, "PCS_WGS72_UTM_zone_8S"},
    {32309, "PCS_WGS72_UTM_zone_9S"},
    {32310, "PCS_WGS72_UTM_zone_10S"},
    {32311, "PCS_WGS72_UTM_zone_11S"},
    {32312, "PCS_WGS72_UTM_zone_12S"},
    {32313, "PCS_WGS72_UTM_zone_13S"},
    {32314, "PCS_WGS72_UTM_zone_14S"},
    {32315, "PCS_WGS72_UTM_zone_15S"},
    {32316, "PCS_WGS72_UTM_zone_16S"},
    {32317, "PCS_WGS72_UTM_zone_17S"},
    {32318, "PCS_WGS72_UTM_zone_18S"},
    {32319, "PCS_WGS72_UTM_zone_19S"},
    {32320, "PCS_WGS72_UTM_zone_20S"},
    {32321, "PCS_WGS72_UTM_zone_21S"},
    {32322, "PCS_WGS72_UTM_zone_22S"},
    {32323, "PCS_WGS72_UTM_zone_23S"},
    {32324, "PCS_WGS72_UTM_zone_24S"},
    {32325, "PCS_WGS72_UTM_zone_25S"},
    {32326, "PCS_WGS72_UTM_zone_26S"},
    {32327, "PCS_WGS72_UTM_zone_27S"},
    {32328, "PCS_WGS72_UTM_zone_28S"},
    {32329, "PCS_WGS72_UTM_zone_29S"},
    {32330, "PCS_WGS72_UTM_zone_30S"},
    {32331, "PCS_WGS72_UTM_zone_31S"},
    {32332, "PCS_WGS72_UTM_zone_32S"},
    {32333, "PCS_WGS72_UTM_zone_33S"},
    {32334, "PCS_WGS72_UTM_zone_34S"},
    {32335, "PCS_WGS72_UTM_zone_35S"},
    {32336, "PCS_WGS72_UTM_zone_36S"},
    {32337, "PCS_WGS72_UTM_zone_37S"},
    {32338, "PCS_WGS72_UTM_zone_38S"},
    {32339, "PCS_WGS72_UTM_zone_39S"},
    {32340, "PCS_WGS72_UTM_zone_40S"},
    {32341, "PCS_WGS72_UTM_zone_41S"},
    {32342, "PCS_WGS72_UTM_zone_42S"},
    {32343, "PCS_WGS72_UTM_zone_43S"},
    {32344, "PCS_WGS72_UTM_zone_44S"},
    {32345, "PCS_WGS72_UTM_zone_45S"},
    {32346, "PCS_WGS72_UTM_zone_46S"},
    {32347, "PCS_WGS72_UTM_zone_47S"},
    {32348, "PCS_WGS72_UTM_zone_48S"},
    {32349, "PCS_WGS72_UTM_zone_49S"},
    {32350, "PCS_WGS72_UTM_zone_50S"},
    {32351, "PCS_WGS72_UTM_zone_51S"},
    {32352, "PCS_WGS72_UTM_zone_52S"},
    {32353, "PCS_WGS72_UTM_zone_53S"},
    {32354, "PCS_WGS72_UTM_zone_54S"},
    {32355, "PCS_WGS72_UTM_zone_55S"},
    {32356, "PCS_WGS72_UTM_zone_56S"},
    {32357, "PCS_WGS72_UTM_zone_57S"},
    {32358, "PCS_WGS72_UTM_zone_58S"},
    {32359, "PCS_WGS72_UTM_zone_59S"},
    {32360, "PCS_WGS72_UTM_zone_60S"},
    {32401, "PCS_WGS72BE_UTM_zone_1N"},
    {32402, "PCS_WGS72BE_UTM_zone_2N"},
    {32403, "PCS_WGS72BE_UTM_zone_3N"},
    {32404, "PCS_WGS72BE_UTM_zone_4N"},
    {32405, "PCS_WGS72BE_UTM_zone_5N"},
    {32406, "PCS_WGS72BE_UTM_zone_6N"},
    {32407, "PCS_WGS72BE_UTM_zone_7N"},
    {32408, "PCS_WGS72BE_UTM_zone_8N"},
    {32409, "PCS_WGS72BE_UTM_zone_9N"},
    {32410, "PCS_WGS72BE_UTM_zone_10N"},
    {32411, "PCS_WGS72BE_UTM_zone_11N"},
    {32412, "PCS_WGS72BE_UTM_zone_12N"},
    {32413, "PCS_WGS72BE_UTM_zone_13N"},
    {32414, "PCS_WGS72BE_UTM_zone_14N"},
    {32415, "PCS_WGS72BE_UTM_zone_15N"},
    {32416, "PCS_WGS72BE_UTM_zone_16N"},
    {32417, "PCS_WGS72BE_UTM_zone_17N"},
    {32418, "PCS_WGS72BE_UTM_zone_18N"},
    {32419, "PCS_WGS72BE_UTM_zone_19N"},
    {32420, "PCS_WGS72BE_UTM_zone_20N"},
    {32421, "PCS_WGS72BE_UTM_zone_21N"},
    {32422, "PCS_WGS72BE_UTM_zone_22N"},
    {32423, "PCS_WGS72BE_UTM_zone_23N"},
    {32424, "PCS_WGS72BE_UTM_zone_24N"},
    {32425, "PCS_WGS72BE_UTM_zone_25N"},
    {32426, "PCS_WGS72BE_UTM_zone_26N"},
    {32427, "PCS_WGS72BE_UTM_zone_27N"},
    {32428, "PCS_WGS72BE_UTM_zone_28N"},
    {32429, "PCS_WGS72BE_UTM_zone_29N"},
    {32430, "PCS_WGS72BE_UTM_zone_30N"},
    {32431, "PCS_WGS72BE_UTM_zone_31N"},
    {32432, "PCS_WGS72BE_UTM_zone_32N"},
    {32433, "PCS_WGS72BE_UTM_zone_33N"},
    {32434, "PCS_WGS72BE_UTM_zone_34N"},
    {32435, "PCS_WGS72BE_UTM_zone_35N"},
    {32436, "PCS_WGS72BE_UTM_zone_36N"},
    {32437, "PCS_WGS72BE_UTM_zone_37N"},
    {32438, "PCS_WGS72BE_UTM_zone_38N"},
    {32439, "PCS_WGS72BE_UTM_zone_39N"},
    {32440, "PCS_WGS72BE_UTM_zone_40N"},
    {32441, "PCS_WGS72BE_UTM_zone_41N"},
    {32442, "PCS_WGS72BE_UTM_zone_42N"},
    {32443, "PCS_WGS72BE_UTM_zone_43N"},
    {32444, "PCS_WGS72BE_UTM_zone_44N"},
    {32445, "PCS_WGS72BE_UTM_zone_45N"},
    {32446, "PCS_WGS72BE_UTM_zone_46N"},
    {32447, "PCS_WGS72BE_UTM_zone_47N"},
    {32448, "PCS_WGS72BE_UTM_zone_48N"},
    {32449, "PCS_WGS72BE_UTM_zone_49N"},
    {32450, "PCS_WGS72BE_UTM_zone_50N"},
    {32451, "PCS_WGS72BE_UTM_zone_51N"},
    {32452, "PCS_WGS72BE_UTM_zone_52N"},
    {32453, "PCS_WGS72BE_UTM_zone_53N"},
    {32454, "PCS_WGS72BE_UTM_zone_54N"},
    {32455, "PCS_WGS72BE_UTM_zone_55N"},
    {32456, "PCS_WGS72BE_UTM_zone_56N"},
    {32457, "PCS_WGS72BE_UTM_zone_57N"},
    {32458, "PCS_WGS72BE_UTM_zone_58N"},
    {32459, "PCS_WGS72BE_UTM_zone_59N"},
    {32460, "PCS_WGS72BE_UTM_zone_60N"},
    {32501, "PCS_WGS72BE_UTM_zone_1S"},
    {32502, "PCS_WGS72BE_UTM_zone_2S"},
    {32503, "PCS_WGS72BE_UTM_zone_3S"},
    {32504, "PCS_WGS72BE_UTM_zone_4S"},
    {32505, "PCS_WGS72BE_UTM_zone_5S"},
    {32506, "PCS_WGS72BE_UTM_zone_6S"},
    {32507, "PCS_WGS72BE_UTM_zone_7S"},
    {32508, "PCS_WGS72BE_UTM_zone_8S"},
    {32509, "PCS_WGS72BE_UTM_zone_9S"},
    {32510, "PCS_WGS72BE_UTM_zone_10S"},
    {32511, "PCS_WGS72BE_UTM_zone_11S"},
    {32512, "PCS_WGS72BE_UTM_zone_12S"},
    {32513, "PCS_WGS72BE_UTM_zone_13S"},
    {32514, "PCS_WGS72BE_UTM_zone_14S"},
    {32515, "PCS_WGS72BE_UTM_zone_15S"},
    {32516, "PCS_WGS72BE_UTM_zone_16S"},
    {32517, "PCS_WGS72BE_UTM_zone_17S"},
    {32518, "PCS_WGS72BE_UTM_zone_18S"},
    {32519, "PCS_WGS72BE_UTM_zone_19S"},
    {32520, "PCS_WGS72BE_UTM_zone_20S"},
    {32521, "PCS_WGS72BE_UTM_zone_21S"},
    {32522, "PCS_WGS72BE_UTM_zone_22S"},
    {32523, "PCS_WGS72BE_UTM_zone_23S"},
    {32524, "PCS_WGS72BE_UTM_zone_24S"},
    {32525, "PCS_WGS72BE_UTM_zone_25S"},
    {32526, "PCS_WGS72BE_UTM_zone_26S"},
    {32527, "PCS_WGS72BE_UTM_zone_27S"},
    {32528, "PCS_WGS72BE_UTM_zone_28S"},
    {32529, "PCS_WGS72BE_UTM_zone_29S"},
    {32530, "PCS_WGS72BE_UTM_zone_30S"},
    {32531, "PCS_WGS72BE_UTM_zone_31S"},
    {32532, "PCS_WGS72BE_UTM_zone_32S"},
    {32533, "PCS_WGS72BE_UTM_zone_33S"},
    {32534, "PCS_WGS72BE_UTM_zone_34S"},
    {32535, "PCS_WGS72BE_UTM_zone_35S"},
    {32536, "PCS_WGS72BE_UTM_zone_36S"},
    {32537, "PCS_WGS72BE_UTM_zone_37S"},
    {32538, "PCS_WGS72BE_UTM_zone_38S"},
    {32539, "PCS_WGS72BE_UTM_zone_39S"},
    {32540, "PCS_WGS72BE_UTM_zone_40S"},
    {32541, "PCS_WGS72BE_UTM_zone_41S"},
    {32542, "PCS_WGS72BE_UTM_zone_42S"},
    {32543, "PCS_WGS72BE_UTM_zone_43S"},
    {32544, "PCS_WGS72BE_UTM_zone_44S"},
    {32545, "PCS_WGS72BE_UTM_zone_45S"},
    {32546, "PCS_WGS72BE_UTM_zone_46S"},
    {32547, "PCS_WGS72BE_UTM_zone_47S"},
    {32548, "PCS_WGS72BE_UTM_zone_48S"},
    {32549, "PCS_WGS72BE_UTM_zone_49S"},
    {32550, "PCS_WGS72BE_UTM_zone_50S"},
    {32551, "PCS_WGS72BE_UTM_zone_51S"},
    {32552, "PCS_WGS72BE_UTM_zone_52S"},
    {32553, "PCS_WGS72BE_UTM_zone_53S"},
    {32554, "PCS_WGS72BE_UTM_zone_54S"},
    {32555, "PCS_WGS72BE_UTM_zone_55S"},
    {32556, "PCS_WGS72BE_UTM_zone_56S"},
    {32557, "PCS_WGS72BE_UTM_zone_57S"},
    {32558, "PCS_WGS72BE_UTM_zone_58S"},
    {32559, "PCS_WGS72BE_UTM_zone_59S"},
    {32560, "PCS_WGS72BE_UTM_zone_60S"},
    {32601, "PCS_WGS84_UTM_zone_1N"},
    {32602, "PCS_WGS84_UTM_zone_2N"},
    {32603, "PCS_WGS84_UTM_zone_3N"},
    {32604, "PCS_WGS84_UTM_zone_4N"},
    {32605, "PCS_WGS84_UTM_zone_5N"},
    {32606, "PCS_WGS84_UTM_zone_6N"},
    {32607, "PCS_WGS84_UTM_zone_7N"},
    {32608, "PCS_WGS84_UTM_zone_8N"},
    {32609, "PCS_WGS84_UTM_zone_9N"},
    {32610, "PCS_WGS84_UTM_zone_10N"},
    {32611, "PCS_WGS84_UTM_zone_11N"},
    {32612, "PCS_WGS84_UTM_zone_12N"},
    {32613, "PCS_WGS84_UTM_zone_13N"},
    {32614, "PCS_WGS84_UTM_zone_14N"},
    {32615, "PCS_WGS84_UTM_zone_15N"},
    {32616, "PCS_WGS84_UTM_zone_16N"},
    {32617, "PCS_WGS84_UTM_zone_17N"},
    {32618, "PCS_WGS84_UTM_zone_18N"},
    {32619, "PCS_WGS84_UTM_zone_19N"},
    {32620, "PCS_WGS84_UTM_zone_20N"},
    {32621, "PCS_WGS84_UTM_zone_21N"},
    {32622, "PCS_WGS84_UTM_zone_22N"},
    {32623, "PCS_WGS84_UTM_zone_23N"},
    {32624, "PCS_WGS84_UTM_zone_24N"},
    {32625, "PCS_WGS84_UTM_zone_25N"},
    {32626, "PCS_WGS84_UTM_zone_26N"},
    {32627, "PCS_WGS84_UTM_zone_27N"},
    {32628, "PCS_WGS84_UTM_zone_28N"},
    {32629, "PCS_WGS84_UTM_zone_29N"},
    {32630, "PCS_WGS84_UTM_zone_30N"},
    {32631, "PCS_WGS84_UTM_zone_31N"},
    {32632, "PCS_WGS84_UTM_zone_32N"},
    {32633, "PCS_WGS84_UTM_zone_33N"},
    {32634, "PCS_WGS84_UTM_zone_34N"},
    {32635, "PCS_WGS84_UTM_zone_35N"},
    {32636, "PCS_WGS84_UTM_zone_36N"},
    {32637, "PCS_WGS84_UTM_zone_37N"},
    {32638, "PCS_WGS84_UTM_zone_38N"},
    {32639, "PCS_WGS84_UTM_zone_39N"},
    {32640, "PCS_WGS84_UTM_zone_40N"},
    {32641, "PCS_WGS84_UTM_zone_41N"},
    {32642, "PCS_WGS84_UTM_zone_42N"},
    {32643, "PCS_WGS84_UTM_zone_43N"},
    {32644, "PCS_WGS84_UTM_zone_44N"},
    {32645, "PCS_WGS84_UTM_zone_45N"},
    {32646, "PCS_WGS84_UTM_zone_46N"},
    {32647, "PCS_WGS84_UTM_zone_47N"},
    {32648, "PCS_WGS84_UTM_zone_48N"},
    {32649, "PCS_WGS84_UTM_zone_49N"},
    {32650, "PCS_WGS84_UTM_zone_50N"},
    {32651, "PCS_WGS84_UTM_zone_51N"},
    {32652, "PCS_WGS84_UTM_zone_52N"},
    {32653, "PCS_WGS84_UTM_zone_53N"},
    {32654, "PCS_WGS84_UTM_zone_54N"},
    {32655, "PCS_WGS84_UTM_zone_55N"},
    {32656, "PCS_WGS84_UTM_zone_56N"},
    {32657, "PCS_WGS84_UTM_zone_57N"},
    {32658, "PCS_WGS84_UTM_zone_58N"},
    {32659, "PCS_WGS84_UTM_zone_59N"},
    {32660, "PCS_WGS84_UTM_zone_60N"},
    {32701, "PCS_WGS84_UTM_zone_1S"},
    {32702, "PCS_WGS84_UTM_zone_2S"},
    {32703, "PCS_WGS84_UTM_zone_3S"},
    {32704, "PCS_WGS84_UTM_zone_4S"},
    {32705, "PCS_WGS84_UTM_zone_5S"},
    {32706, "PCS_WGS84_UTM_zone_6S"},
    {32707, "PCS_WGS84_UTM_zone_7S"},
    {32708, "PCS_WGS84_UTM_zone_8S"},
    {32709, "PCS_WGS84_UTM_zone_9S"},
    {32710, "PCS_WGS84_UTM_zone_10S"},
    {32711, "PCS_WGS84_UTM_zone_11S"},
    {32712, "PCS_WGS84_UTM_zone_12S"},
    {32713, "PCS_WGS84_UTM_zone_13S"},
    {32714, "PCS_WGS84_UTM_zone_14S"},
    {32715, "PCS_WGS84_UTM_zone_15S"},
    {32716, "PCS_WGS84_UTM_zone_16S"},
    {32717, "PCS_WGS84_UTM_zone_17S"},
    {32718, "PCS_WGS84_UTM_zone_18S"},
    {32719, "PCS_WGS84_UTM_zone_19S"},
    {32720, "PCS_WGS84_UTM_zone_20S"},
    {32721, "PCS_WGS84_UTM_zone_21S"},
    {32722, "PCS_WGS84_UTM_zone_22S"},
    {32723, "PCS_WGS84_UTM_zone_23S"},
    {32724, "PCS_WGS84_UTM_zone_24S"},
    {32725, "PCS_WGS84_UTM_zone_25S"},
    {32726, "PCS_WGS84_UTM_zone_26S"},
    {32727, "PCS_WGS84_UTM_zone_27S"},
    {32728, "PCS_WGS84_UTM_zone_28S"},
    {32729, "PCS_WGS84_UTM_zone_29S"},
    {32730, "PCS_WGS84_UTM_zone_30S"},
    {32731, "PCS_WGS84_UTM_zone_31S"},
    {32732, "PCS_WGS84_UTM_zone_32S"},
    {32733, "PCS_WGS84_UTM_zone_33S"},
    {32734, "PCS_WGS84_UTM_zone_34S"},
    {32735, "PCS_WGS84_UTM_zone_35S"},
    {32736, "PCS_WGS84_UTM_zone_36S"},
    {32737, "PCS_WGS84_UTM_zone_37S"},
    {32738, "PCS_WGS84_UTM_zone_38S"},
    {32739, "PCS_WGS84_UTM_zone_39S"},
    {32740, "PCS_WGS84_UTM_zone_40S"},
    {32741, "PCS_WGS84_UTM_zone_41S"},
    {32742, "PCS_WGS84_UTM_zone_42S"},
    {32743, "PCS_WGS84_UTM_zone_43S"},
    {32744, "PCS_WGS84_UTM_zone_44S"},
    {32745, "PCS_WGS84_UTM_zone_45S"},
    {32746, "PCS_WGS84_UTM_zone_46S"},
    {32747, "PCS_WGS84_UTM_zone_47S"},
    {32748, "PCS_WGS84_UTM_zone_48S"},
    {32749, "PCS_WGS84_UTM_zone_49S"},
    {32750, "PCS_WGS84_UTM_zone_50S"},
    {32751, "PCS_WGS84_UTM_zone_51S"},
    {32752, "PCS_WGS84_UTM_zone_52S"},
    {32753, "PCS_WGS84_UTM_zone_53S"},
    {32754, "PCS_WGS84_UTM_zone_54S"},
    {32755, "PCS_WGS84_UTM_zone_55S"},
    {32756, "PCS_WGS84_UTM_zone_56S"},
    {32757, "PCS_WGS84_UTM_zone_57S"},
    {32758, "PCS_WGS84_UTM_zone_58S"},
    {32759, "PCS_WGS84_UTM_zone_59S"},
    {32760, "PCS_WGS84_UTM_zone_60S"}
};

static const TiffGeoTagKeyName tiff_projection_codes[] = {
    {10101, "Proj_Alabama_CS27_East"},
    {10102, "Proj_Alabama_CS27_West"},
    {10131, "Proj_Alabama_CS83_East"},
    {10132, "Proj_Alabama_CS83_West"},
    {10201, "Proj_Arizona_Coordinate_System_east"},
    {10202, "Proj_Arizona_Coordinate_System_Central"},
    {10203, "Proj_Arizona_Coordinate_System_west"},
    {10231, "Proj_Arizona_CS83_east"},
    {10232, "Proj_Arizona_CS83_Central"},
    {10233, "Proj_Arizona_CS83_west"},
    {10301, "Proj_Arkansas_CS27_North"},
    {10302, "Proj_Arkansas_CS27_South"},
    {10331, "Proj_Arkansas_CS83_North"},
    {10332, "Proj_Arkansas_CS83_South"},
    {10401, "Proj_California_CS27_I"},
    {10402, "Proj_California_CS27_II"},
    {10403, "Proj_California_CS27_III"},
    {10404, "Proj_California_CS27_IV"},
    {10405, "Proj_California_CS27_V"},
    {10406, "Proj_California_CS27_VI"},
    {10407, "Proj_California_CS27_VII"},
    {10431, "Proj_California_CS83_1"},
    {10432, "Proj_California_CS83_2"},
    {10433, "Proj_California_CS83_3"},
    {10434, "Proj_California_CS83_4"},
    {10435, "Proj_California_CS83_5"},
    {10436, "Proj_California_CS83_6"},
    {10501, "Proj_Colorado_CS27_North"},
    {10502, "Proj_Colorado_CS27_Central"},
    {10503, "Proj_Colorado_CS27_South"},
    {10531, "Proj_Colorado_CS83_North"},
    {10532, "Proj_Colorado_CS83_Central"},
    {10533, "Proj_Colorado_CS83_South"},
    {10600, "Proj_Connecticut_CS27"},
    {10630, "Proj_Connecticut_CS83"},
    {10700, "Proj_Delaware_CS27"},
    {10730, "Proj_Delaware_CS83"},
    {10901, "Proj_Florida_CS27_East"},
    {10902, "Proj_Florida_CS27_West"},
    {10903, "Proj_Florida_CS27_North"},
    {10931, "Proj_Florida_CS83_East"},
    {10932, "Proj_Florida_CS83_West"},
    {10933, "Proj_Florida_CS83_North"},
    {11001, "Proj_Georgia_CS27_East"},
    {11002, "Proj_Georgia_CS27_West"},
    {11031, "Proj_Georgia_CS83_East"},
    {11032, "Proj_Georgia_CS83_West"},
    {11101, "Proj_Idaho_CS27_East"},
    {11102, "Proj_Idaho_CS27_Central"},
    {11103, "Proj_Idaho_CS27_West"},
    {11131, "Proj_Idaho_CS83_East"},
    {11132, "Proj_Idaho_CS83_Central"},
    {11133, "Proj_Idaho_CS83_West"},
    {11201, "Proj_Illinois_CS27_East"},
    {11202, "Proj_Illinois_CS27_West"},
    {11231, "Proj_Illinois_CS83_East"},
    {11232, "Proj_Illinois_CS83_West"},
    {11301, "Proj_Indiana_CS27_East"},
    {11302, "Proj_Indiana_CS27_West"},
    {11331, "Proj_Indiana_CS83_East"},
    {11332, "Proj_Indiana_CS83_West"},
    {11401, "Proj_Iowa_CS27_North"},
    {11402, "Proj_Iowa_CS27_South"},
    {11431, "Proj_Iowa_CS83_North"},
    {11432, "Proj_Iowa_CS83_South"},
    {11501, "Proj_Kansas_CS27_North"},
    {11502, "Proj_Kansas_CS27_South"},
    {11531, "Proj_Kansas_CS83_North"},
    {11532, "Proj_Kansas_CS83_South"},
    {11601, "Proj_Kentucky_CS27_North"},
    {11602, "Proj_Kentucky_CS27_South"},
    {11631, "Proj_Kentucky_CS83_North"},
    {11632, "Proj_Kentucky_CS83_South"},
    {11701, "Proj_Louisiana_CS27_North"},
    {11702, "Proj_Louisiana_CS27_South"},
    {11731, "Proj_Louisiana_CS83_North"},
    {11732, "Proj_Louisiana_CS83_South"},
    {11801, "Proj_Maine_CS27_East"},
    {11802, "Proj_Maine_CS27_West"},
    {11831, "Proj_Maine_CS83_East"},
    {11832, "Proj_Maine_CS83_West"},
    {11900, "Proj_Maryland_CS27"},
    {11930, "Proj_Maryland_CS83"},
    {12001, "Proj_Massachusetts_CS27_Mainland"},
    {12002, "Proj_Massachusetts_CS27_Island"},
    {12031, "Proj_Massachusetts_CS83_Mainland"},
    {12032, "Proj_Massachusetts_CS83_Island"},
    {12101, "Proj_Michigan_State_Plane_East"},
    {12102, "Proj_Michigan_State_Plane_Old_Central"},
    {12103, "Proj_Michigan_State_Plane_West"},
    {12111, "Proj_Michigan_CS27_North"},
    {12112, "Proj_Michigan_CS27_Central"},
    {12113, "Proj_Michigan_CS27_South"},
    {12141, "Proj_Michigan_CS83_North"},
    {12142, "Proj_Michigan_CS83_Central"},
    {12143, "Proj_Michigan_CS83_South"},
    {12201, "Proj_Minnesota_CS27_North"},
    {12202, "Proj_Minnesota_CS27_Central"},
    {12203, "Proj_Minnesota_CS27_South"},
    {12231, "Proj_Minnesota_CS83_North"},
    {12232, "Proj_Minnesota_CS83_Central"},
    {12233, "Proj_Minnesota_CS83_South"},
    {12301, "Proj_Mississippi_CS27_East"},
    {12302, "Proj_Mississippi_CS27_West"},
    {12331, "Proj_Mississippi_CS83_East"},
    {12332, "Proj_Mississippi_CS83_West"},
    {12401, "Proj_Missouri_CS27_East"},
    {12402, "Proj_Missouri_CS27_Central"},
    {12403, "Proj_Missouri_CS27_West"},
    {12431, "Proj_Missouri_CS83_East"},
    {12432, "Proj_Missouri_CS83_Central"},
    {12433, "Proj_Missouri_CS83_West"},
    {12501, "Proj_Montana_CS27_North"},
    {12502, "Proj_Montana_CS27_Central"},
    {12503, "Proj_Montana_CS27_South"},
    {12530, "Proj_Montana_CS83"},
    {12601, "Proj_Nebraska_CS27_North"},
    {12602, "Proj_Nebraska_CS27_South"},
    {12630, "Proj_Nebraska_CS83"},
    {12701, "Proj_Nevada_CS27_East"},
    {12702, "Proj_Nevada_CS27_Central"},
    {12703, "Proj_Nevada_CS27_West"},
    {12731, "Proj_Nevada_CS83_East"},
    {12732, "Proj_Nevada_CS83_Central"},
    {12733, "Proj_Nevada_CS83_West"},
    {12800, "Proj_New_Hampshire_CS27"},
    {12830, "Proj_New_Hampshire_CS83"},
    {12900, "Proj_New_Jersey_CS27"},
    {12930, "Proj_New_Jersey_CS83"},
    {13001, "Proj_New_Mexico_CS27_East"},
    {13002, "Proj_New_Mexico_CS27_Central"},
    {13003, "Proj_New_Mexico_CS27_West"},
    {13031, "Proj_New_Mexico_CS83_East"},
    {13032, "Proj_New_Mexico_CS83_Central"},
    {13033, "Proj_New_Mexico_CS83_West"},
    {13101, "Proj_New_York_CS27_East"},
    {13102, "Proj_New_York_CS27_Central"},
    {13103, "Proj_New_York_CS27_West"},
    {13104, "Proj_New_York_CS27_Long_Island"},
    {13131, "Proj_New_York_CS83_East"},
    {13132, "Proj_New_York_CS83_Central"},
    {13133, "Proj_New_York_CS83_West"},
    {13134, "Proj_New_York_CS83_Long_Island"},
    {13200, "Proj_North_Carolina_CS27"},
    {13230, "Proj_North_Carolina_CS83"},
    {13301, "Proj_North_Dakota_CS27_North"},
    {13302, "Proj_North_Dakota_CS27_South"},
    {13331, "Proj_North_Dakota_CS83_North"},
    {13332, "Proj_North_Dakota_CS83_South"},
    {13401, "Proj_Ohio_CS27_North"},
    {13402, "Proj_Ohio_CS27_South"},
    {13431, "Proj_Ohio_CS83_North"},
    {13432, "Proj_Ohio_CS83_South"},
    {13501, "Proj_Oklahoma_CS27_North"},
    {13502, "Proj_Oklahoma_CS27_South"},
    {13531, "Proj_Oklahoma_CS83_North"},
    {13532, "Proj_Oklahoma_CS83_South"},
    {13601, "Proj_Oregon_CS27_North"},
    {13602, "Proj_Oregon_CS27_South"},
    {13631, "Proj_Oregon_CS83_North"},
    {13632, "Proj_Oregon_CS83_South"},
    {13701, "Proj_Pennsylvania_CS27_North"},
    {13702, "Proj_Pennsylvania_CS27_South"},
    {13731, "Proj_Pennsylvania_CS83_North"},
    {13732, "Proj_Pennsylvania_CS83_South"},
    {13800, "Proj_Rhode_Island_CS27"},
    {13830, "Proj_Rhode_Island_CS83"},
    {13901, "Proj_South_Carolina_CS27_North"},
    {13902, "Proj_South_Carolina_CS27_South"},
    {13930, "Proj_South_Carolina_CS83"},
    {14001, "Proj_South_Dakota_CS27_North"},
    {14002, "Proj_South_Dakota_CS27_South"},
    {14031, "Proj_South_Dakota_CS83_North"},
    {14032, "Proj_South_Dakota_CS83_South"},
    {14100, "Proj_Tennessee_CS27"},
    {14130, "Proj_Tennessee_CS83"},
    {14201, "Proj_Texas_CS27_North"},
    {14202, "Proj_Texas_CS27_North_Central"},
    {14203, "Proj_Texas_CS27_Central"},
    {14204, "Proj_Texas_CS27_South_Central"},
    {14205, "Proj_Texas_CS27_South"},
    {14231, "Proj_Texas_CS83_North"},
    {14232, "Proj_Texas_CS83_North_Central"},
    {14233, "Proj_Texas_CS83_Central"},
    {14234, "Proj_Texas_CS83_South_Central"},
    {14235, "Proj_Texas_CS83_South"},
    {14301, "Proj_Utah_CS27_North"},
    {14302, "Proj_Utah_CS27_Central"},
    {14303, "Proj_Utah_CS27_South"},
    {14331, "Proj_Utah_CS83_North"},
    {14332, "Proj_Utah_CS83_Central"},
    {14333, "Proj_Utah_CS83_South"},
    {14400, "Proj_Vermont_CS27"},
    {14430, "Proj_Vermont_CS83"},
    {14501, "Proj_Virginia_CS27_North"},
    {14502, "Proj_Virginia_CS27_South"},
    {14531, "Proj_Virginia_CS83_North"},
    {14532, "Proj_Virginia_CS83_South"},
    {14601, "Proj_Washington_CS27_North"},
    {14602, "Proj_Washington_CS27_South"},
    {14631, "Proj_Washington_CS83_North"},
    {14632, "Proj_Washington_CS83_South"},
    {14701, "Proj_West_Virginia_CS27_North"},
    {14702, "Proj_West_Virginia_CS27_South"},
    {14731, "Proj_West_Virginia_CS83_North"},
    {14732, "Proj_West_Virginia_CS83_South"},
    {14801, "Proj_Wisconsin_CS27_North"},
    {14802, "Proj_Wisconsin_CS27_Central"},
    {14803, "Proj_Wisconsin_CS27_South"},
    {14831, "Proj_Wisconsin_CS83_North"},
    {14832, "Proj_Wisconsin_CS83_Central"},
    {14833, "Proj_Wisconsin_CS83_South"},
    {14901, "Proj_Wyoming_CS27_East"},
    {14902, "Proj_Wyoming_CS27_East_Central"},
    {14903, "Proj_Wyoming_CS27_West_Central"},
    {14904, "Proj_Wyoming_CS27_West"},
    {14931, "Proj_Wyoming_CS83_East"},
    {14932, "Proj_Wyoming_CS83_East_Central"},
    {14933, "Proj_Wyoming_CS83_West_Central"},
    {14934, "Proj_Wyoming_CS83_West"},
    {15001, "Proj_Alaska_CS27_1"},
    {15002, "Proj_Alaska_CS27_2"},
    {15003, "Proj_Alaska_CS27_3"},
    {15004, "Proj_Alaska_CS27_4"},
    {15005, "Proj_Alaska_CS27_5"},
    {15006, "Proj_Alaska_CS27_6"},
    {15007, "Proj_Alaska_CS27_7"},
    {15008, "Proj_Alaska_CS27_8"},
    {15009, "Proj_Alaska_CS27_9"},
    {15010, "Proj_Alaska_CS27_10"},
    {15031, "Proj_Alaska_CS83_1"},
    {15032, "Proj_Alaska_CS83_2"},
    {15033, "Proj_Alaska_CS83_3"},
    {15034, "Proj_Alaska_CS83_4"},
    {15035, "Proj_Alaska_CS83_5"},
    {15036, "Proj_Alaska_CS83_6"},
    {15037, "Proj_Alaska_CS83_7"},
    {15038, "Proj_Alaska_CS83_8"},
    {15039, "Proj_Alaska_CS83_9"},
    {15040, "Proj_Alaska_CS83_10"},
    {15101, "Proj_Hawaii_CS27_1"},
    {15102, "Proj_Hawaii_CS27_2"},
    {15103, "Proj_Hawaii_CS27_3"},
    {15104, "Proj_Hawaii_CS27_4"},
    {15105, "Proj_Hawaii_CS27_5"},
    {15131, "Proj_Hawaii_CS83_1"},
    {15132, "Proj_Hawaii_CS83_2"},
    {15133, "Proj_Hawaii_CS83_3"},
    {15134, "Proj_Hawaii_CS83_4"},
    {15135, "Proj_Hawaii_CS83_5"},
    {15201, "Proj_Puerto_Rico_CS27"},
    {15202, "Proj_St_Croix"},
    {15230, "Proj_Puerto_Rico_Virgin_Is"},
    {15914, "Proj_BLM_14N_feet"},
    {15915, "Proj_BLM_15N_feet"},
    {15916, "Proj_BLM_16N_feet"},
    {15917, "Proj_BLM_17N_feet"},
    {17348, "Proj_Map_Grid_of_Australia_48"},
    {17349, "Proj_Map_Grid_of_Australia_49"},
    {17350, "Proj_Map_Grid_of_Australia_50"},
    {17351, "Proj_Map_Grid_of_Australia_51"},
    {17352, "Proj_Map_Grid_of_Australia_52"},
    {17353, "Proj_Map_Grid_of_Australia_53"},
    {17354, "Proj_Map_Grid_of_Australia_54"},
    {17355, "Proj_Map_Grid_of_Australia_55"},
    {17356, "Proj_Map_Grid_of_Australia_56"},
    {17357, "Proj_Map_Grid_of_Australia_57"},
    {17358, "Proj_Map_Grid_of_Australia_58"},
    {17448, "Proj_Australian_Map_Grid_48"},
    {17449, "Proj_Australian_Map_Grid_49"},
    {17450, "Proj_Australian_Map_Grid_50"},
    {17451, "Proj_Australian_Map_Grid_51"},
    {17452, "Proj_Australian_Map_Grid_52"},
    {17453, "Proj_Australian_Map_Grid_53"},
    {17454, "Proj_Australian_Map_Grid_54"},
    {17455, "Proj_Australian_Map_Grid_55"},
    {17456, "Proj_Australian_Map_Grid_56"},
    {17457, "Proj_Australian_Map_Grid_57"},
    {17458, "Proj_Australian_Map_Grid_58"},
    {18031, "Proj_Argentina_1"},
    {18032, "Proj_Argentina_2"},
    {18033, "Proj_Argentina_3"},
    {18034, "Proj_Argentina_4"},
    {18035, "Proj_Argentina_5"},
    {18036, "Proj_Argentina_6"},
    {18037, "Proj_Argentina_7"},
    {18051, "Proj_Colombia_3W"},
    {18052, "Proj_Colombia_Bogota"},
    {18053, "Proj_Colombia_3E"},
    {18054, "Proj_Colombia_6E"},
    {18072, "Proj_Egypt_Red_Belt"},
    {18073, "Proj_Egypt_Purple_Belt"},
    {18074, "Proj_Extended_Purple_Belt"},
    {18141, "Proj_New_Zealand_North_Island_Nat_Grid"},
    {18142, "Proj_New_Zealand_South_Island_Nat_Grid"},
    {19900, "Proj_Bahrain_Grid"},
    {19905, "Proj_Netherlands_E_Indies_Equatorial"},
    {19912, "Proj_RSO_Borneo"}
};

#define TIFF_COORD_TRANS_OFFSET 1
static const char *const tiff_coord_trans_codes[] = {
    "CT_TransverseMercator",
    "CT_TransvMercator_Modified_Alaska",
    "CT_ObliqueMercator",
    "CT_ObliqueMercator_Laborde",
    "CT_ObliqueMercator_Rosenmund",
    "CT_ObliqueMercator_Spherical",
    "CT_Mercator",
    "CT_LambertConfConic_2SP",
    "CT_LambertConfConic_Helmert",
    "CT_LambertAzimEqualArea",
    "CT_AlbersEqualArea",
    "CT_AzimuthalEquidistant",
    "CT_EquidistantConic",
    "CT_Stereographic",
    "CT_PolarStereographic",
    "CT_ObliqueStereographic",
    "CT_Equirectangular",
    "CT_CassiniSoldner",
    "CT_Gnomonic",
    "CT_MillerCylindrical",
    "CT_Orthographic",
    "CT_Polyconic",
    "CT_Robinson",
    "CT_Sinusoidal",
    "CT_VanDerGrinten",
    "CT_NewZealandMapGrid",
    "CT_TransvMercator_SouthOriented"
};

#define TIFF_VERT_CS_OFFSET 5001
static const char *const tiff_vert_cs_codes[] = {
    "VertCS_Airy_1830_ellipsoid",
    "VertCS_Airy_Modified_1849_ellipsoid",
    "VertCS_ANS_ellipsoid",
    "VertCS_Bessel_1841_ellipsoid",
    "VertCS_Bessel_Modified_ellipsoid",
    "VertCS_Bessel_Namibia_ellipsoid",
    "VertCS_Clarke_1858_ellipsoid",
    "VertCS_Clarke_1866_ellipsoid",
    "VertCS_Clarke_1880_Benoit_ellipsoid",
    "VertCS_Clarke_1880_IGN_ellipsoid",
    "VertCS_Clarke_1880_RGS_ellipsoid",
    "VertCS_Clarke_1880_Arc_ellipsoid",
    "VertCS_Clarke_1880_SGA_1922_ellipsoid",
    "VertCS_Everest_1830_1937_Adjustment_ellipsoid",
    "VertCS_Everest_1830_1967_Definition_ellipsoid",
    "VertCS_Everest_1830_1975_Definition_ellipsoid",
    "VertCS_Everest_1830_Modified_ellipsoid",
    "VertCS_GRS_1980_ellipsoid",
    "VertCS_Helmert_1906_ellipsoid",
    "VertCS_INS_ellipsoid",
    "VertCS_International_1924_ellipsoid",
    "VertCS_International_1967_ellipsoid",
    "VertCS_Krassowsky_1940_ellipsoid",
    "VertCS_NWL_9D_ellipsoid",
    "VertCS_NWL_10D_ellipsoid",
    "VertCS_Plessis_1817_ellipsoid",
    "VertCS_Struve_1860_ellipsoid",
    "VertCS_War_Office_ellipsoid",
    "VertCS_WGS_84_ellipsoid",
    "VertCS_GEM_10C_ellipsoid",
    "VertCS_OSU86F_ellipsoid",
    "VertCS_OSU91A_ellipsoid"
};

#define TIFF_ORTHO_VERT_CS_OFFSET 5101
static const char *const tiff_ortho_vert_cs_codes[] = {
    "VertCS_Newlyn",
    "VertCS_North_American_Vertical_Datum_1929",
    "VertCS_North_American_Vertical_Datum_1988",
    "VertCS_Yellow_Sea_1956",
    "VertCS_Baltic_Sea",
    "VertCS_Caspian_Sea"
};

#endif
